/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */
 
/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's Remove command.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: remove.js,v 1.2 2006/08/23 16:55:28 jameso Exp $
 */


mozile.require("mozile.edit.rich");
mozile.require("mozile.xml");
mozile.require("mozile.test.shared.util");
mozile.require("mozile.test.edit.util");
var name = "mozile.edit.remove";


var testNode;
var testContent = '<p id="testContent">_Content.          More.<img/> <img/><img/><img/>More.<!--Comment-->More.<img/><!--Comment--><img/>More.<img/><!--Comment--><!--Comment--><!--Comment--><!--Comment--><strong>Content.</strong><img/>More.<em>Content.</em>More.<strong><em>Content.</em></strong>More._</p>';

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	//return ["testSplitAndMerge"];
	return ["testDeleteManyPrevious", "testDeleteManyNext", "testDeleteMergePrevious", "testDeleteMergeNext", "testDeleteRangeText", "testDeleteRangeNodes", "testDeleteAllChildren", "testDeleteAndPreserve", "testSplitAndMerge", "testChangeBlocks"];
}


/**
 *
 */
function testDeleteManyPrevious() {
	var command = mozile.edit.remove;
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	
	var operations = [ // removed = number of children removed
		{steps: 13, removed: 2},
		{steps: 13, removed: 2},
		{steps: 6, removed: 1},
		{steps: 9, removed: 7},
		{steps: 6, removed: 2},
		{steps: 1, removed: 2}, // index 5
		{steps: 6, removed: 2},
		{steps: 5, removed: 1},
		{steps: 1, removed: 1},
		{steps: 1, removed: 2},
		{steps: 2, removed: 2}, // index 10
		{steps: 14, removed: 0}
		//{steps: 23, removed: 0}, // TODO: Handle white space properly
	];
	var op;

	var event = {
		type: "keypress", 
		keyCode: 8, // backspace key
		content: " ",
		ctrlKey: false, 
		metaKey: false
	}
	if(mozile.browser.isIE) event.type = "keydown";
	
	var states = new Array();
	var state;
	resetTestNode();
	selection.collapse(testNode.lastChild, testNode.lastChild.data.length - 1);
	var totalLength = testNode.childNodes.length;
	var lastLength = totalLength;
	for(var i=0; i < operations.length; i++) {
		op = operations[i];
		for(var j=0; j < op.steps; j++) {
			state = command.trigger(event);
			states.push(state);
			assertTrue("Command "+ i +","+ j +" should have been executed.", state.executed);
		}
		assertEquals("Previous: Number of testNode children removed for index "+ i +" should be "+ op.removed, op.removed, lastLength - testNode.childNodes.length);
		lastLength = testNode.childNodes.length;
	}

	// Unexecute
	for(i = states.length - 1; i >= 0; i--) {
		state = states[i];
		state = command.unexecute(state, false); // Use the "fresh" mode.
		assertFalse("Command "+ i +" should have been unexecuted.", state.executed);
	}
	assertEquals("All the children should be restored.", totalLength, testNode.childNodes.length);

}

/**
 *
 */
function testDeleteManyNext() {
	var command = mozile.edit.remove;
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();
	
	var operations = [ // removed = number of children removed
		{steps: 15, removed: 2},
		{steps: 2, removed: 1},
		{steps: 1, removed: 1},
		{steps: 1, removed: 2},
		{steps: 6, removed: 2},
		{steps: 5, removed: 1}, // 5
		{steps: 1, removed: 3},
		{steps: 6, removed: 5},
		{steps: 9, removed: 3},
		{steps: 13, removed: 2},
		{steps: 13, removed: 2}, 
		{steps: 5, removed: 0}
	];
	var op;

	var event = {
		type: "keypress", 
		keyCode: 46, // delete key
		content: " ",
		ctrlKey: false, 
		metaKey: false
	}
	if(mozile.browser.isIE) event.type = "keydown";
	
	var states = new Array();
	var state;

	resetTestNode();
	selection.collapse(testNode.firstChild, 1);
	var totalLength = testNode.childNodes.length;
	var lastLength = totalLength;
	for(var i=0; i < operations.length; i++) {
		op = operations[i];
		for(var j=0; j < op.steps; j++) {
			state = command.trigger(event);
			states.push(state);
			assertTrue("Command "+ i +","+ j +" should have been executed.", state.executed);
		}
		assertEquals("Previous: Number of testNode children removed for index "+ i +" should be "+ op.removed, op.removed, lastLength - testNode.childNodes.length);
		lastLength = testNode.childNodes.length;
	}

	// Unexecute
	for(i = states.length - 1; i >= 0; i--) {
		state = states[i];
		state = command.unexecute(state, true); // Use the "fresh" mode.
		assertFalse("Command "+ i +" should have been unexecuted.", state.executed);
	}
	assertEquals("All the children should be restored.", totalLength, testNode.childNodes.length);

}

/**
 * 
 */
function testDeleteMergePrevious() {
	mozile.test.shared.preselect();
	var command = mozile.edit.remove;
	var selection = mozile.dom.selection.get();

	var event = {
		type: "keypress", 
		keyCode: 8, // backspace key
		content: " ",
		ctrlKey: false, 
		metaKey: false
	}
	if(mozile.browser.isIE) event.type = "keydown";

	var container = document.getElementById("container");
	var target = document.getElementById("target");
	var firstElement = mozile.dom.getFirstChildElement(container);
	var firstText = firstElement.firstChild.data;
	var secondText = target.firstChild.data;
	selection.collapse(target.firstChild, 0);
	
	// Execute
	var state = command.trigger(event);
	assertTrue("Command "+ i +" should have been executed.", state.executed);

	if(target.parentNode) assertEquals("If the target has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, target.parentNode.nodeType);
	else assertEquals("Check that the target has been removed", null, target.parentNode);
	assertEquals("FirstElement's first child should be a merged text node", firstText + secondText, firstElement.firstChild.data);
	
	// Unexecute
	state = command.unexecute(state, true); // use "fresh" mode
	assertFalse("Command "+ i +" should have been unexecuted.", state.executed);

	assertEquals("Target's parent should be 'container'", container, target.parentNode);
	assertEquals("FirstElement's first child should be firstText", firstText, firstElement.firstChild.data);
	
}

/**
 * 
 */
function testDeleteMergeNext() {
	mozile.test.shared.preselect();
	var command = mozile.edit.remove;
	var selection = mozile.dom.selection.get();


	var event = {
		type: "keypress", 
		keyCode: 46, // backspace key
		content: " ",
		ctrlKey: false, 
		metaKey: false
	}
	if(mozile.browser.isIE) event.type = "keydown";

	var container = document.getElementById("container");
	var target = document.getElementById("target");
	var firstElement = mozile.dom.getFirstChildElement(container);
	var firstText = firstElement.firstChild.data;
	var secondText = target.firstChild.data;
	selection.collapse(firstElement.firstChild, firstText.length);
	
	// Execute
	var state = command.trigger(event);
	assertTrue("Command "+ i +" should have been executed.", state.executed);

	if(target.parentNode) assertEquals("If the target has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, target.parentNode.nodeType);
	else assertEquals("Check that the target has been removed", null, target.parentNode);
	assertEquals("FirstElement's first child should be a merged text node", firstText + secondText, firstElement.firstChild.data);
	
	// Unexecute
	state = command.unexecute(state, true); // use "fresh" mode
	assertFalse("Command "+ i +" should have been unexecuted.", state.executed);

	assertEquals("Target's parent should be 'container'", container, target.parentNode);
	assertEquals("FirstElement's first child should be firstText", firstText, firstElement.firstChild.data);
	
}

/**
 * 
 */
function testDeleteRangeText() {
	mozile.test.shared.preselect();
	var command = mozile.edit.remove;
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target");
	var text = target.firstChild;
	var original = text.data;
	
	// Text only.
	selection.collapse(text, 5);
	selection.extend(text, 15);
	var after = original.substring(0,5) + original.substring(15);
	var state = command.request({}, false);
	assertTrue("The command should have been executed", state.executed);
	assertEquals("Some text should have been removed", after, text.data);
	
	state = command.unexecute(state, false);
	assertFalse("The command should have been unexecuted", state.executed);
	assertEquals("The text node should have been restored", original, text.data);
}

/**
 * 
 */
function testDeleteRangeNodes() {
	mozile.test.shared.preselect();
	var command = mozile.edit.remove;
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target");
	var original1 = target.firstChild.data;
	var original2 = target.lastChild.data;
	
	// Select nodes
	selection.collapse(target.firstChild, 5);
	selection.extend(target.lastChild, 5);
	var after = original1.substring(0,5) + original2.substring(5);
	var length = target.childNodes.length;
	var state = command.request({}, false);
	//alert(target.innerHTML);
	assertTrue("The command should have been executed", state.executed);
	assertEquals("Some text should have been removed", after, target.firstChild.data);
	assertEquals("Target should have only one child node", 1, target.childNodes.length);
	
	state = command.unexecute(state, false);
	assertFalse("The command should have been unexecuted", state.executed);
	assertEquals("The first text node should have been restored", original1, target.firstChild.data);
	assertEquals("The last text node should have been restored", original2, target.lastChild.data);
	assertEquals("Target's child nodes should be restored", length, target.childNodes.length);
}


/**
 * Test without the "preserve" option.
 */
function testDeleteAllChildren() {
	mozile.test.shared.preselect();
	var command = mozile.edit.remove;
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target");
	var target2 = document.getElementById("target2");
	
	// Select nodes
	var IP = mozile.edit.getInsertionPoint(target2, mozile.edit.NEXT);
	IP.select();
	IP = mozile.edit.getInsertionPoint(target2, mozile.edit.PREVIOUS);
	IP.extend();
	var length = target.childNodes.length;
	var length2 = target2.childNodes.length;

	var state = command.request({}, false);
	assertTrue("The command should have been executed", state.executed);
	assertEquals("target should two fewer children", length-2, target.childNodes.length);
	assertEquals("target2 should no children", 0, target2.childNodes.length);
	if(target2.parentNode) assertEquals("If the target2 has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, target2.parentNode.nodeType);
	else assertEquals("Check that the target2 has been removed", null, target2.parentNode);
	
	state = command.unexecute(state, false);
	assertFalse("The command should have been unexecuted", state.executed);
	assertEquals("target's child nodes should be restored", length, target.childNodes.length);
	assertEquals("target2's child nodes should be restored", length2, target2.childNodes.length);
	assertNotNull("Target should have a parentNode", target2.parentNode);
}


/**
 * Test with the "preserve" option.
 */
function testDeleteAndPreserve() {
	mozile.test.shared.preselect();
	var command = mozile.edit.remove;
	var selection = mozile.dom.selection.get();
	var target = document.getElementById("target2");
	
	// Select nodes
	var IP = mozile.edit.getInsertionPoint(target, mozile.edit.NEXT);
	IP.select();
	IP = mozile.edit.getInsertionPoint(target, mozile.edit.PREVIOUS);
	IP.extend();
	var length = target.childNodes.length;

	var state = command.request({}, false, null, null, true);
	assertTrue("The command should have been executed", state.executed);
	assertEquals("Target should have only one child node", 1, target.childNodes.length);
	assertTrue("The child node should be an empty token", mozile.edit.isEmptyToken(target.firstChild));
	
	state = command.unexecute(state, false);
	assertFalse("The command should have been unexecuted", state.executed);
	assertEquals("Target's child nodes should be restored", length, target.childNodes.length);
}

/**
 * 
 */
function testSplitAndMerge() {
	checkSplitAndMerge(1, document.getElementById("target"), 19);
	checkSplitAndMerge(2, document.getElementById("target2"), 4);
}

/**
 * Split nodes and then merge them again using backspace.
 */
function checkSplitAndMerge(i, target, offset) {
	mozile.test.shared.preselect();
	var selection = mozile.dom.selection.get();

	var text = target.firstChild;
	var original = text.data;
	selection.collapse(text, offset);

	var event = {
		type: "keypress",
		charCode: 0,
		keyCode: 13, // enter key
		ctrlKey: false,
		metaKey: false,
		selection: selection,
		range: selection.getRangeAt(0),
		node: text,
		rng: mozile.edit.lookupRNG(text)
	}
	var states = new Array();
	
	// Split the node
	var state = event.rng.handleEvent(event);
	states.push(state);
	assertEquals(i +" The text node should be split", original.substring(0, offset), text.data);
	assertEquals(i +" The focusOffset should be 0", 0, selection.focusOffset);
	
	// Merge the nodes
	event.accel = null;
	event.keyCode = 8; // backspace key
	if(mozile.browser.isIE) event.type = "keydown";
	event.selection = selection;
	event.range = event.selection.getRangeAt(0);
	event.node = selection.focusNode;
	state = event.rng.handleEvent(event);
	states.push(state);
	assertEquals(i +" The text node should be merged", original, text.data);	
	assertEquals(i +" The selection should now be in text", text, selection.focusNode);
	assertEquals(i +" The focusOffset should be the old offset", offset, selection.focusOffset);
	
	for(var j = states.length - 1; j >= 0; j--) {
		states[j].command.unexecute(states[j]);
	}
}

/**
 *
 */
function testChangeBlocks() {
	mozile.test.shared.preselect();
	var command = mozile.edit.remove;
	var selection = mozile.dom.selection.get();

	var event = {
		type: "keypress",
		content: " ",
		ctrlKey: false, 
		metaKey: false
	}
	if(mozile.browser.isIE) event.type = "keydown";
	var startNode, endNode, state;
	var states = new Array();

	// "Previous" Case
	event.keyCode = 8; // backspace key
	startNode = document.getElementById("p1").firstChild;
	endNode = document.getElementById("lastItem").firstChild;
	selection.collapse(startNode, 0);
	state = command.trigger(event);
	states.push(state);
	assertEquals("The focusNode should be endNode", endNode, selection.focusNode);
	assertEquals("The focusOffset should be the last ofset", endNode.data.length, selection.focusOffset);

	// "Next" Case
	event.keyCode = 46; // delete key
	startNode = document.getElementById("target").lastChild;
	endNode = document.getElementById("firstItem").firstChild;
	selection.collapse(startNode, startNode.data.length);
	state = command.trigger(event);
	states.push(state);
	assertEquals("The focusNode should be endNode", endNode, selection.focusNode);
	assertEquals("The focusOffset should be 0", 0, selection.focusOffset);

	for(var i = states.length - 1; i >= 0; i--) {
		states[i].command.unexecute(states[i]);
	}
}


/**
 *
 */
function resetTestNode() {
	if(testNode) testNode.parentNode.removeChild(testNode);
	var test;
	var container = document.getElementById("container");
	if(mozile.browser.isIE) {
		test = document.createElement("div");
		test.innerHTML = testContent;
		testNode = test.firstChild;
		container.appendChild(testNode);
	}
	else {
		var range = document.createRange();
		range.selectNodeContents(container);
		var fragment = range.createContextualFragment(testContent);
		testNode = fragment.firstChild;
		container.appendChild(fragment);
	}
	var selection = mozile.dom.selection.get();
	selection.collapse(testNode.firstChild, 1);
}
