/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */
 
/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's Remove Text command.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: Wrap.js,v 1.2 2006/08/23 16:55:28 jameso Exp $
 */


mozile.require("mozile.edit.rich");
mozile.require("mozile.test.shared.util");
var name = "mozile.edit.Wrap";

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	//return ["testMultipleWrappers"];
	return ["testIsActive", "testCollapsedWrap", "testCollapsedUnwrap", "testCollapsedMove", "testCollapsedToggle", "testCollapsedNested", "testRangeWrapSingle", "testRangeWrapMixed", "testRangeWrapMultiple", "testRangeUnwrapSingle", "testRangeToggle", "testRangeNested", "testRangeMerge", "testMultipleWrappers"];
}

/**
 * 
 */
function testIsActive() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	command.element = "p";
	
	var target = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var container = document.getElementById("container");
	var event = {};

	var test = 1;
	event.node = target2.firstChild;
	assertTrue(test +" Check inside target2", command.isActive(event));
	event.node = target.firstChild;
	assertTrue(test +" Check inside target", command.isActive(event));
	event.node = container;
	assertFalse(test +" Check inside container", command.isActive(event));

	test = 2;
	command.styleName = "color";
	command.styleValue = "red";
	event.node = target2.firstChild;
	assertFalse(test +" Check inside target2", command.isActive(event));
	event.node = target.firstChild;
	assertFalse(test +" Check inside target", command.isActive(event));
}

// Collapsed Tests
// These appy when the selection is collpased. The algorithms are much simpler.

/**
 * Insert a single empty wrapper at the collapsed selection.
 */
function testCollapsedWrap() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);
	command.prompt = function(event, state) { 
		state.test = true; 
		return true;
	}

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	
	var event = { };
	
	// Execute
	var state = command.prepare(event);
	assertNotUndefined("There should be a state");
	assertNotUndefined("There should be a state.wrapper", state.wrapper);
	assertEquals("The wrapper should have the right name", elementName, mozile.dom.getLocalName(state.wrapper).toLowerCase());
	assertEquals("test should be true", true, state.test);
	
	state = command.execute(state);
	assertTrue("The command should have been executed.", state.executed);		
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", state.wrapper, target.childNodes[1]);
	assertEquals("Check the text node contents after execution", 
		original.substring(5), target.childNodes[2].data);
	
	assertEquals("The selection should be in the new element", state.wrapper, selection.focusNode.parentNode);
	assertEquals("The selection offset should be 0", 0, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);

	if(state.wrapper.parentNode) assertEquals("If the state.wrapper has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.wrapper.parentNode.nodeType);
	else assertEquals("Check that the state.wrapper has been removed", null, state.wrapper.parentNode);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Split the wrapper at the collapsed selection, and insert a text node.
 */
function testCollapsedUnwrap() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	var event = { };
	
	// Create a wrapper
	var state1 = command.prepare(event);
	state1 = command.execute(state1);
	var wrapper = state1.wrappers[0];
	assertTrue("The first command should have been executed.", state1.executed);
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", wrapper, target.childNodes[1]);
	assertEquals("The wrapper should contain text", original.substring(5, 15), wrapper.firstChild.data);
	assertEquals("Check the text node contents after execution", 
		original.substring(15), target.childNodes[2].data);

	// Execute
	selection.collapse(wrapper.firstChild, 5);
	var state2 = command.prepare(event);
	state2 = command.execute(state2);
	assertTrue("The second command should have been executed.", state2.executed);
	assertEquals("Check the inner text node contents after execution", 
		original.substring(5, 10), target.childNodes[1].firstChild.data);
	assertEquals("Check the first half of the wrapper", wrapper, target.childNodes[1]);
	assertEquals("There should be an empty text node between wrappers", "", target.childNodes[2].data);
	assertEquals("Check the second half of the wrapper", 
		original.substring(10, 15), target.childNodes[3].firstChild.data);
	
	assertEquals("The selection should be outside the wrapper", target.childNodes[2], selection.focusNode);
	assertEquals("The selection offset should be 0", 0, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Unexecute
	state2 = command.unexecute(state2);
	assertFalse("The first command should have been unexecuted.", state2.executed);
	assertEquals("Check the inner text node contents after unexecution", 
		original.substring(5, 15), wrapper.firstChild.data);
	
	
	// Remove the original wrapper
	state1 = command.unexecute(state1);
	assertFalse("The first command should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Move the selection out of the wrapper.
 */
function testCollapsedMove() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	var event = { };
	
	// Create a wrapper
	var state1 = command.prepare(event);
	state1 = command.execute(state1);
	var wrapper = state1.wrappers[0];
	assertTrue("The first command should have been executed.", state1.executed);

	// Execute
	selection.collapse(wrapper.lastChild, wrapper.lastChild.data.length);
	var state2 = command.prepare(event);
	state2 = command.execute(state2);
	assertTrue("The second command should have been executed.", state2.executed);
	assertEquals("The selection should be past the wrapper", target.childNodes[2], selection.focusNode);
	assertEquals("The selection offset should be 0", 0, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Unexecute
	state2 = command.unexecute(state2);
	assertFalse("The second command should have been unexecuted.", state2.executed);
	assertEquals("The selection should be in the wrapper", wrapper.firstChild, selection.focusNode);
	assertEquals("The selection offset should be 10", 10, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Remove the original wrapper
	state1 = command.unexecute(state1);
	assertFalse("The first command should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Move the selection out of the empty wrapper, then remove the empty wrapper.
 */
function testCollapsedToggle() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	
	var event = { };
	
	// Create a wrapper
	var state1 = command.prepare(event);
	state1 = command.execute(state1);
	var wrapper = state1.wrapper;
	assertTrue("The first command should have been executed.", state1.executed);

	// Execute
	selection.collapse(wrapper.lastChild, wrapper.lastChild.data.length);
	var state2 = command.prepare(event);
	state2 = command.execute(state2);
	assertTrue("The second command should have been executed.", state2.executed);
	// The wrapper should have been removed.
	if(wrapper.parentNode) assertEquals("If the state.wrapper has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, wrapper.parentNode.nodeType);
	else assertEquals("Check that the state.wrapper has been removed", null, wrapper.parentNode);
	assertEquals("The selection should be past the wrapper", target.firstChild, selection.focusNode);
	assertEquals("The selection offset should be 5", 5, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Unexecute
	state2 = command.unexecute(state2);
	assertFalse("The second command should have been unexecuted.", state2.executed);
	assertEquals("The wrapper should be a child of target again", wrapper, target.childNodes[1]);
	assertEquals("The selection should be in the wrapper", wrapper.firstChild, selection.focusNode);
	assertEquals("The selection offset should be 0", 0, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Remove the original wrapper
	state1 = command.unexecute(state1);
	assertFalse("The first command should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Insert a nested wrapper at the collapsed selection.
 */
function testCollapsedNested() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);
	command.nested = true;

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	var event = { };
	
	// Create a wrapper
	var state1 = command.prepare(event);
	state1 = command.execute(state1);
	var wrapper = state1.wrappers[0];
	assertTrue("The first command should have been executed.", state1.executed);
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", wrapper, target.childNodes[1]);
	assertEquals("The wrapper should contain text", original.substring(5, 15), wrapper.firstChild.data);
	assertEquals("Check the text node contents after execution", 
		original.substring(15), target.childNodes[2].data);

	// Execute
	selection.collapse(wrapper.firstChild, 5);
	var state2 = command.prepare(event);
	state2 = command.execute(state2);
	var wrapper2 = state2.wrapper;
	assertTrue("The second command should have been executed.", state2.executed);
	assertEquals("Check the first inner text node contents after execution", 
		original.substring(5, 10), wrapper.firstChild.data);
	assertEquals("The element should have been inserted", wrapper2, wrapper.childNodes[1]);
	assertEquals("The wrapper should contain an empty text node", "", wrapper2.firstChild.data);
	assertEquals("Check the second inner text node contents after execution", 
		original.substring(10, 15), wrapper.childNodes[2].data);
	
	assertEquals("The selection should be inside the second wrapper", wrapper2.firstChild, selection.focusNode);
	assertEquals("The selection offset should be 0", 0, selection.focusOffset);
	assertEquals("The selection should be collapsed", true, selection.isCollapsed);
	
	// Unexecute
	state2 = command.unexecute(state2);
	assertFalse("The first command should have been unexecuted.", state2.executed);
	assertEquals("Check the inner text node contents after unexecution", 
		original.substring(5, 15), wrapper.firstChild.data);
	
	// Remove the original wrapper
	state1 = command.unexecute(state1);
	assertFalse("The first command should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	delete mozile.edit.allCommands[name];
}

// Range Tests
// These appy when the selection is not collpased.

/**
 * A single wrapper wraps a single node.
 */
function testRangeWrapSingle() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	var event = { };
	
	// Execute
	var state = command.prepare(event);
	assertNotUndefined("There should be a state");
	assertNotUndefined("There should be a state.wrapper", state.wrapper);
	assertEquals("The state.wrapper should have the right name", elementName, mozile.dom.getLocalName(state.wrapper).toLowerCase());
	
	state = command.execute(state);
	assertTrue("The command should have been executed.", state.executed);		
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", state.wrappers[0], target.childNodes[1]);
	assertEquals("Check the text node contents after execution", 
		original.substring(15), target.childNodes[2].data);
	
	assertEquals("The selection anchor should be in the new element", state.wrappers[0], selection.anchorNode.parentNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in the new element", state.wrappers[0], selection.focusNode.parentNode);
	assertEquals("The selection focus offset should be 10", 10, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);

	if(state.wrappers[0].parentNode) assertEquals("If the state.wrapper has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.wrappers[0].parentNode.nodeType);
	else assertEquals("Check that the state.wrapper has been removed", null, state.wrappers[0].parentNode);
	
	delete mozile.edit.allCommands[name];
}

/**
 * A single wrapper wraps multiple nodes.
 */
function testRangeWrapMixed() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original1 = target.firstChild.data;
	var original2 = target.childNodes[2].data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.childNodes[2], 15);
	
	var event = { };
	
	// Execute
	var state = command.prepare(event);
	assertNotUndefined("There should be a state");
	assertNotUndefined("There should be a state.wrapper", state.wrapper);
	assertEquals("The state.wrapper should have the right name", elementName, mozile.dom.getLocalName(state.wrapper).toLowerCase());
	
	state = command.execute(state);
	assertTrue("The command should have been executed.", state.executed);		
	assertEquals("Check the text node contents after execution", 
		original1.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", state.wrappers[0], target.childNodes[1]);
	assertEquals("Check the text node contents after execution", 
		original2.substring(15), target.childNodes[2].data);
	
	var wrapper = state.wrappers[0];
	assertEquals("The selection anchor should be in the new element", wrapper.firstChild, selection.anchorNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in the new element", wrapper.lastChild, selection.focusNode);
	assertEquals("The selection focus offset should be 15", 15, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the first text node contents after unexecution", 
		original1, target.firstChild.data);
	assertEquals("Check the second text node contents after unexecution", 
		original2, target.childNodes[2].data);

	if(state.wrappers[0].parentNode) assertEquals("If the state.wrapper has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.wrappers[0].parentNode.nodeType);
	else assertEquals("Check that the state.wrapper has been removed", null, state.wrappers[0].parentNode);
	
	delete mozile.edit.allCommands[name];
}


/**
 * Multiple wrappers wrap multiple nodes.
 */
function testRangeWrapMultiple() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target2.firstChild, 5);
	
	var event = { };
	
	// Execute
	var state = command.prepare(event);
	assertNotUndefined("There should be a state");
	assertNotUndefined("There should be a state.wrapper", state.wrapper);
	assertEquals("The state.wrapper should have the right name", elementName, mozile.dom.getLocalName(state.wrapper).toLowerCase());
	
	state = command.execute(state);
	assertTrue("The command should have been executed.", state.executed);		
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted once", state.wrappers[0], target.childNodes[1]);
	assertEquals("The element should have been inserted twice", state.wrappers[1], target2.childNodes[0]);

	assertEquals("The selection anchor should be in the new element", state.wrappers[0].firstChild, selection.anchorNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in the new element", state.wrappers[1].lastChild, selection.focusNode);
	assertEquals("The selection focus offset should be 5", 5, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);
	
	// Unexecute
	state = command.unexecute(state);
	assertFalse("The command should have been unexecuted.", state.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);

	if(state.wrappers[0].parentNode) assertEquals("If the state.wrapper has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.wrappers[0].parentNode.nodeType);
	else assertEquals("Check that the state.wrapper has been removed", null, state.wrappers[0].parentNode);

	if(state.wrappers[1].parentNode) assertEquals("If the state.wrapper has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, state.wrappers[1].parentNode.nodeType);
	else assertEquals("Check that the state.wrapper has been removed", null, state.wrappers[1].parentNode);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Remove all wrappers in the selection.
 */
function testRangeUnwrapSingle() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	var event = { };
	
	// Create a wrapper
	var state1 = command.prepare(event);
	state1 = command.execute(state1);
	var wrapper = state1.wrappers[0];
	assertTrue("The first command should have been executed.", state1.executed);
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", wrapper, target.childNodes[1]);
	assertEquals("The wrapper should contain text", original.substring(5, 15), wrapper.firstChild.data);
	assertEquals("Check the text node contents after execution", 
		original.substring(15), target.childNodes[2].data);

	// Execute
	selection.collapse(wrapper.firstChild, 2);
	selection.extend(wrapper.firstChild, 8);
	var state2 = command.prepare(event);
	state2 = command.execute(state2);
	assertTrue("The second command should have been executed.", state2.executed);
	assertEquals("Check the inner text node contents after execution", 
		original.substring(5, 7), target.childNodes[1].firstChild.data);
	assertEquals("Check the first half of the wrapper", wrapper, target.childNodes[1]);
	assertEquals("There should be a non-empty text node between wrappers", original.substring(7, 13), target.childNodes[2].data);
	assertEquals("Check the second half of the wrapper", 
		original.substring(13, 15), target.childNodes[3].firstChild.data);
	
	assertEquals("The selection anchor should be in the text node", target.childNodes[2], selection.anchorNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in the text node", target.childNodes[2], selection.focusNode);
	assertEquals("The selection focus offset should be 6", 6, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);
	
	// Unexecute
	state2 = command.unexecute(state2);
	assertFalse("The first command should have been unexecuted.", state2.executed);
	assertEquals("Check the inner text node contents after unexecution", 
		original.substring(5, 15), wrapper.firstChild.data);
	
	
	// Remove the original wrapper
	state1 = command.unexecute(state1);
	assertFalse("The first command should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Add a wrapper and then immediately remove it.
 */
function testRangeToggle() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	var event = { };
	
	// Create a wrapper
	var state1 = command.prepare(event);
	state1 = command.execute(state1);
	var wrapper = state1.wrappers[0];
	assertTrue("The first command should have been executed.", state1.executed);
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", wrapper, target.childNodes[1]);
	assertEquals("The wrapper should contain text", original.substring(5, 15), wrapper.firstChild.data);
	assertEquals("Check the text node contents after execution", 
		original.substring(15), target.childNodes[2].data);
	// The wrapper's text should all be selected.
	assertEquals("The selection anchor should be in the wrapper's text node", wrapper.firstChild, selection.anchorNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in the wrapper's text node", wrapper.firstChild, selection.focusNode);
	assertEquals("The selection focus offset should be 10", 10, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);

	// Execute
	var state2 = command.prepare(event);
	state2 = command.execute(state2);
	assertTrue("The second command should have been executed.", state2.executed);
	assertEquals("The original text should be restored", 
		original, target.firstChild.data);
	
	assertEquals("The selection anchor should be in the text node", target.firstChild, selection.anchorNode);
	assertEquals("The selection anchor offset should be 5", 5, selection.anchorOffset);
	assertEquals("The selection focus should be in the text node", target.firstChild, selection.focusNode);
	assertEquals("The selection focus offset should be 15", 15, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);
	
	// Unexecute
	state2 = command.unexecute(state2);
	assertFalse("The first command should have been unexecuted.", state2.executed);
	assertEquals("Check the inner text node contents after unexecution", 
		original.substring(5, 15), wrapper.firstChild.data);
	
	
	// Remove the original wrapper
	state1 = command.unexecute(state1);
	assertFalse("The first command should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Add a second wrapper inside a first wrapper.
 */
function testRangeNested() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);
	command.nested = true;

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	// Create a wrapper
	var state1 = command.prepare({});
	state1 = command.execute(state1);
	var wrapper = state1.wrappers[0];
	assertTrue("The first command should have been executed.", state1.executed);
	assertEquals("Check the text node contents after execution", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("The element should have been inserted", wrapper, target.childNodes[1]);
	assertEquals("The wrapper should contain text", original.substring(5, 15), wrapper.firstChild.data);
	assertEquals("Check the text node contents after execution", 
		original.substring(15), target.childNodes[2].data);

	// Execute
	selection.collapse(wrapper.firstChild, 2);
	selection.extend(wrapper.firstChild, 8);
	var state2 = command.prepare({});
	state2 = command.execute(state2);
	var wrapper2 = state2.wrappers[0];
	assertTrue("The second command should have been executed.", state2.executed);
	assertEquals("Check the first inner text node contents after execution", 
		original.substring(5, 7), wrapper.firstChild.data);
	assertEquals("Make sure the second wrapper is in place", wrapper2, wrapper.childNodes[1]);
	assertEquals("Check the new wrapper's text node contents after execution", 
		original.substring(7, 13), wrapper2.firstChild.data);
	assertEquals("Check the second inner text node contents after execution", 
		original.substring(13, 15), wrapper.childNodes[2].data);
	
	assertEquals("The selection anchor should be in wrapper2's text node", wrapper2.firstChild, selection.anchorNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in wrapper2's text node", wrapper2.firstChild, selection.focusNode);
	assertEquals("The selection focus offset should be 6", 6, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);
	
	// Unexecute
	state2 = command.unexecute(state2);
	assertFalse("The first command should have been unexecuted.", state2.executed);
	assertEquals("Check the inner text node contents after unexecution", 
		original.substring(5, 15), wrapper.firstChild.data);
	assertEquals("The wrapper should have just one child node", 
		1, wrapper.childNodes.length);
	
	
	// Remove the original wrapper
	state1 = command.unexecute(state1);
	assertFalse("The first command should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Wrap a selection and merge all wrappers.
 */
function testRangeMerge() {
	var name = "testCommand";
	var command = new mozile.edit.Wrap(name);
	var elementName = "element";
	command.element = mozile.dom.createElement(elementName);

	assertEquals("Name should be '"+ name +"'", name, command.name);
	assertEquals("Group should be false", false, command.group);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var targetChildren = target.childNodes.length;
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	var event = { };
	
	// Create a wrapper.
	var state1 = command.prepare(event);
	state1 = command.execute(state1);
	var wrapper = state1.wrappers[0];

	// Remove part of the wrapper.
	selection.collapse(wrapper.firstChild, 2);
	selection.extend(wrapper.firstChild, 8);
	var state2 = command.prepare(event);
	state2 = command.execute(state2);

	// Execute
	selection.collapse(target.childNodes[1].firstChild, 1);
	selection.extend(target.childNodes[3].firstChild, 1);
	var state3 = command.prepare(event);
	state3 = command.execute(state3);

	assertTrue("The third command should have been executed.", state3.executed);
	assertEquals("There should be one wrapper", wrapper, target.childNodes[1]);
	assertEquals("There should be just two more children", targetChildren + 2, target.childNodes.length);
	assertEquals("Check the inner text node contents after execution", 
		original.substring(5, 15), wrapper.firstChild.data);
	
	assertEquals("The selection anchor should be in the wrapper", wrapper.firstChild, selection.anchorNode);
	assertEquals("The selection anchor offset should be 1", 1, selection.anchorOffset);
	assertEquals("The selection focus should be in the wrapper", wrapper.firstChild, selection.focusNode);
	assertEquals("The selection focus offset should be 9", 9, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);


	// Unexecute
	state3 = command.unexecute(state3);
	assertFalse("The first command should have been unexecuted.", state3.executed);


	// Restore the target.
	state2 = command.unexecute(state2);
	state1 = command.unexecute(state1);
	assertFalse("The first command should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution", 
		original, target.firstChild.data);
	
	delete mozile.edit.allCommands[name];
}

/**
 * Wrap a single node with two different wrappers, one after the other.
 */
function testMultipleWrappers() {
	var command1 = new mozile.edit.Wrap("testCommand1");
	var elementName1 = "element1";
	command1.element = mozile.dom.createElement(elementName1);

	var command2 = new mozile.edit.Wrap("testCommand2");
	var elementName2 = "element2";
	command2.element = mozile.dom.createElement(elementName2);

	mozile.test.shared.preselect();
	var target = document.getElementById("target");
	var original = target.firstChild.data;

	var selection = mozile.dom.selection.get();
	selection.collapse(target.firstChild, 5);
	selection.extend(target.firstChild, 15);
	
	// Execute
	var state1 = command1.prepare({});
	state1 = command1.execute(state1);
	var wrapper1 = state1.wrappers[0];
	assertTrue("command1 should have been executed.", state1.executed);		
	assertEquals("Check the text node contents after execution1", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("wrapper1 should have been inserted", wrapper1, target.childNodes[1]);
	assertEquals("Check the wrapper1 text node contents", 
		original.substring(5, 15), wrapper1.firstChild.data);
	assertEquals("Check the text node contents after execution1", 
		original.substring(15), target.childNodes[2].data);
	
	assertEquals("The selection anchor should be in wrapper1", wrapper1, selection.anchorNode.parentNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in wrapper1", wrapper1, selection.focusNode.parentNode);
	assertEquals("The selection focus offset should be 10", 10, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);

	// Execute
	var state2 = command2.prepare({});
	state2 = command2.execute(state2);
	var wrapper2 = state2.wrappers[0];
	assertTrue("command2 should have been executed.", state2.executed);		
	assertEquals("Check the text node contents after execution2", 
		original.substring(0, 5), target.firstChild.data);
	assertEquals("wrapper2 should have been inserted", wrapper2, wrapper1.firstChild);	
	assertEquals("Check the wrapper2 text node contents", 
		original.substring(5, 15), wrapper2.firstChild.data);
	assertEquals("Check the text node contents after execution2", 
		original.substring(15), target.childNodes[2].data);
	
	assertEquals("The selection anchor should be in wrapper2", wrapper2, selection.anchorNode.parentNode);
	assertEquals("The selection anchor offset should be 0", 0, selection.anchorOffset);
	assertEquals("The selection focus should be in wrapper2", wrapper2, selection.focusNode.parentNode);
	assertEquals("The selection focus offset should be 10", 10, selection.focusOffset);
	assertFalse("The selection should not be collapsed", selection.isCollapsed);
	
	// Unexecute
	state2 = command2.unexecute(state2);
	assertFalse("command2 should have been unexecuted.", state2.executed);

	if(wrapper2.parentNode) assertEquals("If the wrapper2 has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, wrapper2.parentNode.nodeType);
	else assertEquals("Check that the wrapper2 has been removed", null, wrapper2.parentNode);

	// Unexecute
	state1 = command1.unexecute(state1);
	assertFalse("command1 should have been unexecuted.", state1.executed);
	assertEquals("Check the text node contents after unexecution1", 
		original, target.firstChild.data);

	if(wrapper1.parentNode) assertEquals("If the wrapper1 has a parent, it has to be a document fragment", mozile.dom.DOCUMENT_FRAGMENT_NODE, wrapper1.parentNode.nodeType);
	else assertEquals("Check that the wrapper1 has been removed", null, wrapper1.parentNode);
	
	delete mozile.edit.allCommands[name];
}

