/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */
 
/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's extensions to the DOM Document object.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: dom.js,v 1.2 2006/08/23 16:55:27 jameso Exp $
 */

mozile.require("mozile.dom");
mozile.require("mozile.xml");

var name = "mozile.dom.*";

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	return ["testAddStyleSheet", "testStyles", "testHTML", "testGetters", "testInsert", "testNamespace", "testCreateNS", "testRemove", "testIgnore", "testIsWhitespace"];
}

/**
 *
 */
function testAddStyleSheet() {
	var links = document.getElementsByTagName("link");
	var count = 0;
	for(var i=0; i < links.length; i++) {
		if(links[i].getAttribute("class") == "mozileLink") count++;
	}
	assertEquals("There should be no Mozile link elements", 0, count);
	
	mozile.dom.addStyleSheet("test.css");
	
	links = document.getElementsByTagName("link");
	count = 0;
	for(var i=0; i < links.length; i++) {
		if(links[i].getAttribute("class") == "mozileLink") count++;
	}
	assertEquals("There should now be 1 Mozile link element", 1, count);
}

/**
 *
 */
function testStyles() {
	var target = document.getElementById("target");
	
	var cssName = "background-color";
	var jsName = "backgroundColor";
	var newName = mozile.dom.convertStyleName(cssName);
	assertEquals("The newName should match the jsName", jsName, newName);
	
	var style = mozile.dom.getStyle(target, "color");
	var oldStyle = style;
	if(mozile.browser.isIE) assertEquals("Color should be black", "#000000", style);
	// TODO: Support other browsers.
	else assertEquals("Color should be black", "rgb(0, 0, 0)", style);
	
	mozile.dom.setStyle(target, "color", "red");
	style = mozile.dom.getStyle(target, "color");
	if(mozile.browser.isIE) assertEquals("Color should now be red", "red", style);
	// TODO: Support other browsers.
	else assertEquals("Color should now be red", "rgb(255, 0, 0)", style);
	
	mozile.dom.setStyle(target, "color", oldStyle);
	assertEquals("Old style should be restored", oldStyle, mozile.dom.getStyle(target, "color"));
}

/**
 *
 */
function testHTML() {
	// HTML Case
	if(document.documentElement.nodeName.toLowerCase() == "html") {
		assertTrue("This is HTML", mozile.dom.isHTML(document));
		var head = document.getElementsByTagName("head");
		assertEquals("There should be a head", head[0], mozile.dom.getHead());
		var body = document.getElementsByTagName("body");
		assertEquals("There should be a body", body[0], mozile.dom.getBody());
	}
	// XML Case
	else {
		assertFalse("This is not HTML", mozile.dom.isHTML(document));
		assertEquals("There should not be a head", document.documentElement, mozile.dom.getHead());
		assertEquals("There should not be a body", document.documentElement, mozile.dom.getBody());
	}
}

/**
 *
 */
function testGetters() {
	var target = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var container = document.getElementById("container");
	
	assertFalse("A child is not an ancestor", mozile.dom.isAncestorOf(target, container));
	assertTrue("A parent is an ancestor", mozile.dom.isAncestorOf(container, target));
	assertFalse("A parent is an ancestor, but the limit is too high", mozile.dom.isAncestorOf(container, target, target));
	
	assertEquals("Check the first child element", target2, mozile.dom.getFirstChildElement(target) );
	assertEquals("Check the next sibling element", document.getElementById("target3"), mozile.dom.getNextSiblingElement(target2) );
	assertEquals("The container sould have child elements", 4, mozile.dom.getChildElements(container).length );

	var common = mozile.dom.getCommonAncestor(target, target2);
	assertEquals("Common ancestor is target", target, common);
	
	var index = 0;
	var node = target.previousSibling;
	while(node) {
		index++;
		node = node.previousSibling;
	}
	assertEquals("Indices should be equal", index, mozile.dom.getIndex(target));

	var position = mozile.dom.getPosition(target);
	assertEquals("Position should be 2", 2, position);

	var x = mozile.dom.getX(target);
	assertEquals("The type of x should be 'number'", "number", typeof(x));
	var y = mozile.dom.getY(target);
	assertEquals("The type of y should be 'number'", "number", typeof(y));

}

/**
 *
 */
function testInsert() {
	var target = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var container = document.getElementById("container");
	
	var element = mozile.dom.createElement("element");
	mozile.dom.insertAfter(element, target.lastChild);
	assertEquals("insertAfter should append the element", element, target.lastChild);
	target.removeChild(element);
	
	element.appendChild(mozile.dom.createElement("child1"));
	element.appendChild(mozile.dom.createElement("child2"));
	assertEquals("The element should have 2 child nodes", 2, element.childNodes.length);
	mozile.dom.removeChildNodes(element);
	assertEquals("The element should have 0 child nodes", 0, element.childNodes.length);
	
	mozile.dom.prependChild(mozile.dom.createElement("child1"), element);
	mozile.dom.prependChild(mozile.dom.createElement("child2"), element);
	assertEquals("The element should have 2 child nodes, again", 2, element.childNodes.length);
	mozile.dom.removeChildNodes(element);
	assertEquals("The element should have 0 child nodes, again", 0, element.childNodes.length);
}

/**
 *
 */
function testRemove() {
	var a = mozile.xml.parseElement('<a><b/>text<!--comment--><c/></a>');
	
	assertEquals("Element 'a' should have 4 child nodes", 4, a.childNodes.length);
	mozile.dom.removeChildNodes(a);
	assertEquals("Element 'a' should have 0 child nodes", 0, a.childNodes.length);
}

/**
 *
 */
function testNamespace() {
	var a = mozile.xml.parseElement('<a xmlns="NS1" xmlns:x="NS2" x:foo="bar"><b xmlns="NS2"/><x:c/></a>');
	var b = a.firstChild;
	var c = b.nextSibling;
	
	assertEquals("The local name of b should be the 'b'", "b", mozile.dom.getLocalName(b) );
	assertEquals("The local name of x:c should be the 'c'", "c", mozile.dom.getLocalName(c) );
	
	assertEquals("The prefix of b should be null", null, mozile.dom.getPrefix(b) );
	assertEquals("The prefix of x:c should be the 'x'", "x", mozile.dom.getPrefix(c) );


	assertEquals("Namespace URI for 'x' should be 'NS2'", "NS2", mozile.dom.lookupNamespaceURI(c, "x") );

	assertEquals("The NS of 'b' is 'NS2'", "NS2", mozile.dom.getNamespaceURI(b));
	assertEquals("The NS of 'c' is 'NS2'", "NS2", mozile.dom.getNamespaceURI(c));

	assertEquals("The default NS of 'b' is 'NS2'", "NS2", mozile.dom.getDefaultNamespaceURI(b));
	assertEquals("The default NS of 'c' is 'NS1'", "NS1", mozile.dom.getDefaultNamespaceURI(c));
	
	assertEquals("The value of 'x:foo' is 'bar'", "bar", mozile.dom.getAttributeNS(a, "NS2", "foo"));

	// Older version of Safari have incomplete namespace support.
	if(mozile.browser.isSafari && mozile.browser.safariVersion <= 418) return;
	
	assertEquals("Namespace prefix for 'NS2' should be 'x'", "x", mozile.dom.lookupPrefix(c, "NS2") );
}

/**
 *
 */
function testCreateNS() {
	var namespace = mozile.xml.ns.rng;
	var elementName = "element";
	var element = mozile.dom.createElementNS(namespace, elementName);
	
	assertEquals("The element name should be "+elementName, elementName, element.nodeName);
	assertEquals("The namespaceURI should be "+namespace, namespace, element.namespaceURI);

}

/**
 *
 */
function testIgnore() {
	var target = document.getElementById("target");
	target.setAttribute("xmlns", "http://cats.com");
	var id = target.getAttributeNode("id");
	var xmlns = target.getAttributeNode("xmlns");

	assertFalse("Should the node be ignored?", mozile.dom.isIgnored(target));
	assertFalse("The id should not be ignored", mozile.dom.isIgnored(id));
	assertTrue("The xmlns should be ignored", mozile.dom.isIgnored(xmlns));
}

/**
 *
 */
function testIsWhitespace() {
	var target1 = document.createTextNode("text");
	var target2 = document.createTextNode(" ");
	
	assertFalse("Target1 is not whitespace", mozile.dom.isWhitespace(target1));
	assertTrue("Target2 is whitespace", mozile.dom.isWhitespace(target2));
}

