/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */
 
/** Mozile Test Script
 * @fileoverview This file contains JsUnit test functions for Mozile's cross-browser range functions.
 * <p>Project Homepage: http://mozile.mozdev.org
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: Range.js,v 1.2 2006/08/23 16:55:27 jameso Exp $
 */


mozile.require("mozile.dom");
var name = "mozile.dom.range.*";

var testNode;
var testContent = '<p><strong>foo<!-- some comment -->bar</strong>Heading 1<em>foo<!-- -->bar</em></p>';

/**
 * Expose the functions named in this array to JsUnit.
 * Make sure to keep it up to date!
 */
function exposeTestFunctionNames() {
	//return ["testCloneContents2"];
	return ["testInit", "testSetStartAndEnd", "testCollapseToStart", "testCollapseToEnd", "testSetRelative", "testSelectNode", "testSelectNodeContents", "testInsertNode", "testCloneContents", "testCloneContents2", "testDeleteContents", "testExtractContents", "testExtractContents2", "testCloneRange", "testStoreRestore" ];
}
//mozile.require("mozile.util");


/**
 *
 */
function testInit() {
	var	r = new mozile.dom.Range();
	assertEquals("Range constructor should create a range object", "object", typeof(r));
}

/**
 *
 */
function testSetStartAndEnd() {
	resetTestNode();
	var r = new mozile.dom.Range();
	r.setStart(testNode.firstChild, 1);
	r.setEnd(testNode.lastChild, 1);
	
	assertEquals("Selection should contain 'barHeading 1foo'",'barHeading 1foo', r.toString());
	r.detach();
}  


/**
 *
 */
function testCollapseToStart() {
	resetTestNode();
	var r = new mozile.dom.Range();
	r.setStart(testNode.firstChild, 1);
	r.setEnd(testNode.lastChild, 1);
	var startContainer = r.startContainer;
	var startOffset = r.startOffset;
	r.collapse(true);
	
	assertEquals("startContainer should be the same before and after collapse(true)", startContainer, r.startContainer);
	assertEquals("startOffset should be the same before and after collapse(true)", startOffset, r.startOffset);
	assertEquals("startContainer should be equal to the endContainer and after collapse(true)", startContainer, r.endContainer);
	assertEquals("startOffset should be equal to the endOffset after collapse(true)", startOffset, r.endOffset);
	assertEquals("collapsed property should be true", true, r.collapsed);
	assertEquals("Range length should be zero", 0, r.toString().length);
	r.detach();
}    

/**
 *
 */
function testCollapseToEnd() {
	resetTestNode();
	var r = new mozile.dom.Range();
	r.setStart(testNode.firstChild, 1);
	r.setEnd(testNode.lastChild, 1);
	var endContainer = r.endContainer;
	var endOffset = r.endOffset;
	r.collapse(false);
	assertEquals("endContainer should be the same before and after collapse(true", endContainer, r.endContainer);
	assertEquals("endOffset should be the same before and after collapse(true", endOffset, r.endOffset);
	assertEquals("startContainer should be equal to the endContainer and after collapse(false)", r.endContainer, r.startContainer);
	assertEquals("startOffset should be equal to the endOffset after collapse(false)", r.endOffset, r.startOffset);
	assertEquals("collapsed property should be true", true, r.collapsed);
	assertEquals("Range length should be zero", 0, r.toString().length);
	r.detach();
}

/**
 *
 */
function testSetRelative() {
	resetTestNode();
	var r = new mozile.dom.Range();
	r.setStartBefore(testNode.firstChild);
	r.setEndBefore(testNode.lastChild);
	assertEquals("Containing text doesn't match", 'foobarHeading 1', r.toString());

	r.setStartBefore(testNode.firstChild);
	r.setEndAfter(testNode.firstChild);
	assertEquals("Containing text doesn't match", 'foobar', r.toString());

	r.setStartBefore(testNode.firstChild);
	r.setEndAfter(testNode.lastChild);
	assertEquals("Containing text doesn't match", 'foobarHeading 1foobar', r.toString());

	r.setStartBefore(testNode);
	r.setEndAfter(testNode);
	assertEquals("Containing text doesn't match", 'foobarHeading 1foobar', r.toString());
	
	r.setStartAfter(testNode.firstChild.firstChild);
	r.setEndAfter(testNode.lastChild);
	assertEquals("Containing text doesn't match", 'barHeading 1foobar', r.toString());
}    

/**
 *
 */
function testSelectNode() {
	resetTestNode();
	var r = new mozile.dom.Range();
	r.selectNode(testNode.firstChild.firstChild);
	assertEquals("Containing text doesn't match", 'foo', r.toString());
	assertEquals("startContainer should be the parentNode", testNode.firstChild.firstChild.parentNode, r.startContainer);
	assertEquals("endContainer should be the parentNode", testNode.firstChild.firstChild.parentNode, r.endContainer);
	
	r.selectNode(testNode.firstChild.firstChild.nextSibling);
	assertEquals("Containing text doesn't match", '', r.toString());
	
	r.selectNode(testNode);
	assertEquals("Containing text doesn't match", 'foobarHeading 1foobar', r.toString());
	
	r.selectNode(testNode.lastChild);
	assertEquals("Containing text doesn't match", 'foobar', r.toString());
	r.detach();
}

/**
 *
 */
function testSelectNodeContents() {
	resetTestNode();
	var r = new mozile.dom.Range();
	r.selectNodeContents(testNode.firstChild);
	assertEquals("Containing text doesn't match (1)", 'foobar', r.toString());
	assertEquals("startContainer should be the given node", testNode.firstChild, r.startContainer);
	assertEquals("endContainer should be the given node", testNode.firstChild, r.endContainer);
	assertEquals("commonAncestorContainer should be the given node", testNode.firstChild, r.commonAncestorContainer);
	
	r.selectNodeContents(testNode.firstChild.firstChild.nextSibling);
	assertEquals("Containing text doesn't match (2)", '', r.toString());
	
	r.selectNodeContents(testNode);
	assertEquals("Containing text doesn't match (3)", 'foobarHeading 1foobar', r.toString());

	r.selectNodeContents(testNode.lastChild);
	assertEquals("Containing text doesn't match (4)", 'foobar', r.toString());
	
	var target = document.getElementById("target");
	r.selectNodeContents(target);
	assertEquals("startContainer should be the target", target, r.startContainer);
	assertEquals("startOffset should be 0", 0, r.startOffset);
	assertEquals("endContainer should be the last child node", target, r.endContainer);
	assertEquals("endOffset should be the target's childNodes length", target.childNodes.length, r.endOffset);
	assertEquals("commonAncestorContainer should be the target", target, r.commonAncestorContainer);
	
	r.detach();
}

/**
 *
 */
function testInsertNode() {
	resetTestNode();
	var node = document.createElement("em");
	node.appendChild(document.createTextNode("Italic"));
	var r = new mozile.dom.Range();
	r.setStartBefore(testNode.firstChild);
	r.setEndBefore(testNode.lastChild);
	r.insertNode(node);
	r.selectNode(testNode);
	//alert(mozile.util.dumpValues(r.store()));
	assertEquals("Containing text doesn't match", 'ItalicfoobarHeading 1foobar', r.toString());
	r.detach();
}

/**
 *
 */
function testCloneContents() {
	resetTestNode();
	var r = new mozile.dom.Range();
	r.selectNode(testNode);
	var originalTxt = r.toString();
	r.setStart(testNode.firstChild.firstChild, 1);
	r.setEndBefore(testNode.lastChild);
	var content = r.cloneContents();
	assertEquals("There should be two child nodes", 2, content.childNodes.length);
	assertEquals("Properly enclose strong tag", 'strong', content.firstChild.tagName.toLowerCase());
	assertEquals("First childnode is a textNode", 'oo', content.firstChild.firstChild.nodeValue);
	assertEquals("Last childnode is a textNode", 'Heading 1', content.lastChild.nodeValue);
	r.selectNode(testNode);
	assertEquals("Source should be untouched", originalTxt, r.toString());
	r.detach();
}

/**
 *
 */
function testCloneContents2() {
	var target = document.getElementById("target");
	var target2 = document.getElementById("target2");
	var r = new mozile.dom.Range();
	r.setStart(target2.firstChild, 5);
	r.setEnd(target.lastChild, 5);
	var content = r.cloneContents();
	assertEquals("There should be one fewer child node", target.childNodes.length - 1, content.childNodes.length);
	assertEquals("Properly enclose span tag", target2.nodeName, content.firstChild.nodeName);
	assertEquals("First childnode is a textNode", target2.firstChild.data.substring(5), content.firstChild.firstChild.data);
	assertEquals("Last childnode is a textNode", target.lastChild.data.substring(0,5), content.lastChild.data);
	r.detach();
}

/**
 *
 */
function testDeleteContents() {
	resetTestNode();
	
	var r = new mozile.dom.Range();
	r.setStart(testNode.firstChild.firstChild, 1);
	r.setEndBefore(testNode.lastChild);
	r.deleteContents();
	assertEquals("Only an 'f' should remain", "f", testNode.firstChild.firstChild.data);
	
	// TODO: Make sure container won't be deleted.
}

/**
 *
 */
function testExtractContents() {
	resetTestNode();
	var r = new mozile.dom.Range();
	r.setStart(testNode.firstChild.firstChild, 1);
	r.setEndBefore(testNode.lastChild);
	var content = r.extractContents();
	assertEquals("Properly enclose strong tag", 'strong', content.firstChild.tagName.toLowerCase());
	r.selectNode(testNode.firstChild);
	assertEquals("first textNode should be split", 'f', r.toString());
}

/**
 *
 */
function testExtractContents2() {
	resetTestNode();
	var el = testNode.firstChild;
	var num = el.childNodes.length;
	
	var r = new mozile.dom.Range();
	r.selectNodeContents(el);
	var content = r.extractContents();
	assertEquals("Extracted content should contain all children of the selectNodeContents reference node.", num, content.childNodes.length);
}

/**
 *
 */
function testCloneRange() {
	var target = document.getElementById("target");
	var r = new mozile.dom.Range();
	r.setStart(target.firstChild, 1);
	r.setEnd(target.lastChild, 1);

	var s = r.cloneRange();
	assertEquals("startContainer should match", r.startContainer, s.startContainer);
	assertEquals("startOffset should match", r.startOffset, s.startOffset);
	assertEquals("endContainer should match", r.endContainer, s.endContainer);
	assertEquals("endOffset should match", r.endOffset, s.endOffset);
	assertEquals("commonAncestorContainer should match", r.commonAncestorContainer, s.commonAncestorContainer);
	assertEquals("collapsed should match", r.collapsed, s.collapsed);

}

/**
 *
 */
function testSurroundContent() {
	// Trivial, provided extractContent and insertNode work.
}

/**
 *
 */
function testCompareBoundaryPoints() {
	// Not yet implemented.
}

/**
 *
 */
function testStoreRestore() {
	resetTestNode();

	var range1 = new mozile.dom.Range();
	range1.selectNode(testNode);
	var state = range1.store();
	
	assertNotNull("There should be a stored state", state);
	//alert(state.format +" "+ state.bookmark);
	
	var range2 = new mozile.dom.Range();
	range2.restore(state);

	//assertEquals("Start should equal start", 0, range1.compareBoundaryPoints(range1.START_TO_START, range2));
	//assertEquals("End should equal end", 0, range1.compareBoundaryPoints(range1.END_TO_END, range2));
	
	assertNotNull("range1 startContainer should not be null", range1.startContainer);
	assertNotNull("range2 startContainer should not be null", range2.startContainer);
	assertEquals("startContainer names should be equal", range1.startContainer.nodeName, range2.startContainer.nodeName);

	assertEquals("startContainers should be equal", range1.startContainer, range2.startContainer);
	assertEquals("startOffsets should be equal", range1.startOffset, range2.startOffset);
	assertEquals("endContainers should be equal", range1.endContainer, range2.endContainer);
	assertEquals("endOffsets should be equal", range1.endOffset, range2.endOffset);
}



/**
 *
 */
function resetTestNode() {
	if(testNode) testNode.parentNode.removeChild(testNode);
	var test;
	var container = document.getElementById("container");
	if(mozile.browser.isIE) {
		test = document.createElement("div");
		test.innerHTML = testContent;
		testNode = test.firstChild;
		container.appendChild(testNode);
	}
	else {
		var range = document.createRange();
		range.selectNodeContents(container);
		var fragment = range.createContextualFragment(testContent);
		testNode = fragment.firstChild;
		container.appendChild(fragment);
	}
}

