/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */

/**
 * @fileoverview Mozile's core properties, methods, and classes.
 * <p>Project Homepage: http://mozile.mozdev.org 
 * @author James A. Overton <james@overton.ca>
 * @version 0.8
 * $Id: core.js,v 1.3 2006/08/23 16:47:53 jameso Exp $
 */

// End of Headers


/**
 * The root Mozile object.
 * @type Object
 */
var mozile = {
	window: window,
	document: document,
	filesep: "/",
	linesep: "\n",
	root: ""
}

//if(navigator.platform.indexOf("Win") > -1) mozile.linesep = "\r\n";

/**
 * Set the version number.
 * @type Object
 */
mozile.version = {
	major: "0",
	minor: "8",
	micro: "0",
	flag:  "a1",
	toString: function() {
		with (mozile.version) {
			return major + "." + minor + "." + micro + flag;
		}
	}
}

/**
 * Other Mozile information.
 */
mozile.about = "Mozile is a WYSIWYG inline edtior for XHTML and XML+CSS files.";
mozile.copyright = "Copyright 2006, The Mozile Team";
mozile.license = "MPL 1.1/GPL 2.0/LGPL 2.1, http://mozile.mozdev.org/0.8/LICENSE";
mozile.homepage = "http://mozile.mozdev.org";
mozile.credits = "James A. Overton, Conor Dowling, Max d'Ayala, Christian Stocker, Paul Everitt, David Palm, Richard Prescott, Lars Kiilerich, Kingsley Kerse, Tobias Minich, Andreas Schamberger, and others...";
mozile.acknowledgements = "Icons created by Mark James <http://www.famfamfam.com/lab/icons/silk/>";



/**** Configuraton Defaults ****/

/**
 * This is set automatically by Mozile's build tools. When true it indicates that Mozile has been "compiled" into a single file.
 * @type Boolean
 */
mozile.precompiled = false;

/**
 * When true, designMode will be used by Mozile in browsers that support it. When false designMode will not be changed.
 * @type String
 */
mozile.help = [mozile.root, "doc", "html", "index.html"].join(mozile.filesep);

/**
 * When true, designMode will be used by Mozile in browsers that support it. When false designMode will not be changed.
 * @type Boolean
 */
mozile.useDesignMode = false;

/**
 * The default namespace is null. This means that the document's namespace is always used, which will usually be HTML.
 * @type String
 */
mozile.defaultNS = null;

/**
 * The empty token is used when there is a text node which is only a place-holder inside an empty element. By default it is a non-breaking space character.
 * @type String
 */
mozile.emptyToken = "\u00A0"; 
// \u00A0 = non-breaking space
// \uFEFF = zero-width non-breaking space

/**
 * Specify a non-breaking space character to be used throughout.
 * @type String
 */
mozile.alternateSpace = null;
// \u00A0 = non-breaking space

/**
 * The default update interval in milliseconds.
 * @type Integer
 */
mozile.updateInterval = 500;


/**** Bootstrapping Methods ****/

/**
 * A cross-browser compatible loading method. 
 * Uses XMLHttpRequest to get the text content of a file.
 * @param {String} uri The URI of the file to be loaded.
 * @type String
 */
mozile.load = function(uri) { 
	var XHR;

	try{
		if(window.XMLHttpRequest) {
			XHR = new XMLHttpRequest();
			XHR.open("GET", uri, false);
			XHR.send(null);
		}
		else if(window.ActiveXObject) {
			XHR = new ActiveXObject('Microsoft.XMLHTTP');
			XHR.open("GET", uri, false);
			XHR.send();
		}
	} catch(e) { 
		if(mozile.debug) mozile.debug.inform("mozile.load", "File load failed loading '"+ uri +"' with error message:\n"+ e);
		return null;
	}
	
	if(XHR) {	
		//alert(XHR +" "+ XHR.status);
		if(XHR.status == 0 || XHR.status == 200) return XHR.responseText;
		else {
			if(mozile.debug) mozile.debug.inform("mozile.load", "File load failed with status '"+ XHR.status +"' and message:\n"+ XHR.responseText);
			return null;
		}
	}
	
	if(mozile.debug) mozile.debug.inform("mozile.load", "No XMLHttpRequest available when trying to load '"+ uri +"'.");
	return null;
}

/**
 * Detects the path to this file by searching for a script with src "mozile.js".
 * The path is cleaned and returned.
 * @type String
 */
mozile.findRoot = function() {
	var scripts = document.getElementsByTagName("script");
	for(var i=0; i < scripts.length; i++) {
		if(scripts[i].getAttribute("src").indexOf("mozile.js") > -1) {
			var root = scripts[i].getAttribute("src");
			root = root.substring(0, root.lastIndexOf("mozile.js"));
			if(root.indexOf("src") > -1)
				root = root.substring(0, root.lastIndexOf("src"));
			return root;
		}
	}
	return "";
}

// Detect the mozile.root.
mozile.root = mozile.findRoot();



/**** Module System ****/

/**
 * This system a simpler version of the one used by the Dojo Toolkit (http://dojotoolkit.org).
 * "Modules" are just the parts of a hierarchy of objects building on the global "mozile" object. At each level they may contain properties, methods, or classes.
 * Names follow the familiar library naming convention: e.g. "mozile.dom.Document".
 * <p>Some modules are stored in their own JavaScript files in the "src" directory of the Mozile installation.
 * Module files must specify their requirements and the sub-modules they provide using mozile.require() and mozile.provide().
 * <p>The module can be retreived using mozile.findModule().
 * Modules are loaded with mozile.loadModule()
 */

/**
 * Finds an object in the "mozile" hierarchy, if it exists.
 * Given a name of the format "mozile.foo.bar", it starts with the global "window" object and checks for the presence of a "mozile" property.
 * Then it checks the "mozile" object for a "foo" property, and so on.
 * @param {String} moduleName A module name of the form "mozile.foo" or "mozile.foo.Bar".
 * @type Object
 */
mozile.findModule = function(moduleName) {
	var levels = moduleName.split(".");
	var current = mozile.window;
	for(var i=0; i < levels.length; i++) {
		if(!current[levels[i]]) return null;
		current = current[levels[i]];
	}
	return current;
}


/**
 * Tries to load a module of code.
 * The method first tries to load the file "$ROOT/src/foo/bar.js".
 * If that fails, it tries to load "$ROOT/src/foo.js".
 * <p>When a file is successfully loaded it is then evaluated using eval().
 * Note that it is evaluated in the context of this function, which will cause problems for any variables in the file which are declared outside the "mozile" hierarchy.
 * @param {String} moduleName A module name of the form "mozile.foo" or "mozile.foo.Bar".
 * @type Boolean
 */
mozile.loadModule = function(moduleName) {
	if(!moduleName) return false;

	var filename = mozile.root + mozile.filesep;
	var levels = moduleName.split(".");
	for(var i=0; i < levels.length; i++) {
		if(levels[i] == "*") levels = levels.splice(i,1);
	}
	if(levels[0] && levels[0] == "mozile") {
		levels.shift();
		if(levels[0] && levels[0] == "test") {
			levels.shift();
			filename += "tests" + mozile.filesep;
		}
		else filename += "src" + mozile.filesep;
	}
	else return false;
	if(levels.length == 0) return false; 
	//alert("Loading module "+moduleName);

	while(levels.length > 1) {
		filename += levels.shift() + mozile.filesep;
	}
	filename += levels.pop() +".js";
	//alert("Filename "+ filename);

	var file = mozile.load(filename);
	// Fix problem loading within JsUnit test files.
	if(!file && mozile.test && mozile.test.root) {
		filename = filename.replace(mozile.root, mozile.test.root);
		file = mozile.load(filename);
	}
	if(!file) {
		//alert("No file for '"+ moduleName +"' at '"+ filename +"' starting at '"+ location +"'.");
		var parent = moduleName.substring(0, moduleName.lastIndexOf("."));
		return mozile.loadModule(parent);
	}
	
	try {
		eval(file);
		return true;
	} catch(e) {
		mozile.debug.inform("mozile.loadModule", "Error evaluating module '"+ moduleName +"' in file '"+ filename +"'.\n"+ mozile.dumpError(e));
		return false;
	}
}

/**
 * Loads a module if it has not already been loaded.
 * This method is used by modules to indicate that they require the support of another module.
 * @param {String} moduleName A module name of the form "mozile.foo" or "mozile.foo.Bar".
 * @type Boolean
 */
mozile.require = function(moduleName) {
	if(!mozile.findModule(moduleName)) {
		if(mozile.precompiled) return false;
		return mozile.loadModule(moduleName);
	}
	else return false;
}

/**
 * This function is used by modules to indicate what properties, methods, and classes they provide.
 * Currently the function does nothing, but its use it still required for all modules.
 * @type Void
 */
mozile.provide = function() {
	// startPackage?
}







/**
 * A container for Mozile code.
 * This is a hack to make JSDoc show modules properly.
 * @constructor
 */
mozile.Module = function() {};





/**** File Tools ****/

/**
 * Determines whether a path is absolute.
 * @param {String} path The path to check.
 * @type Boolean
 */
mozile.isPathAbsolute = function(path) {
	if(!path) return false;
	path = path.toString();
	if(path.indexOf(":") == -1) return false;

	var protocol = path.substring(0, path.indexOf(":"));
	switch(protocol) {
		case "http":
		case "https":
		case "file":
			return true;
	}

	return false;
}

/**
 * Strips non-path information from a URL.  
 * @param {String} url The URL to strip.
 * @type String
 */
mozile.getPath = function(url) {
	if(!url) return "";
	url = url.toString();
	
	// Strip GET arguments.
	if(url.indexOf("?") > -1) url = url.substring(0, url.indexOf("?"));
	
	return url;
}

/**
 * Gets a directory from a path. Becareful, because this method isn't very smart.
 * TODO: Handle Windows paths.
 * @param {String} path The path to use.
 * @type String
 */
mozile.getDirectory = function(path) {
	if(!path) return "";
	path = path.toString();
	path = mozile.getPath(path);
	
	// Strip everything after the last file separator.
	if(path.indexOf(mozile.filesep) > -1) {
		if(path.indexOf(":///") > -1) {
			var partial = path.substring(path.indexOf(":///")+4);
			if(partial.indexOf(mozile.filesep) == -1) return path;
		}
		else if(path.indexOf("://") > -1) {
			var partial = path.substring(path.indexOf("://")+3);
			if(partial.indexOf(mozile.filesep) == -1) return path;
		}
		return path.substring(0, path.lastIndexOf(mozile.filesep)+1);
	}
	
	return path;
}

/**
 * Ensures that a path is absolute. Appends the path to the root if it is not absolute.
 * @param {String} path The path to convert.
 * @param {String} root An absolute root URL. If none is given the document location is used.
 * @type String
 */
mozile.getAbsolutePath = function(path, root) {
	if(!path) return "";
	path = path.toString();
	if(mozile.isPathAbsolute(path)) return path;
	
	if(!root) root = location;
	root = mozile.getDirectory(root);
	root = mozile.getAbsolutePath(root);
	return root + path;
}






/**** Debugging Tools ****/

/**
 * Tools for debugging Mozile.
 * @type Object
 */
mozile.debug = new Object();
// JSDoc hack
mozile.debug.prototype = new mozile.Module;

/**
 * Indicates the level of debugging information to be delivered to the user.
 * Values can be "suppress", "warn", "inform", or "debug", in order from least verbose to most verbose.
 * @type String
 */
mozile.debug.alertLevel = "suppress";

/**
 * Indicates the level of debugging information to be stored in the messages array.
 * Values can be "suppress", "warn", "inform", or "debug", in order from least verbose to most verbose.
 * @type String
 */
mozile.debug.logLevel = "inform";

/**
 * An array of debugging messages.
 * @type Array
 */
mozile.debug.messages = new Array();

/**
 * A window where logged messages are sent.
 * @type Window
 */
mozile.debug.window = null;

/**
 * Compares two debugging levels.
 * @param {String} type The level to check against. Can be "alert" or "log".
 * @param {String} level The level to check. Can be "suppress", "warn", "inform", or "debug".
 * @type Boolean
 */
mozile.debug.isSelected = function(type, level) {
	// Setup
	if(typeof(type) != "string") return false;
	type = type.toLowerCase();
	var checkLevel;
	if(type == "alert") checkLevel = mozile.debug.alertLevel;
	else if(type == "log") checkLevel = mozile.debug.logLevel;
	else return false;
	checkLevel = checkLevel.toLowerCase();
	if(typeof(level) != "string") return false;
	level = level.toLowerCase();

	// Compare levels
	if(checkLevel == "suppress") return false;
	if(checkLevel == "warn") {
		if(level =="warn") return true;
		else return false;
	}
	if(checkLevel == "inform") {
		if(level == "warn" || level == "inform") return true;
		else return false;
	}
	if(checkLevel == "debug") return true;

	return false;
}


/**
 * A high priority debugging notification.
 * @param {String} caller The name of the function sending the notification.
 * @param {String} message The warning.
 * @type Void
 */
mozile.debug.warn = function(caller, message) {
	var level = "warn";

	var msg = "Mozile Warning ["+ caller +"] "+ message;
	if(mozile.debug.isSelected("alert", level)) {
		if(window.warn) warn(msg); // JsUnit tracing
		else mozile.alert(msg);
	}
	if(mozile.debug.isSelected("log", level)) {
		mozile.debug.log(caller, level, message);
	}
}

/**
 * A medium priority debugging notification.
 * @param {String} caller The name of the function sending the notification.
 * @param {String} message The warning.
 * @type Void
 */
mozile.debug.inform = function(caller, message) {
	var level = "inform";

	var msg = "Mozile Information ["+ caller +"] "+ message;
	if(mozile.debug.isSelected("alert", level)) {
		if(window.inform) inform(msg); // JsUnit tracing
		else mozile.alert(msg);
	}
	if(mozile.debug.isSelected("log", level)) {
		mozile.debug.log(caller, level, message);
	}
}

/**
 * A low priority debugging notification.
 * @param {String} caller The name of the function sending the notification.
 * @param {String} message The warning.
 * @type Void
 */
mozile.debug.debug = function(caller, message) {
	var level = "debug";

	var msg = "Mozile Debugging ["+ caller +"] "+ message;
	if(mozile.debug.isSelected("alert", level)) {
		if(window.debug) debug(msg); // JsUnit tracing
		else mozile.alert(msg);
	}
	if(mozile.debug.isSelected("log", level)) {
		mozile.debug.log(caller, level, message);
	}
}

/**
 * Store a debugging message in the messages array.
 * @param {String} caller The name of the function sending the warning.
 * @param {String} level The level of the message.
 * @param {String} message The warning.
 * @type Void
 */
mozile.debug.log = function(caller, level, message) {
	var date = new Date();
	var msg =  { 
		caller: caller, 
		level: level, 
		message: message, 
		date: date.toLocaleString(),
		toString: function() {
			return this.level.toUpperCase() +" ("+ this.date +") ["+ this.caller +"] "+ this.message;
		}
	};
	mozile.debug.messages.push(msg);
	
	// If the window is open, print the message to it and scroll to the bottom.
	if(mozile.debug.window && mozile.debug.window.document) {
		mozile.debug.window.document.write(msg +"<br/>\n");
		mozile.debug.window.scroll(0, document.body.clientHeight);
	}
}

/**
 * Write all logged debugging messages to a new window.
 * @type Void
 */
mozile.debug.show = function() {
	if(!mozile.debug.window || !mozile.debug.window.document) {
		mozile.debug.window = window.open("", "MozileDebugging", "");
		mozile.debug.window.document.write("<h3>Mozile Debugging Messages</h3>");
		mozile.debug.window.document.write(mozile.debug.messages.join("<br/>\n") + "<br/>\n");
	}
	else mozile.debug.window = window.open("", "MozileDebugging", "");
}


/**
 * Informs the user about a problem. 
 * The method used to inform the user varies by what
 * @param {String} message
 * @type Void
 */
mozile.alert = function(message) {
	// TODO: Firebug support
	// TODO: Safari console support
	alert(message);
}


/**
 * Dumps information from error objects.
 * Internet Explorer needs special attention.
 * @param object The error object or object that the error is attached to.
 * @type String
 */
mozile.dumpError = function(object) {
	if(typeof(object) == "string") return object;
	if(!mozile.browser.isIE) return object.toString();

	// Error Object case
	if(object && object.description) {	
		var fields = [
			"Name: "+ object.name,
			"Number: "+ object.number,
			"Message: "+ object.message,
			"Description: "+ object.description,
		];
		return fields.join("\n");
	}

	// parseError case
	if(!object) object = document;
	if(!object.parseError) object = document;
	if(!object.parseError) object = window;
	if(!object.parseError) return "[No error to parse]";
	var fields = [
		"Error Code: "+ object.parseError.errorCode,
		"File Position: "+object.parseError.filepos,
		"Line: "+object.parseError.line,
		"Line Position: "+object.parseError.linepos,
		"Reason: "+object.parseError.reason,
		"Source Text: "+object.parseError.srcText,
		"Url: "+object.parseError.url
	];
	
	return fields.join("\n");
}

/**
 * Makes a single element editable.
 * @param elementOrId Either a DOM element or the id of an element.
 * @type Void
 */
mozile.editElement = function(elementOrId) {
	var element;
	if(typeof(elementOrId) == "string") {
		if(document.documentElement.nodeName.toLowerCase() == "html") // HTML case
			element = document.getElementById(elementOrId);
		else return mozile.editElements(elementOrId, "id", true); // XML case
	}
	else if(elementOrId.nodeType && 
		elementOrId.nodeType == mozile.dom.ELEMENT_NODE) {
		element = elementOrId;
	}

	if(element) {
		mozile.require("mozile.edit");
		mozile.edit.setMark(element, "editable", true);
		// Store the value of contentEditable.
		switch(element.getAttribute("contentEditable")) {
			case "true":
				mozile.edit.setMark(element, "contentEditable", true);
				break;
			case "false":
				mozile.edit.setMark(element, "contentEditable", false);
				break;
		}
		// Set contentEditable to true.
		if(mozile.browser.isIE) {
			element.setAttribute("contentEditable", "true");
		}
	}
	return undefined;
}

/**
 * Makes a set of elements editable.
 * @param listOrValue Either an array (or nodeList) of DOM elements or the value of an attribute (defaults to "class").
 * @param {String} attribute Optional. The name of an attribute to search for. Defaults to "class". If the value is null then the local name of the element is used (lowercase).
 * @param {Boolean} single Optional. When true, only one item is added.
 * @type Void
 */
mozile.editElements = function(listOrValue, name, single) {
	mozile.require("mozile.dom");
	var list = new Array();
	if(name === undefined) name = "class";
	if(typeof(listOrValue) == "string") {
		if(document.createTreeWalker) {
			var treeWalker = document.createTreeWalker(document.documentElement, mozile.dom.NodeFilter.SHOW_ELEMENT, null, false);
			while(treeWalker.nextNode()) {
				var node = treeWalker.currentNode;
				if(name == "class" && mozile.browser.isIE && node.className == listOrValue) list.push(node);
				else if(name && node.getAttribute(name) == listOrValue) list.push(node);
				else if(!name && mozile.dom &&
					mozile.dom.getLocalName(node).toLowerCase() == listOrValue) {
					list.push(node);
				}
				if(single && list.length > 0) break;
			}
		}
	}
	else if(listOrValue.length) {
		list = listOrValue;
	}
	
	if(list.length) {
		for(var i=0; i < list.length; i++) {
			if(list[i] && list[i].nodeType && 
				list[i].nodeType == mozile.dom.ELEMENT_NODE) {
				mozile.editElement(list[i]);
				if(single) break;
			}
		}
	}
	
	return undefined;
}

/**
 * Makes the whole document editable.
 * @param {Document} doc Optional. The document to make editable. Defaults to the current document.
 * @type Void
 */
mozile.editDocument = function(doc) {
	if(!doc) doc = document;
	mozile.editElement(doc.documentElement);
}

/**
 * Makes all text nodes inside editable elements editable by adding default commands to mozile.edit.
 * @param {Boolean} rich Optional. When true rich editing commands are included.
 * @type Void
 */
mozile.editAllText = function(rich) {
	mozile.require("mozile.edit");
	mozile.edit.addDefaultCommand(mozile.edit.navigateLeftRight);
	mozile.edit.addDefaultCommand(mozile.edit.insertText);
	mozile.edit.addDefaultCommand(mozile.edit.removeText);
	
	if(rich) {
		mozile.require("mozile.edit.rich");
		mozile.edit.addDefaultCommand(mozile.edit.remove);
		var splitBlocks = new mozile.edit.Split("splitBlocks");
		splitBlocks.accel = "Return Enter";
		mozile.edit.addDefaultCommand(splitBlocks);
	}
}

/**
 * Load the target RNG schema and use it as the schema for this document.
 * @param target The desired schema. Can be any value accepted by mozile.rng.Schema.parse().
 * @type Boolean
 * @return True if the schema is found and parsed correctly.
 */
mozile.useSchema = function(target) {
	try {
		mozile.require("mozile.rng");
		mozile.require("mozile.edit");
		mozile.require("mozile.edit.rich");
		mozile.edit.extendRNG();
		mozile.schema = new mozile.rng.Schema();
		var validation = mozile.schema.parse(target);
		if(validation.isValid) {
			mozile.edit.generateCommands(mozile.schema);
			return true;
		}
		else {
			mozile.debug.inform("mozile.useSchema", "Schema validation failed.\n"+ validation.report(true));
			return false;
		}
	} catch(e) {
		mozile.debug.inform("mozile.useSchema", "Could not create schema for target '"+ target +"' because of an error:\n"+ mozile.dumpError(e));
		return false;
	}
}



/**** Operating System Detection ****/

/**
 * Tools for dealing with the operating system.
 * @type Object
 */
mozile.os = new Object();
// JSDoc hack
mozile.os.prototype = new mozile.Module;

/**
 * Check to see if the browser is running on a Mac.
 * @type Boolean
 */
mozile.os.isMac = false;
if(navigator.userAgent.match(/Macintosh/)) mozile.os.isMac = true;



/**** Browser Detection ****/

/**
 * Tools for dealing with the browser.
 * @type Object
 */
mozile.browser = new Object();
// JSDoc hack
mozile.browser.prototype = new mozile.Module;


/**
 * Check to see if this browser is Mozilla.
 * @type Boolean
 */
mozile.browser.isMozilla = false;

/**
 * If this is a Mozilla browser, this stores a number representing the major and minor versions. E.g. "1.7" or "1.8".
 * @type Number
 */
mozile.browser.mozillaVersion = 0;

mozile.browser.mozillaVersion = navigator.userAgent.match(/rv\:(\d+\.\d+)/);
if(mozile.browser.mozillaVersion && 
	Number(mozile.browser.mozillaVersion[1])) {
	mozile.browser.isMozilla = true;
	mozile.browser.mozillaVersion = Number(mozile.browser.mozillaVersion[1]);
}

/**
 * Check to see if this browser is Internet Explorer.
 * @type Boolean
 */
mozile.browser.isIE = false;
if(navigator.userAgent.match(/MSIE/)) mozile.browser.isIE = true;

/**
 * Check to see if this browser is Safari.
 * @type Boolean
 */
mozile.browser.isSafari = false;
if(navigator.userAgent.match(/Safari/)) mozile.browser.isSafari = true;

/**
 * If this is a Safari browser, this property stores a number representing the WebKit version. E.g. "418" or "420".
 * @type Number
 */
mozile.browser.safariVersion = 0;

mozile.browser.safariVersion = navigator.userAgent.match(/AppleWebKit\/(\d+)/);
if(mozile.browser.safariVersion && 
	Number(mozile.browser.safariVersion[1])) {
	mozile.browser.safariVersion = Number(mozile.browser.safariVersion[1]);
}

/**
 * Check to see if this browser is Opera.
 * @type Boolean
 */
mozile.browser.isOpera = false;
if(navigator.userAgent.match(/Opera/)) mozile.browser.isOpera = true;




/**** Final Configuration ****/

// Detect the content type as HTML or XML in Mozilla.
if(mozile.browser.isMozilla) {
	if(document.contentType && document.contentType == "text/html") {
		mozile.defaultNS = null;
	}
	else {
		mozile.defaultNS = "http://www.w3.org/1999/xhtml";
	}
}

