/* ***** BEGIN LICENSE BLOCK *****
 * Licensed under Version: MPL 1.1/GPL 2.0/LGPL 2.1
 * Full Terms at http://mozile.mozdev.org/0.8/LICENSE
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is James A. Overton's code (james@overton.ca).
 *
 * The Initial Developer of the Original Code is James A. Overton.
 * Portions created by the Initial Developer are Copyright (C) 2005-2006
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *	James A. Overton <james@overton.ca>
 *
 * ***** END LICENSE BLOCK ***** */




/**
 * The root Mozile object.
 * @type Object
 */
var mozile = {
	window: window,
	document: document,
	filesep: "/",
	linesep: "\n",
	root: ""
}

//if(navigator.platform.indexOf("Win") > -1) mozile.linesep = "\r\n";

/**
 * Set the version number.
 * @type Object
 */
mozile.version = {
	major: "0",
	minor: "8",
	micro: "0",
	flag:  "a1",
	toString: function() {
		with (mozile.version) {
			return major + "." + minor + "." + micro + flag;
		}
	}
}

/**
 * Other Mozile information.
 */
mozile.about = "Mozile is a WYSIWYG inline edtior for XHTML and XML+CSS files.";
mozile.copyright = "Copyright 2006, The Mozile Team";
mozile.license = "MPL 1.1/GPL 2.0/LGPL 2.1, http://mozile.mozdev.org/0.8/LICENSE";
mozile.homepage = "http://mozile.mozdev.org";
mozile.credits = "James A. Overton, Conor Dowling, Max d'Ayala, Christian Stocker, Paul Everitt, David Palm, Richard Prescott, Lars Kiilerich, Kingsley Kerse, Tobias Minich, Andreas Schamberger, and others...";
mozile.acknowledgements = "Icons created by Mark James <http://www.famfamfam.com/lab/icons/silk/>";



/**** Configuraton Defaults ****/

/**
 * This is set automatically by Mozile's build tools. When true it indicates that Mozile has been "compiled" into a single file.
 * @type Boolean
 */
mozile.precompiled = false;

/**
 * When true, designMode will be used by Mozile in browsers that support it. When false designMode will not be changed.
 * @type String
 */
mozile.help = [mozile.root, "doc", "html", "index.html"].join(mozile.filesep);

/**
 * When true, designMode will be used by Mozile in browsers that support it. When false designMode will not be changed.
 * @type Boolean
 */
mozile.useDesignMode = false;

/**
 * The default namespace is null. This means that the document's namespace is always used, which will usually be HTML.
 * @type String
 */
mozile.defaultNS = null;

/**
 * The empty token is used when there is a text node which is only a place-holder inside an empty element. By default it is a non-breaking space character.
 * @type String
 */
mozile.emptyToken = "\u00A0"; 
// \u00A0 = non-breaking space
// \uFEFF = zero-width non-breaking space

/**
 * Specify a non-breaking space character to be used throughout.
 * @type String
 */
mozile.alternateSpace = null;
// \u00A0 = non-breaking space

/**
 * The default update interval in milliseconds.
 * @type Integer
 */
mozile.updateInterval = 500;


/**** Bootstrapping Methods ****/

/**
 * A cross-browser compatible loading method. 
 * Uses XMLHttpRequest to get the text content of a file.
 * @param {String} uri The URI of the file to be loaded.
 * @type String
 */
mozile.load = function(uri) { 
	var XHR;

	try{
		if(window.XMLHttpRequest) {
			XHR = new XMLHttpRequest();
			XHR.open("GET", uri, false);
			XHR.send(null);
		}
		else if(window.ActiveXObject) {
			XHR = new ActiveXObject('Microsoft.XMLHTTP');
			XHR.open("GET", uri, false);
			XHR.send();
		}
	} catch(e) { 
		if(mozile.debug) mozile.debug.inform("mozile.load", "File load failed loading '"+ uri +"' with error message:\n"+ e);
		return null;
	}
	
	if(XHR) {	
		//alert(XHR +" "+ XHR.status);
		if(XHR.status == 0 || XHR.status == 200) return XHR.responseText;
		else {
			if(mozile.debug) mozile.debug.inform("mozile.load", "File load failed with status '"+ XHR.status +"' and message:\n"+ XHR.responseText);
			return null;
		}
	}
	
	if(mozile.debug) mozile.debug.inform("mozile.load", "No XMLHttpRequest available when trying to load '"+ uri +"'.");
	return null;
}

/**
 * Detects the path to this file by searching for a script with src "mozile.js".
 * The path is cleaned and returned.
 * @type String
 */
mozile.findRoot = function() {
	var scripts = document.getElementsByTagName("script");
	for(var i=0; i < scripts.length; i++) {
		if(scripts[i].getAttribute("src").indexOf("mozile.js") > -1) {
			var root = scripts[i].getAttribute("src");
			root = root.substring(0, root.lastIndexOf("mozile.js"));
			if(root.indexOf("src") > -1)
				root = root.substring(0, root.lastIndexOf("src"));
			return root;
		}
	}
	return "";
}

// Detect the mozile.root.
mozile.root = mozile.findRoot();



/**** Module System ****/

/**
 * This system a simpler version of the one used by the Dojo Toolkit (http://dojotoolkit.org).
 * "Modules" are just the parts of a hierarchy of objects building on the global "mozile" object. At each level they may contain properties, methods, or classes.
 * Names follow the familiar library naming convention: e.g. "mozile.dom.Document".
 * <p>Some modules are stored in their own JavaScript files in the "src" directory of the Mozile installation.
 * Module files must specify their requirements and the sub-modules they provide using mozile.require() and mozile.provide().
 * <p>The module can be retreived using mozile.findModule().
 * Modules are loaded with mozile.loadModule()
 */

/**
 * Finds an object in the "mozile" hierarchy, if it exists.
 * Given a name of the format "mozile.foo.bar", it starts with the global "window" object and checks for the presence of a "mozile" property.
 * Then it checks the "mozile" object for a "foo" property, and so on.
 * @param {String} moduleName A module name of the form "mozile.foo" or "mozile.foo.Bar".
 * @type Object
 */
mozile.findModule = function(moduleName) {
	var levels = moduleName.split(".");
	var current = mozile.window;
	for(var i=0; i < levels.length; i++) {
		if(!current[levels[i]]) return null;
		current = current[levels[i]];
	}
	return current;
}


/**
 * Tries to load a module of code.
 * The method first tries to load the file "$ROOT/src/foo/bar.js".
 * If that fails, it tries to load "$ROOT/src/foo.js".
 * <p>When a file is successfully loaded it is then evaluated using eval().
 * Note that it is evaluated in the context of this function, which will cause problems for any variables in the file which are declared outside the "mozile" hierarchy.
 * @param {String} moduleName A module name of the form "mozile.foo" or "mozile.foo.Bar".
 * @type Boolean
 */
mozile.loadModule = function(moduleName) {
	if(!moduleName) return false;

	var filename = mozile.root + mozile.filesep;
	var levels = moduleName.split(".");
	for(var i=0; i < levels.length; i++) {
		if(levels[i] == "*") levels = levels.splice(i,1);
	}
	if(levels[0] && levels[0] == "mozile") {
		levels.shift();
		if(levels[0] && levels[0] == "test") {
			levels.shift();
			filename += "tests" + mozile.filesep;
		}
		else filename += "src" + mozile.filesep;
	}
	else return false;
	if(levels.length == 0) return false; 
	//alert("Loading module "+moduleName);

	while(levels.length > 1) {
		filename += levels.shift() + mozile.filesep;
	}
	filename += levels.pop() +".js";
	//alert("Filename "+ filename);

	var file = mozile.load(filename);
	// Fix problem loading within JsUnit test files.
	if(!file && mozile.test && mozile.test.root) {
		filename = filename.replace(mozile.root, mozile.test.root);
		file = mozile.load(filename);
	}
	if(!file) {
		//alert("No file for '"+ moduleName +"' at '"+ filename +"' starting at '"+ location +"'.");
		var parent = moduleName.substring(0, moduleName.lastIndexOf("."));
		return mozile.loadModule(parent);
	}
	
	try {
		eval(file);
		return true;
	} catch(e) {
		mozile.debug.inform("mozile.loadModule", "Error evaluating module '"+ moduleName +"' in file '"+ filename +"'.\n"+ mozile.dumpError(e));
		return false;
	}
}

/**
 * Loads a module if it has not already been loaded.
 * This method is used by modules to indicate that they require the support of another module.
 * @param {String} moduleName A module name of the form "mozile.foo" or "mozile.foo.Bar".
 * @type Boolean
 */
mozile.require = function(moduleName) {
	if(!mozile.findModule(moduleName)) {
		if(mozile.precompiled) return false;
		return mozile.loadModule(moduleName);
	}
	else return false;
}

/**
 * This function is used by modules to indicate what properties, methods, and classes they provide.
 * Currently the function does nothing, but its use it still required for all modules.
 * @type Void
 */
mozile.provide = function() {
	// startPackage?
}







/**
 * A container for Mozile code.
 * This is a hack to make JSDoc show modules properly.
 * @constructor
 */
mozile.Module = function() {};





/**** File Tools ****/

/**
 * Determines whether a path is absolute.
 * @param {String} path The path to check.
 * @type Boolean
 */
mozile.isPathAbsolute = function(path) {
	if(!path) return false;
	path = path.toString();
	if(path.indexOf(":") == -1) return false;

	var protocol = path.substring(0, path.indexOf(":"));
	switch(protocol) {
		case "http":
		case "https":
		case "file":
			return true;
	}

	return false;
}

/**
 * Strips non-path information from a URL.  
 * @param {String} url The URL to strip.
 * @type String
 */
mozile.getPath = function(url) {
	if(!url) return "";
	url = url.toString();
	
	// Strip GET arguments.
	if(url.indexOf("?") > -1) url = url.substring(0, url.indexOf("?"));
	
	return url;
}

/**
 * Gets a directory from a path. Becareful, because this method isn't very smart.
 * TODO: Handle Windows paths.
 * @param {String} path The path to use.
 * @type String
 */
mozile.getDirectory = function(path) {
	if(!path) return "";
	path = path.toString();
	path = mozile.getPath(path);
	
	// Strip everything after the last file separator.
	if(path.indexOf(mozile.filesep) > -1) {
		if(path.indexOf(":///") > -1) {
			var partial = path.substring(path.indexOf(":///")+4);
			if(partial.indexOf(mozile.filesep) == -1) return path;
		}
		else if(path.indexOf("://") > -1) {
			var partial = path.substring(path.indexOf("://")+3);
			if(partial.indexOf(mozile.filesep) == -1) return path;
		}
		return path.substring(0, path.lastIndexOf(mozile.filesep)+1);
	}
	
	return path;
}

/**
 * Ensures that a path is absolute. Appends the path to the root if it is not absolute.
 * @param {String} path The path to convert.
 * @param {String} root An absolute root URL. If none is given the document location is used.
 * @type String
 */
mozile.getAbsolutePath = function(path, root) {
	if(!path) return "";
	path = path.toString();
	if(mozile.isPathAbsolute(path)) return path;
	
	if(!root) root = location;
	root = mozile.getDirectory(root);
	root = mozile.getAbsolutePath(root);
	return root + path;
}






/**** Debugging Tools ****/

/**
 * Tools for debugging Mozile.
 * @type Object
 */
mozile.debug = new Object();
// JSDoc hack
mozile.debug.prototype = new mozile.Module;

/**
 * Indicates the level of debugging information to be delivered to the user.
 * Values can be "suppress", "warn", "inform", or "debug", in order from least verbose to most verbose.
 * @type String
 */
mozile.debug.alertLevel = "suppress";

/**
 * Indicates the level of debugging information to be stored in the messages array.
 * Values can be "suppress", "warn", "inform", or "debug", in order from least verbose to most verbose.
 * @type String
 */
mozile.debug.logLevel = "inform";

/**
 * An array of debugging messages.
 * @type Array
 */
mozile.debug.messages = new Array();

/**
 * A window where logged messages are sent.
 * @type Window
 */
mozile.debug.window = null;

/**
 * Compares two debugging levels.
 * @param {String} type The level to check against. Can be "alert" or "log".
 * @param {String} level The level to check. Can be "suppress", "warn", "inform", or "debug".
 * @type Boolean
 */
mozile.debug.isSelected = function(type, level) {
	// Setup
	if(typeof(type) != "string") return false;
	type = type.toLowerCase();
	var checkLevel;
	if(type == "alert") checkLevel = mozile.debug.alertLevel;
	else if(type == "log") checkLevel = mozile.debug.logLevel;
	else return false;
	checkLevel = checkLevel.toLowerCase();
	if(typeof(level) != "string") return false;
	level = level.toLowerCase();

	// Compare levels
	if(checkLevel == "suppress") return false;
	if(checkLevel == "warn") {
		if(level =="warn") return true;
		else return false;
	}
	if(checkLevel == "inform") {
		if(level == "warn" || level == "inform") return true;
		else return false;
	}
	if(checkLevel == "debug") return true;

	return false;
}


/**
 * A high priority debugging notification.
 * @param {String} caller The name of the function sending the notification.
 * @param {String} message The warning.
 * @type Void
 */
mozile.debug.warn = function(caller, message) {
	var level = "warn";

	var msg = "Mozile Warning ["+ caller +"] "+ message;
	if(mozile.debug.isSelected("alert", level)) {
		if(window.warn) warn(msg); // JsUnit tracing
		else mozile.alert(msg);
	}
	if(mozile.debug.isSelected("log", level)) {
		mozile.debug.log(caller, level, message);
	}
}

/**
 * A medium priority debugging notification.
 * @param {String} caller The name of the function sending the notification.
 * @param {String} message The warning.
 * @type Void
 */
mozile.debug.inform = function(caller, message) {
	var level = "inform";

	var msg = "Mozile Information ["+ caller +"] "+ message;
	if(mozile.debug.isSelected("alert", level)) {
		if(window.inform) inform(msg); // JsUnit tracing
		else mozile.alert(msg);
	}
	if(mozile.debug.isSelected("log", level)) {
		mozile.debug.log(caller, level, message);
	}
}

/**
 * A low priority debugging notification.
 * @param {String} caller The name of the function sending the notification.
 * @param {String} message The warning.
 * @type Void
 */
mozile.debug.debug = function(caller, message) {
	var level = "debug";

	var msg = "Mozile Debugging ["+ caller +"] "+ message;
	if(mozile.debug.isSelected("alert", level)) {
		if(window.debug) debug(msg); // JsUnit tracing
		else mozile.alert(msg);
	}
	if(mozile.debug.isSelected("log", level)) {
		mozile.debug.log(caller, level, message);
	}
}

/**
 * Store a debugging message in the messages array.
 * @param {String} caller The name of the function sending the warning.
 * @param {String} level The level of the message.
 * @param {String} message The warning.
 * @type Void
 */
mozile.debug.log = function(caller, level, message) {
	var date = new Date();
	var msg =  { 
		caller: caller, 
		level: level, 
		message: message, 
		date: date.toLocaleString(),
		toString: function() {
			return this.level.toUpperCase() +" ("+ this.date +") ["+ this.caller +"] "+ this.message;
		}
	};
	mozile.debug.messages.push(msg);
	
	// If the window is open, print the message to it and scroll to the bottom.
	if(mozile.debug.window && mozile.debug.window.document) {
		mozile.debug.window.document.write(msg +"<br/>\n");
		mozile.debug.window.scroll(0, document.body.clientHeight);
	}
}

/**
 * Write all logged debugging messages to a new window.
 * @type Void
 */
mozile.debug.show = function() {
	if(!mozile.debug.window || !mozile.debug.window.document) {
		mozile.debug.window = window.open("", "MozileDebugging", "");
		mozile.debug.window.document.write("<h3>Mozile Debugging Messages</h3>");
		mozile.debug.window.document.write(mozile.debug.messages.join("<br/>\n") + "<br/>\n");
	}
	else mozile.debug.window = window.open("", "MozileDebugging", "");
}


/**
 * Informs the user about a problem. 
 * The method used to inform the user varies by what
 * @param {String} message
 * @type Void
 */
mozile.alert = function(message) {
	// TODO: Firebug support
	// TODO: Safari console support
	alert(message);
}


/**
 * Dumps information from error objects.
 * Internet Explorer needs special attention.
 * @param object The error object or object that the error is attached to.
 * @type String
 */
mozile.dumpError = function(object) {
	if(typeof(object) == "string") return object;
	if(!mozile.browser.isIE) return object.toString();

	// Error Object case
	if(object && object.description) {	
		var fields = [
			"Name: "+ object.name,
			"Number: "+ object.number,
			"Message: "+ object.message,
			"Description: "+ object.description,
		];
		return fields.join("\n");
	}

	// parseError case
	if(!object) object = document;
	if(!object.parseError) object = document;
	if(!object.parseError) object = window;
	if(!object.parseError) return "[No error to parse]";
	var fields = [
		"Error Code: "+ object.parseError.errorCode,
		"File Position: "+object.parseError.filepos,
		"Line: "+object.parseError.line,
		"Line Position: "+object.parseError.linepos,
		"Reason: "+object.parseError.reason,
		"Source Text: "+object.parseError.srcText,
		"Url: "+object.parseError.url
	];
	
	return fields.join("\n");
}

/**
 * Makes a single element editable.
 * @param elementOrId Either a DOM element or the id of an element.
 * @type Void
 */
mozile.editElement = function(elementOrId) {
	var element;
	if(typeof(elementOrId) == "string") {
		if(document.documentElement.nodeName.toLowerCase() == "html") // HTML case
			element = document.getElementById(elementOrId);
		else return mozile.editElements(elementOrId, "id", true); // XML case
	}
	else if(elementOrId.nodeType && 
		elementOrId.nodeType == mozile.dom.ELEMENT_NODE) {
		element = elementOrId;
	}

	if(element) {
		mozile.require("mozile.edit");
		mozile.edit.setMark(element, "editable", true);
		// Store the value of contentEditable.
		switch(element.getAttribute("contentEditable")) {
			case "true":
				mozile.edit.setMark(element, "contentEditable", true);
				break;
			case "false":
				mozile.edit.setMark(element, "contentEditable", false);
				break;
		}
		// Set contentEditable to true.
		if(mozile.browser.isIE) {
			element.setAttribute("contentEditable", "true");
		}
	}
	return undefined;
}

/**
 * Makes a set of elements editable.
 * @param listOrValue Either an array (or nodeList) of DOM elements or the value of an attribute (defaults to "class").
 * @param {String} attribute Optional. The name of an attribute to search for. Defaults to "class". If the value is null then the local name of the element is used (lowercase).
 * @param {Boolean} single Optional. When true, only one item is added.
 * @type Void
 */
mozile.editElements = function(listOrValue, name, single) {
	mozile.require("mozile.dom");
	var list = new Array();
	if(name === undefined) name = "class";
	if(typeof(listOrValue) == "string") {
		if(document.createTreeWalker) {
			var treeWalker = document.createTreeWalker(document.documentElement, mozile.dom.NodeFilter.SHOW_ELEMENT, null, false);
			while(treeWalker.nextNode()) {
				var node = treeWalker.currentNode;
				if(name == "class" && mozile.browser.isIE && node.className == listOrValue) list.push(node);
				else if(name && node.getAttribute(name) == listOrValue) list.push(node);
				else if(!name && mozile.dom &&
					mozile.dom.getLocalName(node).toLowerCase() == listOrValue) {
					list.push(node);
				}
				if(single && list.length > 0) break;
			}
		}
	}
	else if(listOrValue.length) {
		list = listOrValue;
	}
	
	if(list.length) {
		for(var i=0; i < list.length; i++) {
			if(list[i] && list[i].nodeType && 
				list[i].nodeType == mozile.dom.ELEMENT_NODE) {
				mozile.editElement(list[i]);
				if(single) break;
			}
		}
	}
	
	return undefined;
}

/**
 * Makes the whole document editable.
 * @param {Document} doc Optional. The document to make editable. Defaults to the current document.
 * @type Void
 */
mozile.editDocument = function(doc) {
	if(!doc) doc = document;
	mozile.editElement(doc.documentElement);
}

/**
 * Makes all text nodes inside editable elements editable by adding default commands to mozile.edit.
 * @param {Boolean} rich Optional. When true rich editing commands are included.
 * @type Void
 */
mozile.editAllText = function(rich) {
	mozile.require("mozile.edit");
	mozile.edit.addDefaultCommand(mozile.edit.navigateLeftRight);
	mozile.edit.addDefaultCommand(mozile.edit.insertText);
	mozile.edit.addDefaultCommand(mozile.edit.removeText);
	
	if(rich) {
		mozile.require("mozile.edit.rich");
		mozile.edit.addDefaultCommand(mozile.edit.remove);
		var splitBlocks = new mozile.edit.Split("splitBlocks");
		splitBlocks.accel = "Return Enter";
		mozile.edit.addDefaultCommand(splitBlocks);
	}
}

/**
 * Load the target RNG schema and use it as the schema for this document.
 * @param target The desired schema. Can be any value accepted by mozile.rng.Schema.parse().
 * @type Boolean
 * @return True if the schema is found and parsed correctly.
 */
mozile.useSchema = function(target) {
	try {
		mozile.require("mozile.rng");
		mozile.require("mozile.edit");
		mozile.require("mozile.edit.rich");
		mozile.edit.extendRNG();
		mozile.schema = new mozile.rng.Schema();
		var validation = mozile.schema.parse(target);
		if(validation.isValid) {
			mozile.edit.generateCommands(mozile.schema);
			return true;
		}
		else {
			mozile.debug.inform("mozile.useSchema", "Schema validation failed.\n"+ validation.report(true));
			return false;
		}
	} catch(e) {
		mozile.debug.inform("mozile.useSchema", "Could not create schema for target '"+ target +"' because of an error:\n"+ mozile.dumpError(e));
		return false;
	}
}



/**** Operating System Detection ****/

/**
 * Tools for dealing with the operating system.
 * @type Object
 */
mozile.os = new Object();
// JSDoc hack
mozile.os.prototype = new mozile.Module;

/**
 * Check to see if the browser is running on a Mac.
 * @type Boolean
 */
mozile.os.isMac = false;
if(navigator.userAgent.match(/Macintosh/)) mozile.os.isMac = true;



/**** Browser Detection ****/

/**
 * Tools for dealing with the browser.
 * @type Object
 */
mozile.browser = new Object();
// JSDoc hack
mozile.browser.prototype = new mozile.Module;


/**
 * Check to see if this browser is Mozilla.
 * @type Boolean
 */
mozile.browser.isMozilla = false;

/**
 * If this is a Mozilla browser, this stores a number representing the major and minor versions. E.g. "1.7" or "1.8".
 * @type Number
 */
mozile.browser.mozillaVersion = 0;

mozile.browser.mozillaVersion = navigator.userAgent.match(/rv\:(\d+\.\d+)/);
if(mozile.browser.mozillaVersion && 
	Number(mozile.browser.mozillaVersion[1])) {
	mozile.browser.isMozilla = true;
	mozile.browser.mozillaVersion = Number(mozile.browser.mozillaVersion[1]);
}

/**
 * Check to see if this browser is Internet Explorer.
 * @type Boolean
 */
mozile.browser.isIE = false;
if(navigator.userAgent.match(/MSIE/)) mozile.browser.isIE = true;

/**
 * Check to see if this browser is Safari.
 * @type Boolean
 */
mozile.browser.isSafari = false;
if(navigator.userAgent.match(/Safari/)) mozile.browser.isSafari = true;

/**
 * If this is a Safari browser, this property stores a number representing the WebKit version. E.g. "418" or "420".
 * @type Number
 */
mozile.browser.safariVersion = 0;

mozile.browser.safariVersion = navigator.userAgent.match(/AppleWebKit\/(\d+)/);
if(mozile.browser.safariVersion && 
	Number(mozile.browser.safariVersion[1])) {
	mozile.browser.safariVersion = Number(mozile.browser.safariVersion[1]);
}

/**
 * Check to see if this browser is Opera.
 * @type Boolean
 */
mozile.browser.isOpera = false;
if(navigator.userAgent.match(/Opera/)) mozile.browser.isOpera = true;




/**** Final Configuration ****/

// Detect the content type as HTML or XML in Mozilla.
if(mozile.browser.isMozilla) {
	if(document.contentType && document.contentType == "text/html") {
		mozile.defaultNS = null;
	}
	else {
		mozile.defaultNS = "http://www.w3.org/1999/xhtml";
	}
}



/**
 * A collection of DOM classes and methods.
 * @type Object
 */
mozile.dom = new Object();
// JSDoc hack
mozile.dom.prototype = new mozile.Module;

/**
 * Define some shared constants.
 */
mozile.dom.ELEMENT_NODE                = 1;
mozile.dom.ATTRIBUTE_NODE              = 2;
mozile.dom.TEXT_NODE                   = 3;
mozile.dom.CDATA_SECTION_NODE          = 4;
mozile.dom.ENTITY_REFERENCE_NODE       = 5;
mozile.dom.ENTITY_NODE                 = 6;
mozile.dom.PROCESSING_INSTRUCTION_NODE = 7;
mozile.dom.COMMENT_NODE                = 8;
mozile.dom.DOCUMENT_NODE               = 9;
mozile.dom.DOCUMENT_TYPE_NODE          = 10;
mozile.dom.DOCUMENT_FRAGMENT_NODE      = 11;
mozile.dom.NOTATION_NODE               = 12;


/**
 * An array of "link" elements which have been added to this document.
 * @type Array
 */
mozile.dom.links = new Array();


/**
 * Gets the document's body element, if it exists. If none exists the document element is returned.
 * @param {Node} node Optional. A node belonging to the document to be searched. If none is given, the current document is used.
 * @type Element
 */
mozile.dom.getBody = function(node) {
	var doc = document;
	if(node) doc = node.ownerDocument;
	var elements = doc.getElementsByTagName("body");
	if(elements && elements[0]) return elements[0];
	else return doc.documentElement;
}

/**
 * Gets the document's head element, if it exists. If none exists the document element is returned.
 * @param {Node} node Optional. A node belonging to the document to be searched. If none is given, the current document is used.
 * @type Element
 */
mozile.dom.getHead = function(node) {
	var doc = document;
	if(node) doc = node.ownerDocument;
	var elements = doc.getElementsByTagName("head");
	if(elements && elements[0]) return elements[0];
	else return doc.documentElement;
}

/**
 * Gets the given node's first child element, if one exists.
 * @param {Node} parent The parent node to check.
 * @type Element
 */
mozile.dom.getFirstChildElement = function(parent) {
	for(var i=0; i < parent.childNodes.length; i++) {
		if(parent.childNodes[i].nodeType == mozile.dom.ELEMENT_NODE) 
			return parent.childNodes[i];
	}
	return null;
}

/**
 * Returns an array of all given node's child elements.
 * @param {Node} parent The parent node to check.
 * @type Array
 */
mozile.dom.getChildElements = function(parent) {
	var children = new Array();
	for(var i=0; i < parent.childNodes.length; i++) {
		if(parent.childNodes[i].nodeType == mozile.dom.ELEMENT_NODE) 
			children.push(parent.childNodes[i]);
	}
	return children;
}

/**
 * Gets the given node's next sibling element, if one exists.
 * @param {Node} node The node to start with.
 * @type Element
 */
mozile.dom.getNextSiblingElement = function(node) {
	var sibling = node.nextSibling;
	while(sibling) {
		if(sibling.nodeType == mozile.dom.ELEMENT_NODE) return sibling;
		sibling = sibling.nextSibling;
	}
	return null;
}

/**
 * Inserts the newNode after the refNode in the refNode's parent node.
 * @param {Node} newNode The node to insert.
 * @param {Node} refNode Insert the new node after this node.
 * @type Node
 */
mozile.dom.insertAfter = function(newNode, refNode) {
	if(!newNode) throw("Error [mozile.dom.insertAfter]: No newNode. newNode: "+ newNode +" refNode: "+ refNode);
	if(!refNode) throw("Error [mozile.dom.insertAfter]: No refNode. newNode: "+ newNode +" refNode: "+ refNode);
	var parentNode = refNode.parentNode;
	if(!parentNode) return null;
	if(refNode.nextSibling) return parentNode.insertBefore(newNode, refNode.nextSibling);
	else return parentNode.appendChild(newNode);
}

/**
 * Inserts a new node as the first child of the given parent node.
 * @param {Node} newNode The node to insert.
 * @param {Node} parentNode The parent to attach this node to.
 * @type Node
 */
mozile.dom.prependChild = function(newNode, parentNode) {
	if(parentNode.firstChild) return parentNode.insertBefore(newNode, parentNode.firstChild);
	else return parentNode.appendChild(newNode);
}

/**
 * Determines whether the ancestorNode is an ancestor of the descendantNode. If the ancestorNode is the descendantNode, the method returns true.
 * @param {Node} ancestorNode 
 * @param {Node} descendantNode 
 * @param {Node} limitNode Optional. The search will stop at this node, no matter what happens. 
 * @type Boolean
 */
mozile.dom.isAncestorOf = function(ancestorNode, descendantNode, limitNode) {
	var checkNode = descendantNode;
	while(checkNode) {
		if(checkNode == ancestorNode) return true;
		else if(checkNode == limitNode) return false;
		else checkNode = checkNode.parentNode;
	}
	return false;
}

/**
 * Returns the first node which is an ancestor of both given nodes.
 * @param {Node} firstNode 
 * @param {Node} secondNode
 * @type Node
 */
mozile.dom.getCommonAncestor = function(firstNode, secondNode) {
	var ancestor = firstNode;
	while(ancestor) {
		if(mozile.dom.isAncestorOf(ancestor, secondNode)) return ancestor;
		else ancestor = ancestor.parentNode;
	}
	return null;
}

/**
 * A regular expression to check for non-whitespace characters.
 * @private
 * @type RegExp
 */
mozile.dom._matchNonWhitespace = /\S/;

/**
 * Decides whether the given node contains non-whitespace text.
 * @param {Node} node The node to be checked.
 * @type Boolean
 */
mozile.dom.isWhitespace = function(node) {
	if(!node || !node.nodeValue) return false;
	if(node.nodeValue.match(mozile.dom._matchNonWhitespace)) return false;
	return true;
}


/**
 * A test to see if the given node belongs to an X/HTML document. Fairly crude test at the moment.
 * @param {Node} node A node belonging to the document to be checked.
 * @type Boolean
 */
mozile.dom.isHTML = function(node) {
	if(!node) node = document;
	var doc = node;
	if(node.ownerDocument) doc = node.ownerDocument;
	if(!doc.documentElement) return false;
	var name = doc.documentElement.nodeName;
	if(doc.documentElement.nodeName) name = doc.documentElement.nodeName;
	if(name.toLowerCase() == "html") return true;
	else return false;
}


/**
 * A test to see if the given node should be ignored.
 * @param {Node} node The node to check.
 * @type Boolean
 */
mozile.dom.isIgnored = function(node) {
	if(node.nodeType == mozile.dom.ATTRIBUTE_NODE){
		if(node.nodeName.indexOf("xmlns") == 0) return true;
		if(mozile.browser.isOpera && node.nodeName.toLowerCase() == "shape") return true;
	}
	return false;
}

/**
 * Gets the local part of the qualified name for this node.
 * If the name is not qualified, the nodeName is returned.
 * @param {Node} node The node to check.
 * @type String
 */
mozile.dom.getLocalName = function(node) {
	if(!node) return null;
	if(node.localName) return node.localName;
	else if(node.nodeName && node.nodeName.indexOf(":") > -1) 
		return node.nodeName.substring(node.nodeName.indexOf(":") + 1);
	else return node.nodeName;
}

/**
 * Gets the prefix part of the qualified name for this node.
 * If the name is not qualified, a null value is returned.
 * @param {Node} node The node to check.
 * @type String
 */
mozile.dom.getPrefix = function(node) {
	if(!node) return null;
	if(node.prefix) return node.prefix;
	else if(node.nodeName.indexOf(":") > -1) 
		return node.nodeName.substring(0, node.nodeName.indexOf(":"));
	else return null;
}

/**
 * Gets the index of this node among all its parent's child nodes.
 * @param {Node} node The node to check.
 * @type Integer
 */
mozile.dom.getIndex = function(node) {
	if(!node.parentNode) return null;
	for(var c=0; c < node.parentNode.childNodes.length; c++) {
		if(node.parentNode.childNodes[c] == node) return c;
	}
	return c;
}

/**
 * Gets the index of this node among its parent's children of the same type and name. Used for XPath positions.
 * @param {Node} node The node to check.
 * @type Integer
 */
mozile.dom.getPosition = function(node) {
	if(!node.parentNode) return null;
	var s=1;
	for(var c=0; c < node.parentNode.childNodes.length; c++) {
		if(node.parentNode.childNodes[c] == node) break;
		else if(node.nodeType == mozile.dom.ELEMENT_NODE) {
			if(node.parentNode.childNodes[c].nodeName == node.nodeName) s++;
		}
		else if(node.parentNode.childNodes[c].nodeType == node.nodeType) s++;
	}
	return s;
}

/**
 * Removes all the child nodes of the given node.
 * Does not remove attributes.
 * @param {Node} node The node to remove the child nodes from.
 * @type Void
 */
mozile.dom.removeChildNodes = function(node) {
	if(node.childNodes.length == 0) return;
	while(node.firstChild) {
		node.removeChild(node.firstChild);
	}
}

/**
 * Get the namespace of a node.
 * @param {Node} node The node to check.
 * @type String
 */
mozile.dom.getNamespaceURI = function(node) {
	if(node.namespaceURI) return node.namespaceURI;
	else if(node.nodeName.indexOf(":") > -1) {
		var prefix = node.nodeName.substring(0, node.nodeName.indexOf(":"));
		return mozile.dom.lookupNamespaceURI(node, prefix);
	}
	return mozile.dom.getDefaultNamespaceURI(node);
}

/**
 * Climb the tree looking for the first "xmlns" attribute.
 * @param {Node} node The node to check.
 * @type String
 */
mozile.dom.getDefaultNamespaceURI = function(node) {
	var namespaceURI = null;
	while(node) {
	  if(node.nodeType == mozile.dom.ELEMENT_NODE && node.getAttribute("xmlns"))
	  	return node.getAttribute("xmlns");
	  node = node.parentNode;
	}
	return namespaceURI;
}

/**
 * Tries to find and return the URI of the namespace of this node.
 * In browsers that support lookupNamespaceURI(prefix), when a prefix is given that method is used.
 * Otherwise, if the node has a qualified name with a prefix, the method looks for an "xmlns:prefix" attribute. 
 * If the node has no prefix, the method looks for an "xmlns" attribute.
 * Searches climb the DOM tree, starting with the node, and return the first match found.
 * @param {Node} node The node to check.
 * @param {String} prefix Optional. The prefix to check for. If none is given, the prefix of the node is used (if there is one).
 * @type String
 */
mozile.dom.lookupNamespaceURI = function(node, prefix) {
	if(!prefix) prefix = mozile.dom.getPrefix(node);
	var attr = "xmlns";

	// Proper method when a prefix is given.
	// Mozilla automatically climbs the DOM tree, but Opera does not.
	if(prefix && node.lookupNamespaceURI) {
		while(node) {
			var ns = node.lookupNamespaceURI(prefix);
			if(ns) return ns;
			else node = node.parentNode;
		}
		return null;
	}

	// Safari has to use getAttributeNS, while other browsers do not.
	if(prefix && mozile.browser.isSafari) {
		while(node && node.getAttributeNS) {
			//alert("Prefix: "+ node.nodeName +" "+ attr +":"+ prefix +" "+  node.getAttributeNS(attr, prefix));
			if(node.getAttributeNS(attr, prefix)) return node.getAttributeNS(attr, prefix);
			else node = node.parentNode;
		}
		return null;
	}

	// IE supported method for prefixes.
	if(prefix) attr = "xmlns:"+prefix;
	
	// General case.
	while(node) {
		//alert("No prefix: "+ node.nodeName +" "+ attr +" "+ node.getAttribute(attr));
		if(node.getAttribute(attr)) return node.getAttribute(attr);
		else node = node.parentNode;
	}
	return null;
}

/**
 * Tries to find and return the prefix for the qualified names of node with the given namespace URI.
 * Climbs the DOM tree, checking for any attributes starting with "xmlns:". It then checks those against the given namesace URI. If they match, the prefix part of the attribute name is returned.
 * @param {Node} node The node to check.
 * @param {String} namespaceURI Optional. The namespace URI to find the prefix for. If none is given, this node's namespaceURI is used.
 * @type String
 */
mozile.dom.lookupPrefix = function(node, namespaceURI) {
	if(!namespaceURI) namespaceURI = node.namespaceURI;
	if(!namespaceURI) return null;

	while(node && node.attributes) {
		for(var i=0; i < node.attributes.length; i++) {
			if(node.attributes[i].nodeName.indexOf("xmlns:") == 0 &&
				node.attributes[i].nodeValue == namespaceURI) {
				return node.attributes[i].nodeName.substring(6);
			}
		}
		node = node.parentNode;
	}
	return null;
}


/**
 * Converts a CSS style name (hyphenated) to a JavaScript style name (camelCase).
 * @param {String} styleName The name of a CSS rule.
 * @type String
 */
mozile.dom.convertStyleName = function(styleName) {
	if(!styleName || typeof(styleName) != "string") return null;
	return styleName.replace(/\-(\w)/g, function (strMatch, p1){
			return p1.toUpperCase();
	});
}

/**
 * Get the string value of a CSS style declaration.
 * Adapted from http://www.robertnyman.com/2006/04/24/get-the-rendered-style-of-an-element/
 * @param {Node} node The node which has the style.
 * @param {String} cssRule The name of a CSS rule.
 * @type String
 */
mozile.dom.getStyle = function(node, cssRule) {
	var value = "";
	if(!node) return value;
	if(node.nodeType != mozile.dom.ELEMENT_NODE) node = node.parentNode;
	if(!node || node.nodeType != mozile.dom.ELEMENT_NODE) return value;

	if(document.defaultView && document.defaultView.getComputedStyle){
		value = document.defaultView.getComputedStyle(node, "").getPropertyValue(cssRule);
	}
	else if(node.currentStyle){
		cssRule = mozile.dom.convertStyleName(cssRule);
		value = node.currentStyle[cssRule];
	}

	return value;
}

/**
 * Sets the CSS style of an element.
 * @param {Element} element The element which has the style.
 * @param style Either a string denoting the CSS style or an object with keys
 * @type String
 */
mozile.dom.setStyle = function(element, rule, value) {
	if(!element) return;
	if(!rule || typeof(rule) != "string") return;
	
	rule = mozile.dom.convertStyleName(rule);
	if(element.style) element.style[rule] = value;
	else mozile.debug.debug("mozile.dom.setStyle", "Element does not have a 'style' attribute.");
}

/**
 * Adds a "link" element with the given href and type to the head of the document.
 * @param {String} href The target of the link.
 * @param {String} type Optional. The type of the link. Defaults to "text/css".
 * @type Void
 */
mozile.dom.addStyleSheet = function(href, type) {
	var link;
	if(mozile.defaultNS != null) {
		mozile.require("mozile.xml");
		link = mozile.dom.createElementNS(mozile.xml.ns.xhtml, "link");
	}
	else link = mozile.dom.createElement("link");
	link.setAttribute("class", "mozileLink");
	link.setAttribute("href", href);
	link.setAttribute("rel", "stylesheet");
	if(!type) type = "text/css";
	link.setAttribute("type", type);
	mozile.dom.getHead().appendChild(link);
	mozile.dom.links.push(link);
}


/**
 * Gets the position of the element using offsetLeft relative to the container (if provided) or the document element.
 * Adapted from http://www.quirksmode.org/js/findpos.html
 * @param {Node} obj The target element.
 * @param {Container} container Optional. The element relative to which the X position will be measured.
 * @type String
 */
mozile.dom.getX = function(obj, container) {
	var x = 0;
	if(obj.offsetParent) {
		while(obj.offsetParent) {
			x += obj.offsetLeft;
			obj = obj.offsetParent;
			if(obj == container) break;
		}
	}
	else if (obj.x) x += obj.x;
	return x;
}

/**
 * Gets the position of the element using offsetTop relative to the container (if provided) or the document element.
 * Adapted from http://www.quirksmode.org/js/findpos.html
 * @param {Node} obj The target element.
 * @param {Container} container Optional. The element relative to which the X position will be measured.
 * @type String
 */
mozile.dom.getY = function(obj, container) {
	var y = 0;
	if(obj.offsetParent) {
		while(obj.offsetParent) {
			y += obj.offsetTop;
			obj = obj.offsetParent;
			if(obj == container) break;
		}
	}
	else if (obj.y) y += obj.y;
	return y;
}


/**
 * Selection operations.
 * See this for some ideas: http://www.teria.com/~koseki/memo/xbselection/Selection.js
 * @type Object
 */
mozile.dom.selection = new Object();
// JSDoc hack
mozile.dom.selection.prototype = new mozile.Module;

/**
 * Get the global selection object. Initiate it if necessary.
 * @type Selection
 */
mozile.dom.selection.get = function() {
	//return new mozile.dom.Selection();
	if(!mozile.dom._selection) mozile.dom._selection = new mozile.dom.Selection();
	else if(mozile.browser.isIE) mozile.dom._selection._init();
	return mozile.dom._selection;
}

/**
 * Get a namespaced attribute's value.
 * @param {Element} element
 * @param {String} namespaceURI The URI of the namespace to check.
 * @param {String} name The local name of the attribute.
 * @type String
 */
mozile.dom.getAttributeNS = function(element, namespaceURI, name) {
	if(!element) return null;
	if(element.getAttributeNS) return element.getAttributeNS(namespaceURI, name);
	else {
		prefix = mozile.dom.lookupPrefix(element, namespaceURI);
		if(prefix) return element.getAttribute(prefix+":"+name);
	}
	return null;
}


/**
 * Creates an element with a qualified name using mozile.defaultNS if this variable is defined, otherwise an unqualified element.
* @param {String} name The name of the element.
* @type Element
*/
mozile.dom.createElement = function(name) {
	if(mozile.defaultNS) {
		return mozile.dom.createElementNS(mozile.defaultNS, name);
	} else {
		return document.createElement(name);
	}
}

/**
 * Tries to find and return the prefix for the qualified names of node with the given namespace URI.
 * Climbs the DOM tree, checking for any attributes starting with "xmlns:". It then checks those against the given namesace URI. If they match, the prefix part of the attribute name is returned.
 * @param {Node} node The node to check.
 * @param {String} namespaceURI The namespace URI to find the prefix for.
 * @type String
 */
mozile.dom.createElementNS = function(namespaceURI, name) {
	if(document.createElementNS && !mozile.browser.isSafari) return document.createElementNS(namespaceURI, name);
	else {
		// This is the only hack I could figure out that would work in IE6.
		mozile.require("mozile.xml");
		return mozile.xml.parseElement('<'+ name +' xmlns="'+ namespaceURI +'"/>');
	}
	return null;
}

/**
 *
 */
mozile.dom.Range = function(selectRange) {
  if (document.createRange) {
  	var range;
  	if(selectRange) range = selectRange.cloneRange();
  	else range = document.createRange();
  	for(var field in mozile.dom.Range.prototype) 
  		range[field] = mozile.dom.Range.prototype[field];
  	return range;
  } else {
  	if(selectRange && selectRange._range) {
  		var range = new mozile.dom.InternetExplorerRange(selectRange._range.duplicate());
  		range._init();
  		return range;
  	}
  	else return new mozile.dom.InternetExplorerRange();
  }
}


/**
 * Store the details about this range in an object which can be used to restore the range.
 * @type Object
 */
mozile.dom.Range.prototype.store = function () {
	var state = new Object();
	// TODO: Speed optmization for IE. Not stable.
	if(false && this._range) {
		state.format = "IE";
		state.bookmark = this._range.getBookmark();
	}
	else {
		mozile.require("mozile.xpath");
		state.type = "Range";
		state.format = "W3C";
		state.startContainer = mozile.xpath.getXPath(this.startContainer);
		state.startOffset = this.startOffset;
		if(this.startContainer == this.endContainer)
			state.endContainer = state.startContainer;
		else state.endContainer = mozile.xpath.getXPath(this.endContainer);
		state.endContainer = mozile.xpath.getXPath(this.endContainer);
		state.endOffset = this.endOffset;
	}
	return state;
}

/**
 * Takes a stored range object and creates a new range with the same properties.
 * @param {Object} state A state object from 
 * @type Void
 */
mozile.dom.Range.prototype.restore = function(state) {
try {
	// TODO: Speed optmization for IE. Actually slower?
	if(false && this._range) {
		this._range.moveToBookmark(state.bookmark);
		this._init();
	}
	else {
		mozile.require("mozile.xpath");
		//alert(mozile.util.dumpValues(state));
		var startContainer, endContainer;
		startContainer = mozile.xpath.getNode(state.startContainer);
		//alert(["startContainer", state.startContainer, startContainer.nodeName].join("\n"));
		this.setStart(startContainer, state.startOffset);
		if(state.endContainer == state.startContainer) 
			endContainer = startContainer;
		else endContainer = mozile.xpath.getNode(state.endContainer);
		//endContainer = mozile.xpath.getNode(state.endContainer);
		//alert(["endContainer", state.endContainer, endContainer.nodeName].join("\n"));
		this.setEnd(endContainer, state.endOffset);
	}
} catch(e) { 
	alert("Error [mozile.dom.Range.restore]:\n"+ mozile.dumpError(e))
	// +"\nUsing state oject:\n"+ mozile.util.dumpValues(state)); 
}
}

/**
 *
 */
mozile.dom.Selection = function() {
  if (window.getSelection) {
  	var selection = window.getSelection();
  	for(var field in mozile.dom.Selection.prototype) 
  		selection[field] = mozile.dom.Selection.prototype[field];
  	return selection;
  } else {
    return new mozile.dom.InternetExplorerSelection();
  }
}

/**
 * Store the details about this range in an object which can be used to restore the range.
 * @type Object
 */
mozile.dom.Selection.prototype.store = function(oldState, newOffset) {
	var state = new Object();
	if(oldState) {
		for(var i in oldState) state[i] = oldState[i];
		//state.startOffset = newOffset;
		//state.endOffset = newOffset;
		state.anchorOffset = newOffset;
		state.focusOffset = newOffset;
		return state;
	}
	else if(this.rangeCount > 0) {
		//var range = this.getRangeAt(0);
		//if(range) {
		//	range.store = mozile.dom.Range.prototype.store;
		//	return range.store();
		//}
		//else return null;
		// New idea.
		mozile.require("mozile.xpath");
		state.type = "Selection";
		state.format = "W3C";
		state.anchorNode = mozile.xpath.getXPath(this.anchorNode);
		state.anchorOffset = this.anchorOffset;
		if(this.focusNode == this.anchorNode) state.focusNode = state.anchorNode;
		else state.focusNode = mozile.xpath.getXPath(this.focusNode);
		state.focusOffset = this.focusOffset;
		state.isCollapsed = this.isCollapsed;
		return state;
	}
	return null;
}

/**
 * Takes a stored range object and creates a new range with the same properties.
 * @param {Object} state Optional. A state object from a store() call. If none is given, the data from mozile.dom.selection.last is used.
 * @type Void
 */
mozile.dom.Selection.prototype.restore = function(state) {
	if(state) {
		if(state.type == "Selection") {
			mozile.require("mozile.xpath");
			var anchorNode, focusNode;
			anchorNode = mozile.xpath.getNode(state.anchorNode);
			this.collapse(anchorNode, state.anchorOffset);
			//if(!state.isCollapsed) {
			if(state.focusNode != state.anchorNode ||
				state.focusOffset != state.anchorOffset ) {
				if(state.focusNode == state.anchorNode) focusNode = anchorNode;
				else focusNode = mozile.xpath.getNode(state.focusNode);
				try {
					this.extend(focusNode, state.focusOffset);
				} catch(e) {
					mozile.debug.debug("mozile.dom.Selection.restore", "Error extending selection to '"+ state.focusNode +" "+ state.focusOffset +"'.\n"+ mozile.dumpError(e) );
				}
			}
		}
		else if(state.type == "Range") {
			var range = new mozile.dom.Range();
			range.restore(state);
			this.removeAllRanges();
			this.addRange(range);
		}
	}
	else if(mozile.dom.selection.last) {		
		this.collapse(mozile.dom.selection.last.anchorNode, 
			mozile.dom.selection.last.anchorOffset);
		if(!mozile.dom.selection.last.isCollapsed) {
			this.extend(mozile.dom.selection.last.focusNode, 
				mozile.dom.selection.last.focusOffset);
		}
	}
}



	
/**
 * Old function, is now in the Selection constructor
 */
mozile.dom.Selection.getSelection = function () {
	if(window.getSelection) { //FF, Safari
		return window.getSelection();
	} else if (document.getSelection) { // Opera?
		return document.getSelection();
	} else if(document.selection) { // IE win
		return new mozile.dom.Selection();
	}
	return null;
}




if(mozile.browser.isIE) {
	mozile.require("mozile.dom.TreeWalker");
	mozile.require("mozile.dom.InternetExplorerRange");
	mozile.require("mozile.dom.InternetExplorerSelection");
}
else {
	mozile.dom.NodeFilter = NodeFilter;
	mozile.dom.TreeWalker = TreeWalker;
}





/**
 * A collection of XML functions.
 * @type Object
 */
mozile.xml = new Object();
// JSDoc hack
mozile.xml.prototype = new mozile.Module;


/**
 * A comprehensive list of XML namespaces.
 * Copied from Dojo 2.2.
 */
mozile.xml.ns = {
	AdobeExtensions : "http://ns.adobe.com/AdobeSVGViewerExtensions/3.0/",
	cml : "http://www.xml-cml.org",
	dc : "http://purl.org/dc/elements/1.1/",
	dcq : "http://purl.org/dc/qualifiers/1.0",
	dt : "http://www.w3.org/2001/XMLSchema-datatypes",
	fo : "http://www.w3.org/1999/XSL/Format",
	mes : "http://mozile.mozdev.org/ns/mes/1.0", // Mozile's editing scheme
	mml : "http://www.w3.org/1998/Math/MathML",
	rdf : "http://www.w3.org/1999/02/22-rdf-syntax-ns#",
	rdfs : "http://www.w3.org/2000/01/rdf-schema#",
	rng : "http://relaxng.org/ns/structure/1.0",
	saxon : "http://icl.com/saxon",
	"soap-env" : "http://schemas.xmlsoap.org/soap/envelope/",
	smil : "http://www.w3.org/2001/SMIL20/",
	svg : "http://www.w3.org/2000/svg",
	wsdl : "http://schemas.xmlsoap.org/wsdl/",
	xalan : "http://xml.apache.org/xslt",
	xbl : "http://www.mozilla.org/xbl",
	xforms : "http://www.w3.org/2002/01/xforms",
	xhtml : "http://www.w3.org/1999/xhtml",
	xi : "http://www.w3.org/2001/XInclude",
	xlink : "http://www.w3.org/1999/xlink",
	xsd : "http://www.w3.org/2001/XMLSchema",
	xsi : "http://www.w3.org/2001/XMLSchema-instance",
	xsl : "http://www.w3.org/1999/XSL/Transform",
	xslt : "http://www.w3.org/1999/XSL/Transform",
	xul : "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul"
};

/**
 * Looks up the prefix corresponding to a namespace in the mozile.xml.ns list.
 * @param {String} namespaceURI The URI of the namespace to find the prefix for.
 * @type String
 */
mozile.xml.lookupPrefix = function(namespaceURI) {
	for(var prefix in mozile.xml.ns) {
		if(mozile.xml.ns[prefix] == namespaceURI) return prefix;
	}
	return null;
}

/**
 * Loads and returns an XML document.
 * This method should work in most browsers.
 * The method checks for a ":" in the filepath. 
 *   If one is found the filepath is treated as an absolute path.
 *   If none is found it treats the filepath as a relative path.
 * <p>This method should work in most browsers.
 * @param {String} string
 * @type Document
 */
mozile.xml.load = function(filepath) {
	//alert("Loading");
	if(typeof(filepath) != "string") return null;


	// Make filepath absolute.
	var uri;
	if(filepath.indexOf(":") > 0) uri = filepath;
	else {
		var loc = location.toString();
		loc = loc.substring(0, loc.lastIndexOf("?"));
		loc = loc.substring(0, loc.lastIndexOf("/") + 1);
		uri = loc + filepath;
	}
	// Hack for accessing local files with Safari
	if(mozile.browser.isSafari) uri = uri.replace("file://", "file:///");

	try { 
		// Try to find an XMLHTTPRequest object
		var XHR = new XMLHttpRequest();
		XHR.overrideMimeType("text/xml");
		XHR.open("GET", uri, false);
		try {
			XHR.send(null);
			return XHR.responseXML;
		} catch(e) {
			mozile.debug.inform("mozile.xml.load", "Error loading document: "+ e);
			return null;
		}
	}
	catch(e) {
		try	{
			// An IE XML load method
			var xmlDoc = new ActiveXObject("Microsoft.XMLDOM");
			xmlDoc.async = false;
			try {
				var loaded = xmlDoc.load(uri);
				if(loaded) return xmlDoc;
				else {
					mozile.debug.inform("mozile.xml.load", "Failed to load document.");
				}
			} catch(e) {
				mozile.debug.inform("mozile.xml.load", "Error loading document: "+ mozile.dumpError(e));
			}
		}
		catch(e) {
			mozile.debug.inform("mozile.xml.load", "No XML loading technique avaliable in this browser.");
		}
	}

	return null;
}

/*
	var uri;
	if(filepath.indexOf(":") > 0) uri = filepath;
	else {
		var loc = location.toString();
		loc = loc.substring(0, loc.lastIndexOf("?"));
		loc = loc.substring(0, loc.lastIndexOf("/") + 1);
		uri = loc + filepath;
	}
	//alert("URI:"+uri);
	
	var xmlDoc = null;
	if(window.ActiveXObject) {
		xmlDoc = new ActiveXObject("Microsoft.XMLDOM");
	}
	else if(document.implementation) {
		var xmlDoc = document.implementation.createDocument("", "", null);
		
		// Handle Safari, which has no XMLDocument object or load() method.
		if(!xmlDoc.load) {
			// Hack for accessing local files.
			uri = uri.replace("file://", "file:///");
			var XHR = new XMLHttpRequest;
			XHR.open("GET", uri, false);
			XHR.send(null);

			// An idea to prevent crashes (doesn't help me):
			// http://www.jtauber.com/blog/2006/03/09/crashing_safari
			//var container = document.createElement("div");
			//container.innerHTML = XHR.responseText;
			//return container.innerHTML;


			return XHR.responseXML;
		}

	}
	else {
		mozile.debug.inform("mozile.xml.load", "No XML loading technique avaliable in this browser.");
		return null;
	}
	
	if(xmlDoc) {
		xmlDoc.async = false;
		try {
			alert("About to load "+ uri.substring(20));
			var loaded = xmlDoc.load(uri);
			alert("Loaded "+ uri.substring(20));
			//if(!loaded) throw Error("Could not load file "+ uri);
		} catch(e) {
			mozile.debug.inform("mozile.xml.load", "Failed to load document '"+ filepath +"' using path '"+ uri +"' because of an error:\n"+ mozile.dumpError(e));
		}
	}
	else {
		mozile.debug.inform("mozile.xml.load", "Failed to load document '"+ filepath +"' using path '"+ uri +"'.");
	}

	return xmlDoc;
}
*/

/**
 * Parses a string into an XML document, and returns the document.
 * <p>This method should work in most browsers.
 * @param {String} string
 * @type Document
 */
mozile.xml.parse = function(string) {
	if(window.ActiveXObject) {
		var xmlDoc = new ActiveXObject("Microsoft.XMLDOM");
		xmlDoc.async = "false";
		xmlDoc.loadXML(string);
		return xmlDoc;
	}
	else if(window.DOMParser) {
		var parser = new DOMParser();
		return parser.parseFromString(string, "text/xml");
	}
	else {
		mozile.debug.inform("mozile.xml.serialize", "No XML parsing technique avaliable in this browser.");
		return null;
	}
}

/**
 * Parses a string into an XML document, and returns the documentElement.
 * <p>This method should work in most browsers.
 * @param {String} string
 * @type Element
 */
mozile.xml.parseElement = function(string) {
	var doc = mozile.xml.parse(string);
	if(doc && doc.documentElement) return doc.documentElement;
	else return null;
}

/**
 * Serializes an XML node (or document) into a string.
 * <p>This method should work in most browsers.
 * @param {Node} node An XML node or document.
 * @type String
 */
mozile.xml.serialize = function(node) {
	if(!node) return null;
	if(node.xml) return node.xml;
	else if(window.XMLSerializer) {
		var serializer = new XMLSerializer()
		return serializer.serializeToString(node);
	}
	else if(node.outerHTML) return node.outerHTML;
	else if(node.innerHTML) {
		var container = document.createElement("container");
		container.appendChild(node.cloneNode(true));
		return container.innerHTML;
	}
	else {
		mozile.debug.inform("mozile.xml.serialize", "No XML serialization technique avaliable in this browser.");
		return null;
	}
}



/**
 * A collection of XPath functions.
 * @type Object
 */
mozile.xpath = new Object();
// JSDoc hack
mozile.xpath.prototype = new mozile.Module;



/** 
 * Returns a simple unique XPath location for the given node.
 * @param {Node} node The target node.
 * @param {Node} root The root of the XPath expression. Defaults to the document.
 * @type String
 */
mozile.xpath.getXPath = function(node, root) {
	if(!node || node.nodeType == undefined || !node.nodeType) return "";
	if(node == root) return "";
	var parent = node.parentNode;
	
	switch(node.nodeType) {
		// This should match up with the namespace resolver.
		// If it has a prefix, use the prefix.
		// If it has a namespaceURI, look it up.
		case mozile.dom.ELEMENT_NODE:
			var nodeName;
			if(node.prefix) nodeName = node.nodeName;
			//else if(node.namespaceURI) {
			//	var prefix = mozile.dom.lookupPrefix(node, node.namespaceURI);
			//	if(!prefix) prefix = mozile.xml.lookupPrefix(node.namespaceURI);
			//	if(prefix) nodeName = prefix +":"+ node.nodeName;
			//}
			else nodeName = node.nodeName;
			//if(!nodeName) nodeName = "xmlns:"+ node.nodeName;
			if(!nodeName) nodeName = node.nodeName;
			// TODO: This is a hack. Try to fix it.
			if(mozile.dom.isHTML(node)) nodeName = nodeName.toLowerCase();

			if(node.ownerDocument && node.ownerDocument.documentElement == node)
				return "/"+ nodeName +"[1]";
			else return mozile.xpath.getXPath(parent, root) 
				+"/"+ nodeName +"["+ mozile.dom.getPosition(node) +"]";
		
		case mozile.dom.ATTRIBUTE_NODE:
			if(!parent) parent = node.ownerElement;
			return mozile.xpath.getXPath(parent, root) +"/@"+ node.nodeName;

		case mozile.dom.TEXT_NODE:
			// Basic caching
			if(node == this._lastNode && root == this._lastRoot)
				return this._lastXPath;
			var xpath = mozile.xpath.getXPath(parent, root) 
				+"/text()["+ mozile.dom.getPosition(node) +"]";
			this._lastNode = node;
			this._lastRoot = root;
			this._lastXPath = xpath;
			return xpath;

		case mozile.dom.COMMENT_NODE:
			return mozile.xpath.getXPath(parent, root) 
				+"/comment()["+ mozile.dom.getPosition(node) +"]";

		default:
			if(parent) return mozile.xpath.getXPath(parent, root);
			else return "";
	}
}

/**
 * Takes an XPath expression, splits it at "/" characters, and returns a cleaned array.
 * @param {String} expression The XPath expression to be split.
 * @type Array
 */
mozile.xpath.getComponents = function(expression) {
	if(typeof(expression) != "string") return [];
	var components = expression.split("/");
	for(var c=0; c < components.length; c++) {
		if(!components[c]) components.splice(c,1);
	}
	return components;
}

/**
 * Takes a simple XPath expression and returns an object with "name", "localName", and "position" properties for the last node in the path.
 * @param {String} expression The XPath expression to be used.
 * @type Object
 */
mozile.xpath.getComponent = function(expression) {
	var result = new Object();
	var components = mozile.xpath.getComponents(expression);

	if(components.length == 0) return result;
	var component = components[components.length - 1];
	var match = component.match(/(\S+:)?(\S+)\[(\d+)\]|(\S+:)?(\S+)/);
	if(match) {
		if(match[1] && match[2]) {
			result.name = match[1] + match[2];
			result.localName = match[2];
		}
		else if(match[2]) {
			result.name = match[2];
			result.localName = match[2];
		}
		else if(match[4] && match[5]) {
			result.name = match[4] + match[5];
			result.localName = match[5];
		}
		else if(match[5]) {
			result.name = match[5];
			result.localName = match[5];
		}
		if(match[3]) result.position = match[3];
		else result.position = null;
	}

	return result;
}


/**
 * A basic XPath processor. Expects an XPath expression with only indices as positions which selects a single node, and returns that node.
 * Works on paths of the form /html[1]/body[1]/p[3]/text()[1]
 * <p>For simple paths, this should work in any browser. Under Mozilla and IE evaluateXPath is used.
 * @param {String} expression An XPath expression.
 * @param {Node} root An XML node or document. The path is evaluated relative to this node.
 * @type Node
 */
mozile.xpath.getNode = function(expression, root) {
	if(!root) root = document;
	if(!expression) return root;
	// Basic caching method
	if(expression == this._lastXPath && root == this._lastRoot) return this._lastNode;	

	var node;
	
	// TODO: Make use of native XPath systems.
	/*
	if(window.XPathEvaluator) {
	  var results = mozile.xpath.evaluate(expression, root);
	  if(results && results[0]) node = results[0];
	  else return null;
	}
	*/

	/*
	try {
		//if(root.selectSingleNode) return root.selectSingleNode(expression);
	} catch(e) {
		alert(e +"\n"+ mozile.dumpError(root));
	}
	//alert("crude method");
	*/

	
	if(!node) {
		var components = mozile.xpath.getComponents(expression);
		var node = root;
		for(var i=0; i < components.length; i++) {
			var component = mozile.xpath.getComponent(components[i]);
			node = mozile.xpath._getNthChild(node, component.name, component.position);
			if(!node) return null;
		}

	}

	//alert("Result:"+ node);
	if(node) {
		this._lastNode = node;
		this._lastRoot = root;
		this._lastXPath = expression;
		return node;	
	}
	else return null;
}

/**
 * Returns the Nth child of the given name. 
 * Used to resolve simple XPaths.
 * Works on paths of the form /html[1]/body[1]/p[3]/text()[1]
 * @private
 * @param {Node} parent The parent node to search.
 * @param {String} name The name of the target.
 * @param {Integer} position The position of the target.
 * @type Node
 */
mozile.xpath._getNthChild = function(parent, name, position) {
	var c = 0;
	var p = 1;
	//alert(parent +" "+ name +" "+ position);
	
	if(parent.nodeType == mozile.dom.DOCUMENT_NODE) 
		return parent.documentElement;

	if(name == "text()") {
		for(c=0; c < parent.childNodes.length; c++) {
			if(parent.childNodes[c].nodeType == mozile.dom.TEXT_NODE) {
				if(p == position) return parent.childNodes[c];
				p++;
			}
		}
	}

	else if(name == "comment()") {
		for(c=0; c < parent.childNodes.length; c++) {
			if(parent.childNodes[c].nodeType == mozile.dom.COMMENT_NODE) {
				if(p == position) return parent.childNodes[c];
				p++;
			}
		}
	}

	if(name.indexOf("@") == 0) {
		name = name.substring(1);
		return parent.getAttributeNode(name);
	}
	
	else {
		// Ignore the null namespace of "xmlns:"
		if(name.indexOf("xmlns:") > -1) name = name.substring(6);
		for(c=0; c < parent.childNodes.length; c++) {
			var childName = parent.childNodes[c].nodeName;
			if(childName == name || childName == name.toUpperCase()) {
				if(p == position) return parent.childNodes[c];
				p++;
			}
		}
	}

	return null;
}


/**
 * Evaluates an XPath expression in the context of the document or a node.
 * <p>This method only supports Mozilla browsers and IE versions with MSXML 3.0+.
 * For name space resolution, see http://www.faqts.com/knowledge_base/view.phtml/aid/34022/fid/119
 * TODO: The namespace resolver should be more clever.
 * @param {String} expression The XPath expression.
 * @param {Node} node Optional. The context within which the expression should be executed. The default is the documentElement.
 * @type Array
 * @return An array of results.
 */
mozile.xpath.evaluate = function(expression, root) {
	if(!root) root = document;
	
	var doc;
	if(root.ownerDocument) doc = root.ownerDocument;
	else {
		doc = root;
		root = root.documentElement;
	}
	
	var nodes = new Array();
	if(window.XPathEvaluator) {
		var XPE = new XPathEvaluator;
		
		// Create a namespace resolver based on the root.
		var NSE = function(prefix) {		
			//mozile.debug.debug("mozile.xpath.nsResolver", "Prefix: "+ prefix);
			var namespaceURI = mozile.dom.lookupNamespaceURI(root, prefix);
			if(namespaceURI) return namespaceURI;
			else if(mozile.xml.ns[prefix]) return mozile.xml.ns[prefix];
			return mozile.defaultNS;
		}

		try {
		var results = XPE.evaluate(expression, root, NSE, 0, null);
		if(results) {
			var result = results.iterateNext();
			while(result) {
				nodes.push(result);
				result = results.iterateNext();
			}
		}
		} catch(e) {
			alert(doc.documentElement.getAttribute("xmlns") +"\n"+ e);
		}
	}

	else if(root.selectNodes) {
		var results = root.selectNodes(expression);
		var result = results.nextNode();
		while(result) {
			nodes.push(result);
			result = results.nextNode();
		}
	}

	return nodes;
}



/**
 * The NodeFilter is used to describe which node should be included by the TreeWalker, and which should be ignored.
 * @type Object
 */
mozile.dom.NodeFilter = {
	FILTER_ACCEPT : 1, // means the node should be used
	FILTER_REJECT : 2, // means the node should not be used
	FILTER_SKIP   : 3, // means the node should not be used
	SHOW_ALL		 : -1,
	SHOW_ELEMENT : 1,
	SHOW_ATTRIBUTE : 2,
	SHOW_TEXT : 4,
	SHOW_CDATA_SECTION : 8,
	SHOW_ENTITY_REFERENCE : 16,
	SHOW_ENTITY : 32,
	SHOW_PROCESSING_INSTRUCTIONS : 64,
	SHOW_COMMENT : 128,
	SHOW_DOCUMENT : 256,
	SHOW_DOCUMENT_TYPE : 512,
	SHOW_DOCUMENT_FRAGMENT : 1024,
	SHOW_NOTATION : 2048
}


/**
 * The TreeWalker is used to traverse the DOM tree.
 * @param {Element} root A reference to the node that will serve as the root of the TreeWalker.
 * @param {Integer} whatToShow A bitmask of one or more of the NodeFilter flags that indicate what node types to look at. Though all the NodeFilter flags are defined, only the following will actually work in IE5-6: NodeFilter.SHOW_ALL, NodeFilter.SHOW_ELEMENT, NodeFilter.SHOW_ELEMENT, NodeFilter.SHOW_ATTRIBUTE, NodeFilter.SHOW_TEXT, NodeFilter.SHOW_COMMENT, NodeFilter.SHOW_DOCUMENT, NodeFilter.SHOW_DOCUMENT_FRAGMENT.
 * @param {NodeFilter} filter This is an object that has an acceptNode() method which accepts a node as it's single argument and returns either NodeFilter.FILTER_ACCEPT, NodeFilter.FILTER_REJECT, or NodeFilter.FILTER_SKIP to indicate if the node should be included in the TreeWalker.
 * @param {Boolean} expandEntityReferences Not implemented. A boolean to indicate if entity references should be expanded. Because this script was developed for HTML documents, this argument is ignored.
 * @constructor
 */
mozile.dom.TreeWalker = function(root, whatToShow, filter, expandEntityReferences) {
	/**
	 * The root of the tree being walked.
	 * @type Element
	 */
	this.root = root;
	
	/**
	 * A flag indicating which node types to show.
	 * @type Integer
	 */
	this.whatToShow = whatToShow;
	
	/**
	 * A NodeFilter object which will be used to filter the nodes.
	 * @type NodeFIlter
	 */
	this.filter = filter;
	
	/**
	 * Indicates whether entity references should be expanded.
	 * @type Boolean
	 */
	this.expandEntityReferences = expandEntityReferences;
	
	/**
	 * A reference to the current node being examined by this TreeWalker.
	 * @type Node
	 */
	this.currentNode = root;
}



//*************************************
// Public Members

/**
 * The parentNode() method looks at the current node's ancestor nodes (starting with the parent node and working up) to find the first ancestor that passes the treeWalker's filter. If one is found, that node is returned and it becomes the current node for the TreeWalker. If the root node is reached before any acceptable nodes are found, the method returns null and the current node is not changed.
 * @type Node
 * @return The first ancestor node of the current node that passes the treeWalker's filter; null if the treeWalker's root node is reached before any acceptable nodes are found or if no parent node is found.
 */
mozile.dom.TreeWalker.prototype.parentNode = function() {
	var testNode = this.currentNode;
	do {
		if (testNode != this.root &&
			testNode.parentNode != this.root && 
			testNode.parentNode != null) {
			testNode = testNode.parentNode;
		}
		else return null;
	} while (this._getFilteredStatus(testNode) != mozile.dom.NodeFilter.FILTER_ACCEPT);

	if (testNode != null) this.currentNode = testNode;
	return testNode;
}

/**
 * The firstChild() method looks through the current node's child nodes (starting with the first child and working down) to find a child that passes the treeWalker's filter. If one is found, that child is returned and it becomes the current node for the TreeWalker. If no acceptable node is found, the method returns null and the current node is not changed.
 * @type Node
 * @return The first child node of the current node that passes the treeWalker's filter; null if the current node has no child nodes or if none of the child nodes pass the filter.
 */
mozile.dom.TreeWalker.prototype.firstChild = function() {
	var testNode = this.currentNode.firstChild;
	while(testNode != null) {
		if(this._getFilteredStatus(testNode) == mozile.dom.NodeFilter.FILTER_ACCEPT) {
			break;
		}
		testNode = testNode.nextSibling;
	}

	if (testNode != null) this.currentNode = testNode;
	return testNode;
}

/**
 * The lastChild() method looks through the current node's child nodes (starting with the last child and working up) to find a child that passes the treeWalker's filter. If one is found, that child is returned and it becomes the current node for the TreeWalker. If no acceptable node is found, the method returns null and the current node is not changed.
 * @type Node
 * @return The last child node of the current node that passes the treeWalker's filter; null if the current node has no child nodes or if none of the child nodes pass the filter.
 */
mozile.dom.TreeWalker.prototype.lastChild = function() {
	var testNode = this.currentNode.lastChild;
	while(testNode != null) {
		if(this._getFilteredStatus(testNode) == mozile.dom.NodeFilter.FILTER_ACCEPT) {
			break;
		}
		testNode = testNode.previousSibling;
	}

	if (testNode != null) this.currentNode = testNode;
	return testNode;	
}

/**
 * The nextNode() and previousNode() methods look at a flattened version of the document (instead of the other methods that preserve the document's structure). The nextNode() method returns the next node after the current node that passes the treeWalker's filter. If one is found, that node is returned and it becomes the current node for the TreeWalker. If there are no root node descendants after the current node that pass the filter, the method returns null and the current node is not changed.
 * @type Node
 * @return The node after the current node in a flattened view of the document that passes the treeWalker's filter; null if there are no nodes after the current node that are also descendants of the root node.
 */
mozile.dom.TreeWalker.prototype.nextNode = function() {
	var testNode = this.currentNode;
	while (testNode != null) {
		// next node is the first child, if any
		if (testNode.childNodes.length != 0) testNode = testNode.firstChild;
		
		// or the next sibling, if any
		else if (testNode.nextSibling != null) testNode = testNode.nextSibling;
		
		// or the closest ancestor's next sibling, if any
		else {
			while (testNode != null) {
				if (testNode.parentNode != this.root && testNode.parentNode != null) {
					if (testNode.parentNode.nextSibling != null) {
						testNode = testNode.parentNode.nextSibling;
						break;
					}
					else testNode = testNode.parentNode;
				}
				else return null;
			}
		}
		
		if(testNode != null && this._getFilteredStatus(testNode) == mozile.dom.NodeFilter.FILTER_ACCEPT) {
			break;
		}
	}
	
	if (testNode != null) this.currentNode = testNode;
	return testNode;
}

/**
 * The nextNode() and previousNode() methods look at a flattened version of the document (instead of the other methods that preserve the document's structure). The previousNode() method returns the node before the current node that passes the treeWalker's filter. If one is found, that node is returned and it becomes the current node for the TreeWalker. If there are no root node descendants before the current node that pass the filter, the method returns null and the current node is not changed.
 * @type Node
 * @return The node before the current node in a flattened view of the document that passes the treeWalker's filter; null if there are no nodes before the current node that are also descendants of the root node.
 */
mozile.dom.TreeWalker.prototype.previousNode = function() {
	 
	// look for a filter-acceptable node before current node
	var testNode = this.currentNode;
	while (testNode != null) {
		
		// previous node is the previous sibling's last child, 
		// or the previous sibling, if one exists
		if (testNode.previousSibling != null) {
			testNode = testNode.previousSibling;
			while (testNode.lastChild != null) {
				testNode = testNode.lastChild;
			}
		}
		
		// or the parent node
		else {
			if (testNode.parentNode != this.root && testNode.parentNode != null) {
				testNode = testNode.parentNode;
			}
			else testNode = null;
		}
		
		if (testNode != null && this._getFilteredStatus(testNode) == mozile.dom.NodeFilter.FILTER_ACCEPT) {
			break;
		}
	}
	
	if (testNode != null) this.currentNode = testNode;
	return testNode;
}

/**
 * The nextSibling() method looks through the current node's siblings nodes (starting with the next sibling and working down) to find a sibling that passes the treeWalker's filter. If one is found, that node is returned and it becomes the current node for the TreeWalker. If no acceptable node is found, the method returns null and the current node is not changed.
 * @type Node
 * @return The next sibling node of the current node that passes the treeWalker's filter; null if the current node has no sibling nodes after it or if none of the later sibling nodes pass the filter.
 */
mozile.dom.TreeWalker.prototype.nextSibling = function() {
	var testNode = this.currentNode;
	while(testNode != null) {
		if(testNode.nextSibling != null) testNode = testNode.nextSibling;
		if(this._getFilteredStatus(testNode) == mozile.dom.NodeFilter.FILTER_ACCEPT) {
			break;
		}
	}
	
	if (testNode != null) this.currentNode = testNode;
	return testNode;
}

/**
 * The previousSibling() method looks through the current node's siblings nodes (starting with the previous sibling and working up) to find a sibling that passes the treeWalker's filter. If one is found, that node is returned and it becomes the current node for the TreeWalker. If no acceptable node is found, the method returns null and the current node is not changed.
 * @type Node
 * @return The previous sibling node of the current node that passes the treeWalker's filter; null if the current node has no sibling nodes before it or if none of the previous sibling nodes pass the filter.
 */
mozile.dom.TreeWalker.prototype.previousSibling = function() {
	var testNode = this.currentNode;
	while(testNode != null) {
		if(testNode.previousSibling != null) testNode = testNode.previousSibling;
		if(this._getFilteredStatus(testNode) == mozile.dom.NodeFilter.FILTER_ACCEPT) {
			break;
		}
	}
	
	if (testNode != null) this.currentNode = testNode;
	return testNode;
}



//*************************************
// Private Members

/**
 * Determines the filtered status of the given node.
 * Uses a bitmask system. For using bitmasks, see http://developer.mozilla.org/en/docs/Core_JavaScript_1.5_Reference:Operators:Bitwise_Operators#Example:_Flags_and_bitmasks
 * @param {Node} node The node to check against the "whatToShow" value for the TreeWalker.
 * @private
 * @type Integer
 */
mozile.dom.TreeWalker.prototype._getFilteredStatus = function(node) {

	var mask = null;
	switch(node.nodeType) {
		case mozile.dom.ELEMENT_NODE:
			mask = mozile.dom.NodeFilter.SHOW_ELEMENT;
			break;
		case mozile.dom.ATTRIBUTE_NODE:
			mask = mozile.dom.NodeFilter.SHOW_ATTRIBUTE;
			break;
		case mozile.dom.TEXT_NODE:
			mask = mozile.dom.NodeFilter.SHOW_TEXT;
			break;
		case mozile.dom.CDATA_SECTION_NODE:
			mask = mozile.dom.NodeFilter.SHOW_CDATA_SECTION;
			break;
		case mozile.dom.ENTITY_REFERENCE_NODE:
			mask = mozile.dom.NodeFilter.SHOW_ENTITY_REFERENCE;
			break;
		case mozile.dom.ENTITY_NODE:
			mask = mozile.dom.NodeFilter.SHOW_PROCESSING_INSTRUCTION;
			break;
		case mozile.dom.PROCESSING_INSTRUCTION_NODE:
			mask = mozile.dom.NodeFilter.SHOW_PROCESSING_INSTRUCTION;
			break;
		case mozile.dom.COMMENT_NODE:
			mask = mozile.dom.NodeFilter.SHOW_COMMENT;
			break;
		case mozile.dom.DOCUMENT_NODE:
			mask = mozile.dom.NodeFilter.SHOW_DOCUMENT;
			break;
		case mozile.dom.DOCUMENT_TYPE_NODE:
			mask = mozile.dom.NodeFilter.SHOW_DOCUMENT_TYPE;
			break;
		case mozile.dom.DOCUMENT_FRAGMENT_NODE:
			mask = mozile.dom.NodeFilter.SHOW_DOCUMENT_FRAGMENT;
			break;
		case mozile.dom.NOTATION_NODE:
			mask = mozile.dom.NodeFilter.SHOW_NOTATION;
			break;
	}
	
	// Use a bit mask to determine if whatToShow includes this node.
	if(mask != null && (this.whatToShow & mask) == 0) {
		return mozile.dom.NodeFilter.FILTER_REJECT;
	}
	// Check node against filter if one exists
	if (this.filter != null && this.filter.acceptNode != null) {
		return this.filter.acceptNode(node);
	}
	return mozile.dom.NodeFilter.FILTER_ACCEPT;
}


/**** Final Configuration ****/

if(mozile.browser.isIE) {

/**
 * Creates a new TreeWalker object
 * @param {Element} root A reference to the node that will serve as the root of the TreeWalker.
 * @param {Integer} whatToShow A bitmask of one or more of the NodeFilter flags that indicate what node types to look at. Though all the NodeFilter flags are defined, only the following will actually work in IE5-6: NodeFilter.SHOW_ALL, NodeFilter.SHOW_ELEMENT, NodeFilter.SHOW_ELEMENT, NodeFilter.SHOW_ATTRIBUTE, NodeFilter.SHOW_TEXT, NodeFilter.SHOW_COMMENT, NodeFilter.SHOW_DOCUMENT, NodeFilter.SHOW_DOCUMENT_FRAGMENT.
 * @param {NodeFilter} filter This is an object that has an acceptNode() method which accepts a node as it's single argument and returns either NodeFilter.FILTER_ACCEPT, NodeFilter.FILTER_REJECT, or NodeFilter.FILTER_SKIP to indicate if the node should be included in the TreeWalker.
 * @param {Boolean} expandEntityReferences Not implemented. A boolean to indicate if entity references should be expanded. Because this script was developed for HTML documents, this argument is ignored.
 * @type TreeWalker
 */
document.createTreeWalker = function(root, whatToShow, filter, expandEntityReferences) {
	return new mozile.dom.TreeWalker(root, whatToShow, filter, expandEntityReferences);
};

	mozile.dom.NodeFilter = mozile.dom.NodeFilter;
	mozile.dom.TreeWalker = mozile.dom.TreeWalker;
}
else {
	mozile.dom.NodeFilter = NodeFilter;
	mozile.dom.TreeWalker = TreeWalker;
}



/**
 * Range object, see http://www.w3.org/TR/DOM-Level-2-Traversal-Range/ranges.html
 * @constructor
 */
mozile.dom.InternetExplorerRange = function(range) {
	/**
	 * A reference to an IE native TextRange object.
	 * @private
	 * @type TextRange
	 */
	this._range = null;

	if(range)
		this._range = range;
	else {
		this._range = document.body.createTextRange();
	}

	/**
	 * A boolean indicating whether the range's start and end points are at the same position.
	 * @type Boolean
	 */
	this.collapsed = null;

	/**
	 * The deepest Node that contains the startContainer and endContainer Nodes.
	 * @type Node
	 */
	this.commonAncestorContainer = null;
	
	/**
	 * The Node within which the Range starts.
	 * @type Node
	 */
	this.startContainer = null;

	/**
	 * A number representing where in the endContainer the Range starts.
	 * @type Integer
	 */
	this.startOffset = null;

	/**
	 * The Node within which the Range ends.
	 * @type Node
	 */
	this.endContainer = null;

	/**
	 * A number representing where in the endContainer the Range ends.
	 * @type Integer
	 */
	this.endOffset = null;
}


/**
 * Initializes the properties of this range according to the internal
 * IE range (_range).
 * @private
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype._init = function () {
	
	//startPoint
	var beginRange = this._range.duplicate();
	beginRange.collapse(true);
	var position = this._getPosition(beginRange);
	this.startContainer = position.node;
	this.startOffset = position.offset;

	//endPoint
	var endRange = this._range.duplicate();
	endRange.collapse(false);
	position = this._getPosition(endRange);
	this.endContainer = position.node;
	this.endOffset = position.offset;
	
	this._commonAncestorContainer();
	this._collapsed();
	
}


/**
 * Takes an Internet Explorer TextRange object and returns a W3C node and offset pair.
 * <p>The basic method is as follows:
 * <ul><li>Create a new range with its start at the beginning of the element and its end at the target position. Set the rangeLength to the length of the range's text.
 * <li>Starting with the first child, for each child:
 * <ul><li>If the child is a text node, and its length is less than the rangeLength, then move the range's start by the text node's length.
 * <li>If the child is a text node and its length is less than the rangeLength then we've found the target. Return the node and use the remaining rangeLength as the offset.
 * <li>If the child is an element, move the range's start by the length of the element's innerText.
 * </ul></ul>
 * <p>This algorithm works fastest when the target is close to the beginning of the parent element.
 * The current implementation is smart enough pick the closest end point of the parent element (i.e. the start or the end), and work forward or backward from there.
 * @private
 * @param {TextRange} textRange A TextRange object. Its start position will be found.
 * @type Object
 * @return An object with "node" and "offset" properties.
 */
mozile.dom.InternetExplorerRange.prototype._getPosition = function(textRange) {
	var element = textRange.parentElement();
	var range = document.body.createTextRange();
	range.moveToElementText(element);
	range.setEndPoint("EndToStart", textRange);
	var rangeLength = range.text.length;

	// Choose Direction
	if(rangeLength < element.innerText.length / 2) {
		var direction = 1;
		var node = element.firstChild;
	}
	else {
		direction = -1;
		node = element.lastChild;
		range.moveToElementText(element);
		range.setEndPoint("StartToStart", textRange);
		rangeLength = range.text.length;
	}

	// Loop through child nodes
	while(node) {
		switch(node.nodeType) {
			case mozile.dom.TEXT_NODE:
				nodeLength = node.data.length;
				if(nodeLength < rangeLength) {
					var difference = rangeLength - nodeLength;
					if(direction == 1) range.moveStart("character", difference);
					else range.moveEnd("character", -difference);
					rangeLength = difference;
				}
				else {
					if(direction == 1) return {node: node, offset: rangeLength};
					else return {node: node, offset: nodeLength - rangeLength};
				}
				break;

			case mozile.dom.ELEMENT_NODE:
				nodeLength = node.innerText.length;
				if(direction == 1) range.moveStart("character", nodeLength);
				else range.moveEnd("character", -nodeLength);
				rangeLength = rangeLength - nodeLength;
				break;
		}
	
		if(direction == 1) node = node.nextSibling;
		else node = node.previousSibling;
	}


	// TODO: This should throw a warning.
	//throw("Error in mozile.dom.InternetExplorerRange._getPosition: Ran out of child nodes before the range '"+ textRange.text +"' inside '"+ mozile.xpath.getXPath(element) +"' was found.");
	
	// The TextRange was not found. Return a reasonable value instead.
	return {node: element, offset: 0};
	
}


/**
 * Find the TextRange offset for a given text node and offset. Effectively the opposite of getPosition().
 * The method used is to count the innerText length for elements and the data length for text nodes.
 * @private
 * @param {Text} startNode The target text node.
 * @param {Integer} startOffset
 * @type Integer
 */
mozile.dom.InternetExplorerRange.prototype._getOffset = function (startNode, startOffset) {
	var node, moveCharacters;
	if(startNode.nodeType == mozile.dom.TEXT_NODE) {
		moveCharacters = startOffset;
		node = startNode.previousSibling;
	}
	else if(startNode.nodeType == mozile.dom.ELEMENT_NODE) {
		moveCharacters = 0;
		if(startOffset > 0) node = startNode.childNodes[startOffset - 1];
		else return 0;
	}
	else {
		mozile.debug.inform("mozile.dom.InternetExplorerRange.prototype._getOffset", "Bad node given: "+ mozile.xpath.getXPath(startNode));
		return 0;
	}

	while (node) {
		var nodeLength = 0;
		if(node.nodeType == mozile.dom.ELEMENT_NODE) {
			nodeLength = node.innerText.length;
			if(this._isChildless(node)) nodeLength = 1; // Tweak childless nodes.
			if(this._isBlock(node)) nodeLength++; // Tweak block level elements.
			//if(nodeLength == 0) nodeLength++; // minimum length is 1
		}
		else if(node.nodeType == mozile.dom.TEXT_NODE) {
			 nodeLength = node.data.length;
		}
		moveCharacters += nodeLength;
		node = node.previousSibling;
	}
	
	return moveCharacters;
}

/**
 * Internet Explorer pads certain elements with an extra space at the end. This method detects those elements.
 * TODO: This method should be smarter about detecting non-HTML or using CSS.
 * @param {Node} node The node to check.
 * @type Boolean
 */
mozile.dom.InternetExplorerRange.prototype._isBlock = function(node) {
	switch (node.nodeName.toLowerCase()) {
		case 'p':
		case 'div':
		case 'h1':
		case 'h2':
		case 'h3':
		case 'h4':
		case 'h5':
		case 'h6':
		case 'pre':
			return true;
	}
	return false;
}

/**
 * Internet Explorer sets the length of certain elements which cannot have child nodes to 1.
 * TODO: Complete this list.
 * @param {Node} node The node to check.
 * @type Boolean
 */
mozile.dom.InternetExplorerRange.prototype._isChildless = function(node) {
	switch (node.nodeName.toLowerCase()) {
		case 'img':
		case 'br':
		case 'hr':
			return true;
	}
	return false;
}


// == positioning ==
/**
 * Sets the start position of a Range
 * If the startNode is a Node of type Text, Comment, or CDATASection, then startOffset 
 * is the number of characters from the start of startNode. For other Node types, 
 * startOffset is the number of child nodes between the start of the startNode.
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.setStart = function (startNode, startOffset) {
	var container = startNode;
	if(startNode.nodeType == mozile.dom.TEXT_NODE || 
		startNode.nodeType == mozile.dom.COMMENT_NODE || 
		startNode.nodeType == mozile.dom.CDATA_SECTION_NODE) {
		container = container.parentNode;
	}

	var copyRange = this._range.duplicate();
	copyRange.moveToElementText(container);
	copyRange.collapse(true);
	copyRange.move('Character', this._getOffset(startNode, startOffset));
	this._range.setEndPoint('StartToStart', copyRange);

	//update object properties
	this.startContainer = startNode;
	this.startOffset    = startOffset;
	if (this.endContainer == null && this.endOffset == null) {
		this.endContainer = startNode;
		this.endOffset    = startOffset;
	}
	this._commonAncestorContainer();
	this._collapsed();
}


/**
 * Sets the end position of a Range.
 * Creates a clone of the current range, moves it to the desired spot
 * and the we move the endPoint of the current range to the clones endpoint
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.setEnd = function (endNode, endOffset) {
	// Store the start of the range
	var copyRange = this._range.duplicate();
	copyRange.collapse(true);
	
	var container = endNode;
	if(endNode.nodeType == mozile.dom.TEXT_NODE || 
		endNode.nodeType == mozile.dom.COMMENT_NODE || 
		endNode.nodeType == mozile.dom.CDATA_SECTION_NODE) {
		container = container.parentNode;
	}

	var copyRange = this._range.duplicate();
	copyRange.moveToElementText(container);
	copyRange.collapse(true);
	copyRange.move('Character', this._getOffset(endNode, endOffset));
	this._range.setEndPoint('EndToEnd', copyRange);
	
	//update object properties
	this.endContainer = endNode;
	this.endOffset    = endOffset;
	if (this.startContainer == null && this.startOffset == null) {
		this.startContainer = endNode;
		this.startOffset    = endOffset;
	}
	this._commonAncestorContainer();
	this._collapsed();
}


/**
 * Sets the start position of a Range relative to another Node.
 * The parent Node of the start of the Range will be the same as 
 * that for the referenceNode.
 * @param {Node} referenceNode
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.setStartBefore = function (referenceNode) {
	this.setStart(referenceNode.parentNode, mozile.dom.getIndex(referenceNode));
};

/**
 * Sets the start position of a Range relative to another Node.
 * @param {Node} referenceNode
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.setStartAfter = function (referenceNode) {
	this.setStart(referenceNode.parentNode, mozile.dom.getIndex(referenceNode) + 1);
};

/**
 * Sets the end position of a Range relative to another Node.
 * @param {Node} referenceNode
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.setEndBefore = function (referenceNode) {
	this.setEnd(referenceNode.parentNode, mozile.dom.getIndex(referenceNode));
};

/**
 * Sets the end position of a Range relative to another Node.
 * @param {Node} referenceNode
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.setEndAfter = function (referenceNode) {
	this.setEnd(referenceNode.parentNode, mozile.dom.getIndex(referenceNode) + 1);
};

/**
 * Sets the Range to contain the node and its contents.
 * The parent Node of the start and end of the Range will be the same as
 * the parent of the referenceNode.
 * @param {Node} referenceNode
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.selectNode = function (referenceNode) {
	this.setStartBefore(referenceNode);
	this.setEndAfter(referenceNode);
};

/**
 * Sets the Range to contain the contents of a Node.
 * The parent Node of the start and end of the Range will be the referenceNode. The 
 * startOffset is 0, and the endOffset is the number of child Nodes or number of characters 
 * contained in the reference node.
 * @param {Node} referenceNode
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.selectNodeContents = function (referenceNode) {
	this.setStart(referenceNode, 0);
	if(referenceNode.nodeType == mozile.dom.TEXT_NODE)
		this.setEnd(referenceNode, referenceNode.data.length);
	else
		this.setEnd(referenceNode, referenceNode.childNodes.length);
};

/**
 * Collapses the Range to one of its boundary points.
 * @param {Boolean} toStart When true the Range is collapsed to the start position, when false to the end position.
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.collapse = function (toStart) {
	this._range.collapse(toStart);
	
	//update the properties
	if(toStart) {
		this.endContainer = this.startContainer;
		this.endOffset = this.startOffset;
	} else {
		this.startContainer = this.endContainer;
		this.startOffset = this.endOffset;
	}
	this._commonAncestorContainer();
	this._collapsed();
};

// == editing ==
/**
 * Returns a document fragment copying the nodes of a Range.
 * Partially selected nodes include the parent tags necessary to make the 
 * document fragment valid.
 * @type Range
 */
mozile.dom.InternetExplorerRange.prototype.cloneContents = function () {
	var df = document.createDocumentFragment();
	
	var container = this.commonAncestorContainer;
	if(container.nodeType == mozile.dom.TEXT_NODE) {
		df.appendChild(document.createTextNode(this._range.text));
		return df;
	}
	
	var startNode = this.startContainer;
	if(this.startContainer.nodeType != mozile.dom.TEXT_NODE)
		startNode = this.startContainer.childNodes[this.startOffset];
	var endNode = this.endContainer;
	if(this.endContainer.nodeType != mozile.dom.TEXT_NODE)
		endNode = this.endContainer.childNodes[this.endOffset - 1];

	if(startNode == endNode) {
		df.appendChild(startNode.cloneNode(true));
		return df;
	}

	// Walk the tree.
	var current = container.firstChild;
	var parent = null;
	var clone;
	while(current) {
		//alert(current.nodeName +"\n"+ df.innerHTML);
		// Watch for the start node, then start adding nodes.
		if(!parent) {
			if(mozile.dom.isAncestorOf(current, startNode, container)) {
				parent = df;
			}
			// Skip this node.
			else {
				current = current.nextSibling;
				continue;
			}
		}

		// Clone the node.
		if(current == startNode && this.startContainer.nodeType == mozile.dom.TEXT_NODE) {
			content = this.startContainer.data.substring(this.startOffset);
			parent.appendChild(document.createTextNode(content));
		}
		else if(current == endNode) {
			if(this.endContainer.nodeType == mozile.dom.TEXT_NODE) {
				content = this.endContainer.data.substring(0, this.endOffset);
				parent.appendChild(document.createTextNode(content));
			}
			else parent.appendChild(endNode.cloneNode(false));
			// We're done.
			break; 
		}
		else {
			clone = current.cloneNode(false);
			parent.appendChild(clone);
		}
		
		// Move
		if(current.firstChild) {
			parent = clone;
			current = current.firstChild;
		}
		else if(current.nextSibling) {
			current = current.nextSibling;
		}
		// Climb the tree
		else while(current) {
			if(current.parentNode) {
				parent = parent.parentNode;
				current = current.parentNode;
				if(current.nextSibling) {
					current = current.nextSibling;
					break;
				}
			}
			else current = null;
		}
	}
	
	return df;
};

/**
 * Removes the contents of a Range from the document.
 * Unlike extractContents, this method does not return a documentFragment 
 * containing the deleted content. 
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.deleteContents = function () {
	this._range.pasteHTML('');//This is incorrect, it might also delete the container

	//update properties
	this.endContainer = this.startContainer;
	this.endOffset = this.startOffset;
	this._commonAncestorContainer();
	this._collapsed();
};

/**
 * Moves contents of a Range from the document tree into a document fragment.
 * @type DocumentFragment
 */
mozile.dom.InternetExplorerRange.prototype.extractContents = function () {
	var fragment = this.cloneContents();
	this.deleteContents();
	return fragment;
};

/**
 * Insert a node at the start of a Range.
 * newNode is inserted at the start boundary point of the Range. If the newNodes 
 * is to be added to a text Node, that Node is split at the insertion point, and 
 * the insertion occurs between the two text Nodes
 * 
 * If newNode is a document fragment, the children of the document fragment are  
 * inserted instead.
 * @param {Node} newNode The node to insert.
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.insertNode = function (newNode) {
	//salert('mozile.dom.InternetExplorerRange.insertNode() is not implemented yet');
	if(this.startContainer.nodeType == mozile.dom.TEXT_NODE){
		this.startContainer.splitText(this.startOffset);
		this.startContainer.parentNode.insertBefore(newNode, this.startContainer.nextSibling);
		this.setStart(this.startContainer, this.startOffset);
		//Mozilla collapses, is this needed?
		//this.collapse(true);
		//this._range.select();
		return;
	} else { //Element node
		var parentNode = this.startContainer.parentNode;
		if(this.startContainer.childNodes.length == this.startOffset) {
			parentNode.appendChild(newNode);
		}else {
			this.startContainer.insertBefore(newNode, this.startContainer.childNodes.item(this.startOffset));
			this.setStart(this.startContainer, this.startOffset+1);
			//this._range.select();
			return;
		}
	}
};

/**
 * Moves content of a Range into a new node.
 * SurroundContents is equivalent to newNode.appendChild(range.extractContents());
 * range.insertNode(newNode). After surrounding, the boundary points of the Range 
 * include newNode.
 * @param {Node} newNode The node to select.
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.surroundContents = function (newNode) {
	newNode.appendChild(this.extractContents());
	this.insertNode(newNode);
};

// == Other ==
/**
 * NOT IMPLEMENTED. Compares the boundary points of two Ranges
 * Returns a number, 1, 0, or -1, indicating whether the corresponding boundary-point of range is respectively before, equal to, or after the corresponding boundary-point of sourceRange
 * Any of the following constants can be passed as the value of how parameter:
 * Range.END_TO_END compares the end boundary-point of sourceRange to the end boundary-point of range.
 * Range.END_TO_START compares the end boundary-point of sourceRange to the start boundary-point of range.
 * Range.START_TO_END compares the start boundary-point of sourceRange to the end boundary-point of range.
 * Range.START_TO_START compares the start boundary-point of sourceRange to the start boundary-point of range
 * @param {Integer} how A code for the comparison method.
 * @param {Range} sourceRange
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.compareBoundaryPoints = function (how, sourceRange) {
	alert('mozile.dom.InternetExplorerRange.compareBoundaryPoints() is not implemented yet');
};

/**
 * Returns a Range object with boundary points identical to the cloned Range.
 * @type Range
 */
mozile.dom.InternetExplorerRange.prototype.cloneRange = function () {
	var r = new mozile.dom.InternetExplorerRange(this._range.duplicate());
	var properties = ["startContainer", "startOffset", "endContainer", "endOffset", "commonAncestorContainer", "collapsed"];
	for(var i=0; i < properties.length; i++) {
		r[properties[i]] = this[properties[i]];
	}
	return r;
};

/**
 * Releases Range from use to improve performance.
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.detach = function () {};

/**
 * Returns the text of the Range.
 * @type String
 */
mozile.dom.InternetExplorerRange.prototype.toString = function () {
	return this._range.text;
};

/**
 * Finds the commonAncestorComtainer.
 * @private
 * @type Element
 */
mozile.dom.InternetExplorerRange.prototype._commonAncestorContainer = function () {
	if(this.startContainer == null || this.endContainer == null){
		this.commonAncestorContainer = null;
		return;
	}
	if(this.startContainer == this.endContainer) {
		this.commonAncestorContainer = this.startContainer;	
	}
	else {
		this.commonAncestorContainer = mozile.dom.getCommonAncestor(this.startContainer, this.endContainer);
	}
}

/**
 * Check to see if this selection is collapsed, and assign a value to this.collapsed.
 * @private
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype._collapsed = function () {
	this.collapsed = (this.startContainer == this.endContainer && this.startOffset == this.endOffset);
}



/**
 * Store the details about this range in an object which can be used to restore the range.
 * @type Object
 */
mozile.dom.InternetExplorerRange.prototype.store = mozile.dom.Range.prototype.store;

/**
 * Takes a stored range object and creates a new range with the same properties.
 * @param {Object} state A state object from Selection.store() or Range.store().
 * @type Void
 */
mozile.dom.InternetExplorerRange.prototype.restore = mozile.dom.Range.prototype.restore;




/**
 * Selection object, based on the mozilla implementation.
 * See http://www.xulplanet.com/references/objref/Selection.html
 * @constructor
 */
mozile.dom.InternetExplorerSelection = function() {
	/**
	 * A reference to IE's native selection object.
	 * @private
	 * @type Selection
	 */
	this._selection = document.selection;

	/**
	 * The node in wich the selection begins.
	 * @type Node
	 */
	this.anchorNode = null; 

	/**
	 * The number of characters that the selection's anchor is offset within the ancherNode.
	 * @type Integer
	 */
	this.anchorOffset = null; 

	/**
	 * The node in which the selection ends.
	 * @type Node
	 */
	this.focusNode = null; 

	/**
	 * The number of characters that the selection's focus is offset within the focusNode.
	 * @type Integer
	 */
	this.focusOffset = null;

	/**
	 * A boolean indicating whether the selection's start and end points are at the same position.
	 * @type Boolean
	 */
	this.isCollapsed = null;

	/**
	 * The number of ranges in the selection.
	 * @type Integer
	 */
	this.rangeCount = 0;

	/**
	 * The direction of the selection. Can be Left-to-Right or Right-to-Left, using coded integers.
	 * @private
	 * @type Integer
	 */
	this._direction = this._LTR;
	
	// Initialize the properties.
	this._init();
}

/**
 * Code indicating the selection direction. Left-to-right.
 * @private
 * @type Integer
 */
mozile.dom.InternetExplorerSelection.prototype._LTR = 1;

/**
 * Code indicating the selection direction. Right-to-left.
 * @private
 * @type Integer
 */
mozile.dom.InternetExplorerSelection.prototype._RTL = -1;

/**
 * Returns a range object representing one of the ranges currently selected.
 * @private
 * @param {Range} range Optional. A range object to use rather than using the getRangeAt(0).
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype._init = function(range) {
	if(!range) range = this.getRangeAt(0);
	
	if(!this._direction) this._direction = this._LTR;
	
	if(range && range.startContainer) {
		if(this._direction == this._LTR) {
			this.anchorNode = range.startContainer; 
			this.anchorOffset = range.startOffset; 
			this.focusNode = range.endContainer;
			this.focusOffset = range.endOffset;
		}
		else {
			this.anchorNode = range.endContainer; 
			this.anchorOffset = range.endOffset; 
			this.focusNode = range.startContainer;
			this.focusOffset = range.startOffset;
		}
		this.isCollapsed = range.collapsed;
	}
	else {
		this.anchorNode = null; 
		this.anchorOffset = null; 
		this.focusNode = null;
		this.focusOffset = null;
		this.isCollapsed = true;
	}
	//this.currentRange = range;
	this.rangeCount = this._selection.createRangeCollection().length;
}

/**
 * Returns a range object representing one of the ranges currently selected
 * @param {Integer} index The index of the desired range.
 * @type Range
 */
mozile.dom.InternetExplorerSelection.prototype.getRangeAt = function (index) {
	/*
	range = new mozile.dom.InternetExplorerRange(this._selection.createRangeCollection().item(index));
	range._init();
	return range;
	*/

	// Attempted optimization:
	var textRange = this._selection.createRangeCollection().item(index).duplicate();
	var range;
	// Trying to cache range, but it's too slow.
	if(this._lastTextRange && this._lastTextRange.isEqual(textRange)) {
		range = this._lastRange;
	}
	else {
		range = new mozile.dom.InternetExplorerRange(textRange);
		range._init();
		this._lastTextRange = textRange.duplicate();
		this._lastRange = range;
		this._direction = this._LTR;
	}

	return range.cloneRange();
}

/**
 * Collapses the current selection to a single point. 
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.collapse = function (parentNode, offset) {
	try {
		var range = this.getRangeAt(0);
		range.setStart(parentNode, offset);
		range.setEnd(parentNode, offset);
		range.collapse(false);
		this._direction = this._LTR;
		//alert("Collapsed 1 "+ mozile.util.dumpValues(range.store()));
		// Store the values first, then select the TextRange.
		this._init(range); 
		//alert("Collapsed 2 "+ mozile.util.dumpValues(this.store()));
		this._lastTextRange = range._range.duplicate();
		this._lastRange = range;
		range._range.select();
		//alert("Collapsed 3 "+ mozile.util.dumpValues(this.store()));
	} catch(e) {
		mozile.debug.debug("mozile.dom.InternetExplorerSelection.collapse", "Error: "+ mozile.dumpError(e));
	}
}

/**
 * Moves the focus of the selection to a specified point.
 * Somewhat tricky because the direction of the extension has to be accounted for.
 * @param {Node} parentNode The new focusNode.
 * @param {Integer} offset The new focusOffset.
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.extend = function (parentNode, offset) {
	var range = this.getRangeAt(0);
	var direction;

	// Compare offsets in the same node.
	if(parentNode == range.startContainer && 
		range.startOffset <= offset) {
		direction = this._LTR;
	}
	else if(parentNode == range.endContainer && 
		range.endOffset >= offset){
		direction = this._RTL;
	}

	// Compare offsets in the same container.
	// Suppose the startContainer/endContainer is a text node.
	else if(parentNode == range.startContainer.parentNode &&
	  range.startContainer.nodeType == mozile.dom.TEXT_NODE &&
	  mozile.dom.getIndex(range.startContainer) <= offset) {
	  direction = this._LTR;
	}
	else if(parentNode == range.endContainer.parentNode &&
	  range.endContainer.nodeType == mozile.dom.TEXT_NODE &&
	  mozile.dom.getIndex(range.endContainer) >= offset) {
	  direction = this._RTL;
	}

	// Use a TreeWalker to the parentNode relative to the anchorNode.
	else {
		var ancestor = mozile.dom.getCommonAncestor(parentNode, range.commonAncestorContainer)
		var treeWalker = document.createTreeWalker(ancestor, mozile.dom.NodeFilter.SHOW_ALL, null, false);
		treeWalker.currentNode = this.anchorNode;
		while(treeWalker.previousNode()) {
			if(treeWalker.currentNode == parentNode) {
				direction = this._RTL;
				break;
			}
		}
		if(!direction) direction = this._LTR;
	}
	
	// Extend the selection.
	if(direction == this._LTR) range.setEnd(parentNode, offset);
	else if(direction == this._RTL) range.setStart(parentNode, offset);
	else return;
	
	//alert("Extend "+ this._direction +"\nFocus "+ this.focusOffset +"\n"+  mozile.util.dumpValues(range.store()));
	this._direction = direction;
	this._init(range);
	this._lastTextRange = range._range.duplicate();
	this._lastRange = range;
	range._range.select();
}

/**
 * Moves the focus of the selection to the same point as the anchor.
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.collapseToStart = function () {
	var range = this.getRangeAt(0);
	range.collapse(true);
	this._init(range);
	range._range.select();
}

/**
 * Moves the anchor of the selection to the same point as the focus. The focus does not move.
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.collapseToEnd = function () {
	var range = this.getRangeAt(0);
	range.collapse();
	this._init(range);
	range._range.select();
}

/**
 * Adds all the children of the specified node to the selection.
 * @param {Node} parentNode
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.selectAllChildren = function (parentNode) {
	var range = this.getRangeAt(0);
	range.selectNodeContents(parentNode);
	this._init(range);
	range._range.select();
}

/**
 * A range object that will be added to the selection.
 * @param {Range} range The range to add.
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.addRange = function (range) {
	this._direction = this._LTR;
	this._init(range);
	range._range.select();
}

/**
 * Removes a range from the selection.
 * @param {Range} range The range to remove.
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.removeRange = function (range) {
	range.collapse();
	this._init(range);
}

/**
 * Removes all ranges from the selection.
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.removeAllRanges = function () {
	this._selection.empty();
	this._init();
}

/**
 * Deletes the selection's content from the document.
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.deleteFromDocument = function () {
	this._selection.clear();
	this._init();
}

/**
 * NOT IMPLEMENTED. Changes the selection direction from Left-to-Right to Right-to-Left.
 * @param {Boolean} langRTL True indicates Right-to-Left, false indicates Left-to-Right.
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.selectionLanguageChange = function () {}

/**
 * Returns a string currently being represented by the selection object, i.e. the 
 * currently selected text, no markup
 * @type String
 */
mozile.dom.InternetExplorerSelection.prototype.toString = function () {
	var range = this.getRangeAt(0);
	return range.toString();
}

/**
 * NOT IMPLEMENTED. Indicates if a certain node is part of the selection.
 * @param {Node} aNode
 * @param {Boolean} aPartlyContained
 * @type Boolean
 */
mozile.dom.InternetExplorerSelection.prototype.containsNode = function (aNode, aPartlyContained) {
	alert('mozile.dom.InternetExplorerSelection.containsNode() is not implemented yet');
}



/**
 * Store the details about this range in an object which can be used to restore the range.
 * @type Object
 */
mozile.dom.InternetExplorerSelection.prototype.store = mozile.dom.Selection.prototype.store;

/**
 * Takes a stored range object and creates a new range with the same properties.
 * @param {Object} state A state object from Selection.store() or Range.store().
 * @type Void
 */
mozile.dom.InternetExplorerSelection.prototype.restore = mozile.dom.Selection.prototype.restore;


/**
 * Restore the give range as the current selection (only allows for a single range).
 * @param {Range} r The range to restore.
 * @type Void
 */
mozile.dom.Selection.restoreSelection = function (r) {
	var range = new mozile.Range();
	try {
		range.setStart(r.startContainer, r.startOffset);
		range.setEnd(r.endContainer, r.endOffset);
	} catch(e) {}

	// Restore the selection.
	var s = new mozile.dom.Selection();
	s.removeAllRanges();
	s.addRange(range);
}







/**
 * Tools for parsing RelaxNG schemas and validating XML documents.
 * @type Object
 */
mozile.rng = new Object();
// JSDoc hack
mozile.rng.prototype = new mozile.Module;

/**
 * Indicates verbose debugging messages.
 * @type Boolean
 */
mozile.rng.debug = false;


/**** Common Methods ****/
/**
 * NOTE: These functions are intended to be assigned as methods to RNG objects.
 * The idea is to maximize reuse of code without an awkward hiearchy of classes.
 * They will not work on their own.
 */

/**
 * Returns the 'name' attribute of the element, stripped of whitespace.
 * @type String
 */
mozile.rng.getName = function() {
	if(this._name) return this._name;
	if(this._element.getAttribute("name")) {
		var match = this._element.getAttribute("name").match(/^\s*(.+?)\s*$/);
		if(match && match.length == 2) {
			this._name = match[1];
			return this._name;
		}
	}
	throw Error("This RNG '"+ this.getType() +"' element must have a 'name' attribute.");
}

/**
 * Gets the local part of the qualified name of the element.
 * @type String
 */
mozile.rng.getLocalName = function() {
	if(this._localName) return this._localName;
	var name = this.getName();
	if(name.indexOf(":")) this._localName = name.substring(name.indexOf(":") + 1, name.length);
	else this._localName = name;
	return this._localName;
}

/**
 * Gets the prefix part of the qualified name of the element.
 * @type String
 */
mozile.rng.getPrefix = function() {
	if(this._prefix) return this._prefix;
	var name = this.getName();
	if(name.indexOf(":")) this._prefix = name.substring(0, name.indexOf(":"));
	else this._prefix = null;
	return this._prefix;
}

/**
 * Checks to see if a RNG element type is in a given list of types.
 * @param types The list to check. Can be a string or an array of strings.
 * @param type The type to find in the list.
 * @type Boolean
 */
mozile.rng.checkType = function(types, type) {
	if(typeof(types) == "string" && types == type) return true;
	if(types.length) {
		for(var i=0; i < types.length; i++) {
			if(types[i] == type) return true;
		}
	}
	return false;
}

/**
 * Validates nodes in a sequence. 
 * As long as the validation isValid, each RNG child is checked.
 * If there is a validation.getCurrentElement(), then we mark that element as the last valid element and try to advance to the nextSiblingElement.
 * @private
 * @param {Node} node The first node to be validated.
 * @param {mozile.rng.Validation} validation The validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.validateSequence = function(node, validation) {
	if(!node) throw Error("mozile.rng.validateSequence() requires a node.");
	if(!validation) throw Error("mozile.rng.validateSequence() requires an mozile.rng.Validation object.");

	var RNGChildren = this.getChildNodes();
	if(RNGChildren.length == 0) return validation;

	for(var r=0; r < RNGChildren.length; r++) {
		if(!validation.isValid) break;

		validation = RNGChildren[r].validate(node, validation);

		if(node.nodeType == mozile.dom.ELEMENT_NODE && validation.isEmpty)
			validation.logError(this, node, "The parent element should be empty, but it contains child elements.");
		
		if(validation.getCurrentElement()) {
			node = validation.getCurrentElement();
			if(mozile.dom.getNextSiblingElement(node)) node = mozile.dom.getNextSiblingElement(node);
		}
	}
	
	return validation;
}

/**
 * Validates the set of the RNG children as many times as possible. 
 * When the validation of the set fails, the last successful validation object is returned.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.validateMany = function(node, validation) {
	if(!node) throw Error("mozile.rng.validateMany() requires a node.");
	if(!validation) throw Error("mozile.rng.validateMany() requires an mozile.rng.Validation object.");

	var result;
	validation.count = 0;
	while(true) {
		result = validation.branch();
		result = this._validateSequence(node, result);

		if(result.isValid) {
			validation.count++;
			validation.merge(result);

			if(result.getCurrentElement() && mozile.dom.getNextSiblingElement(result.getCurrentElement()) ) {
				node = mozile.dom.getNextSiblingElement(result.getCurrentElement());
				continue;
			}
		}


		if(mozile.rng.debug) validation.append(result, true);
		break;
	}
	
	return validation;
}

/**
 * Validates the set of the RNG children where the order of validation does not matter.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.validateInterleave = function(node, validation) {
	if(!node) throw Error("mozile.rng.validateInterleave() requires a node.");
	if(!validation) throw Error("mozile.rng.validateInterleave() requires an mozile.rng.Validation object.");
	
	validation.logMessage(this, node, "Validating interleave...");
	// create a new list of RNG children
	var RNGChildren = new Array();
	for(var c=0; c < this.getChildNodes().length; c++) {
		RNGChildren.push(this.getChildNodes()[c]);
	}
	
	var type;
	var length = RNGChildren.length;
	// while true:
	for(var i=0; i < length; i++) {
		// for each node try each RNG child on the list:
		for(var r=0; r < RNGChildren.length; r++) {
			var result = validation.branch();
			result = RNGChildren[r].validate(node, result);

			// if one works
			if(result.isValid) {
				// zeroOrMore and optional are special cases: check (count && count > 0)
				type = RNGChildren[r].getType();
				if( (RNGChildren[r].getType() == "optional" || 
					RNGChildren[r].getType() == "zeroOrMore") && 
					!result.count ) 
						continue;

				validation.merge(result);
				if(result.getCurrentElement() && mozile.dom.getNextSiblingElement(result.getCurrentElement()) ) {
					node = mozile.dom.getNextSiblingElement(result.getCurrentElement());
				}

				// remove the working RNG child from the list
				RNGChildren.splice(r, 1);
				break;
			}
		}
	}

	// if there are children left other than optional and zeroOrMore, fail
	if(RNGChildren.length > 0) {
		for(r=0; r < RNGChildren.length; r++) {
			if(!validation.isValid) break;
			type = RNGChildren[r].getType();
			if(type != "optional" && type != "zeroOrMore")
				validation.logError(this, node, "There were non-optional RNG children which did not match any elements.");
		}
	}
	
	if(validation.isValid) validation.logMessage(this, node, "Interleave is valid.");

	return validation;
}

/**
 * Combines this mozile.rng.Define (or mozile.rng.Start) object with another mozile.rng.Define (or mozile.rng.Start) object.
 * <p>At least one of the elements must have a 'combine' attribute.
 * If both elements have a 'combine' attribute, then the values must match.
 * The value of the 'combine' attribute can be either 'choice' or 'interleave'.
 * <p>If this object has a single child of the combineType, then it is used as a container for the new objects.
 * Otherwise a new container of choiceType is created and all children moved into it.
 * Similarly, if the given object has a single child of combineType, all of that object's children are moved into the container. 
 * Otherwise all the object's children are moved into the container.
 * <a href="http://www.oasis-open.org/committees/relax-ng/spec.html#IDA0KZR">Specification</a>
 * @param {mozile.rng.Node} target The object to combine with this object.
 * @type mozile.rng.Node
 */
mozile.rng.combine = function(target) {
	var combineType;
	var combineA = this._element.getAttribute("combine");
	var combineB = target._element.getAttribute("combine");

	if(combineA && combineB) {
		if(combineA == combineB) combineType = combineA;
		else throw Error("In order to combine RNG elements, their 'combine' attributes muct match. '"+ combineA +"' != '"+ combineB +"'");
	}
	else {
		if(combineA) combineType = combineA;
		else if(combineB) combineType = combineB; 
		else throw Error("In order to combine RNG elements, at least one must have a 'combine' attribute.");
	}
	
	if(combineType != "choice" && combineType != "interleave")
	 throw Error("RNG 'define' or 'start' elements can only have 'combine' attribute values of 'choice' or 'interleave', not '"+ combineType +"'.");
	this._element.setAttribute("combine", combineType);



	var combineObject;
	if(this.getChildNodes().length == 1 && 
		this.getChildNode(0).getType() == combineType) {
			combineObject = this.getChildNode(0);
	}
	else {
		var validation = new mozile.rng.Validation();
		combineObject = this.getSchema().parseElement(mozile.dom.createElementNS(mozile.xml.ns.rng, combineType), validation, true);
		this.appendChild(combineObject);
		while(this.getChildNodes().length > 1) {
			combineObject.appendChild(this.removeChild(this.getChildNode(0)));
		}
	}

	var targetObject = target;
	if(target.getChildNodes().length == 1 && 
		target.getChildNode(0).getType() == combineType) {
			targetObject = target.getChildNode(0);
  }
	while(targetObject.getChildNodes().length) {
		combineObject.appendChild(targetObject.removeChild(targetObject.getChildNode(0)));
	}

	return this;
}


/**
 * Represents a RelaxNG schema.
 * <p>A schema has a single "root" element, and can include zero or more document objects representing external RNG files.
 * @constructor
 * @param target Optional. An element or path to a file, which will be parsed.
 */
mozile.rng.Schema = function(target) {
	/**
	 * An associative array of document objects for this schema, each of which represents an external RNG file. Keys are filepaths, values are documents.
	 * @private
	 * @type Object
	 */
	this._documents = new Object();

	/**
	 * An associative array of all RNG objects in this schema. Keyes are types and values are arrays of objects.
	 * @private
	 * @type Object
	 */
	this._types = new Object();

	/**
	 * A multi-dimenstional associative array of all named RNG objects in this schema. Keyes are types and values are associative arrays of names. The values of the latter are arrays of objects.
	 * @private
	 * @type Object
	 */
	this._names = new Object();

	/**
	 * The unique root where validation begins. Can be an mozile.rng.Element or mozile.rng.Start node.
	 * @private
	 * @type mozile.rng.Node
	 */
	this._root = null;
	
	if(target) this.parse(target);
}
mozile.rng.Schema.prototype.constructor = mozile.rng.Schema;

/**
 * Returns "[object mozile.rng.Schema]".
 * @type String
 */
mozile.rng.Schema.prototype.toString = function() { return "[object mozile.rng.Schema]" };

/**
 * Gets the element's localName.
 * @type String
 */
mozile.rng.Schema.prototype.getType = function() {
	return "schema";
}

/**
 * Parses a RNG into a hierarchy of RNG nodes.
 * This method can handle XMl strings, file paths, and elements as input.
 * @param target Optional. A XML string, path to a file, or element to parse.
 * @type mozile.rng.Validation
 */
mozile.rng.Schema.prototype.parse = function(target) {
	var element;
	var filepath;
	if(typeof(target) == "string") {
		if(target.indexOf("<") > - 1) {
			element = mozile.xml.parse(target).documentElement;
			filepath = location.toString();
		}
		else {
			var doc = this.load(target);
			if(!doc) throw Error("Could not load schema '"+ target +"'.");
			element = doc.documentElement;
			filepath = target;
		}
	} 
	else if(target.nodeType) {
		element = target;
		filepath = location.toString();
	}
	else throw Error("Unknown target given in mozile.rng.Schema.parse().");

	if(!this.filepath) this.filepath = filepath;
	//alert("Filepath: "+ this.filepath);
	
	var validation = new mozile.rng.Validation();
	validation.logMessage(this, null, "Starting to parse schema");
	var result = this.parseElement(element, validation);
	if(!result) {
		validation.logError(this, null, "An invalid result was returned from the parsing operation: "+result);
		return validation;
	}
	var root = result;
	if(root.getType() == "element" || root.getType() == "grammar") {
		this._root = root;
	}
	else {
		validation.logError(this, null, "Schema has no root element.");
		return validation;
	}
	
	
	// Parse includes.
	var includes = this.getNodes("include");
	if(!includes) return validation;
	
	for(var i=0; i < includes.length; i++) {
		validation = includes[i].include(validation);
	}	
	return validation;
}

/**
 * Parses an RNG element into an mozile.rng.Node. Acts recursively on the node's children.
 * @param {Element} element The RNG element to parse.
 * @param {mozile.rng.Validation} validation The validation object for this validation operation.
 * @param {Boolean} omitValidation Optional. When true the element will not be self-validated.
 * @type mozile.rng.Node
 */
mozile.rng.Schema.prototype.parseElement = function(element, validation, omitValidation) {
	var rngNode = this._createRNGNode(element, validation);
	if(!rngNode) {
		//validation.logMessage(null, element, "Element '"+ element.nodeName  +"' is not an RNG element.");
		return null;
	}
	else validation.logMessage(rngNode, element, "Adding RNG '"+ rngNode.getType()  +"' node.");

	if(omitValidation != true) {
		validation = rngNode.selfValidate(validation);
		if(!validation.isValid) return {node: null, validation: validation};
	}

	if(element.hasChildNodes()) {
		var child;
		for(var c=0; c < element.childNodes.length; c++) {
			child = this.parseElement(element.childNodes[c], validation);
			if(child) rngNode.appendChild(child);
		}
	}
	
	return rngNode;
}

/**
 * Creates the right kind of RNG object for a given RNG element.
 * Adds the object to the list of descendants.
 * @private
 * @param {Element} element The RNG element to parse.
 * @type mozile.rng.Node
 */
mozile.rng.Schema.prototype._createRNGNode = function(element, validation) {
	if(!element.nodeType || element.nodeType != mozile.dom.ELEMENT_NODE) return null;
	if(element.namespaceURI != mozile.xml.ns.rng) return null;
	
	var result;
	var type = mozile.dom.getLocalName(element);
	switch(type) {
		case "grammar":     result = new mozile.rng.Grammar(element, this); break;
		case "start":       result = new mozile.rng.Start(element, this); break;
		case "element":     result = new mozile.rng.Element(element, this); break;
		case "attribute":   result = new mozile.rng.Attribute(element, this); break;
		case "empty":       result = new mozile.rng.Empty(element, this); break;
		case "text":        result = new mozile.rng.Text(element, this); break;
		case "group":       result = new mozile.rng.Group(element, this); break;
		case "optional":    result = new mozile.rng.Optional(element, this); break;
		case "oneOrMore":   result = new mozile.rng.OneOrMore(element, this); break;
		case "zeroOrMore":  result = new mozile.rng.ZeroOrMore(element, this); break;
		case "choice":      result = new mozile.rng.Choice(element, this); break;
		case "define":      result = new mozile.rng.Define(element, this); break;
		case "ref":         result = new mozile.rng.Ref(element, this); break;
		case "include":     result = new mozile.rng.Include(element, this); break;
		case "data":        result = new mozile.rng.Data(element, this); break;
		case "param":       result = new mozile.rng.Param(element, this); break;
		case "value":       result = new mozile.rng.Value(element, this); break;
		case "interleave":  result = new mozile.rng.Interleave(element, this); break;
		case "div":         result = new mozile.rng.Div(element, this); break;
		//case "mixed":       result = new mozile.rng.Mixed(element, this); break;
		default: 
			validation.logError(this, null, "Method mozile.rng.Schema._createRNGNode() found an unknown element '"+ element.nodeName +"' with the RNG namespace."); 
			return null;
	}

	if(result) {
		this._indexNode(result);
		return result;
	}
	else return null;

}

/**
 * Stores the RNG node in the appropriate spot inside the _types and _names objects.
 * @private
 * @param {mozile.rng.Node} node The node to index
 * @type Void
 */
mozile.rng.Schema.prototype._indexNode = function(node) {
	// Index by type
	var type = node.getType();
	if(!this._types[type]) this._types[type] = new Array();
	this._types[type].push(node);
	
	// Index by name
	if(node.getName) {
		var name = node.getName();
		if(!this._names[type]) this._names[type] = new Object();
		if(!this._names[type][name]) this._names[type][name] = new Array();
		this._names[type][name].push(node);
	}
}

/**
 * Returns an array of nodes of the given type.
 * @param {String} type A node type, such as "element" or "zeroOrMore".
 * @type Array
 */
mozile.rng.Schema.prototype.getNodes = function(type, name) {
	if(name) {
		if(this._names[type] && this._names[type][name]) 
			return this._names[type][name];
		else return new Array();
	}
	else {
		if(this._types[type]) return this._types[type];
		else return new Array();
	}
}


/**
 * Loads an external RNG XML file, adds it to the documents array, and returns it. 
 * If the file has already been loaded a stored copy will be used unless forceLoad is true. 
 * @param {String} filepath The path to the RNG XML file.
 * @param {Boolean} forceLoad Optional. When true the method will not check for a cached copy of the document.
 * @type Document
 */
mozile.rng.Schema.prototype.load = function(filepath, forceLoad) {
	if(!forceLoad && this._documents[filepath]) return this._documents[filepath];

	var xmlDoc = mozile.xml.load(filepath);
	if(xmlDoc) this._documents[filepath] = xmlDoc;

	return xmlDoc;
}

/**
 * Validates an element against the schema.
 * Before validating, all caches are reset.
 * @param {Element} element Optional. The element to validate. If not is given, th e documentElement is used.
 * @type Boolean
 */
mozile.rng.Schema.prototype.validate = function(element) {
	if(!element) element = document.documentElement;
	if(!this._root) return false;

	var validation = new mozile.rng.Validation();
	this._root.resetAll(); // clear all cached values
	this._root.validate(element, validation);

	return validation;
}



/**
 * Represents the result of validating an element.
 * <p>This object is used to pass inforamtion down the RNG element hierarchy to an element which is validating its children. It stores information about the validity of the child, as well as: the last validated node, the attributes which have been validated, and whether text is allowed in this element.
 * @constructor
 */
mozile.rng.Validation = function() {
	/*
	 * A Boolean which indicates whether the validation succeeded or failed.
	 * @type Boolean
	 */
	this.isValid = true;

	/*
	 * A Boolean which indicates whether the element should be empty.
	 * @type Boolean
	 */
	this.isEmpty = false;

	/*
	 * A Boolean which indicates whether non-whitespace text nodes are allowed as children of the element.
	 * @type Boolean
	 */
	this.allowText = false;


	/*
	 * The element currently being validated.
	 * @private
	 * @type Boolean
	 */
	this._currentElement = null;

	/*
	 * The oparent of the element currently being validated.
	 * @private
	 * @type Boolean
	 */
	this._currentParent = null;

	/*
	 * The last element to be successfully validated.
	 * @private
	 * @type Boolean
	 */
	this._lastValidElement = null;

	/*
	 * An array containing the names of validated attributes.
	 * @private
	 * @type Array
	 */
	this._validAttributes = new Array();

	/*
	 * An array containing all the messages that have been logged.
	 * @private
	 * @type Array
	 */
	this._messages = new Array();

}

/**
 * Returns "[object mozile.rng.Validation]".
 * @type String
 */
mozile.rng.Validation.prototype.toString = function() { return "[object mozile.rng.Validation]" };

/**
 * Gets the object's type.
 * @type String
 */
mozile.rng.Validation.prototype.getType = function() {
	return "validation";
}

/**
 * Gets the last valid element.
 * @type Element
 */
mozile.rng.Validation.prototype.getCurrentElement = function() { 
	return this._currentElement;
}

/**
 * Sets the element to be validated and marks the last element as validated.
 * @type Element
 */
mozile.rng.Validation.prototype.setCurrentElement = function(element) {
	if(element.nodeType == mozile.dom.ELEMENT_NODE) {
		if(element == this._lastValidElement) {
			this.logError(this, element, "An element has been marked as valid twice, which indicated an error");
		}
		else {
			this._lastValidElement = this._currentElement;
			this._currentElement = element;
		}
	}
	return element;
}

/**
 * Gets the current parent element.
 * @type Element
 */
mozile.rng.Validation.prototype.getCurrentParent = function() { 
	return this._currentParent;
}

/**
 * Sets the current parent element.
 * @type Element
 */
mozile.rng.Validation.prototype.setCurrentParent = function(element) {
	this._currentParent = element;
	return element;
}


/**
 * Adds an attribute to the list of valid attributes.
 * @param {Attr} attr Th attribute node to be added.
 * @type Attr
 */
mozile.rng.Validation.prototype.addAttribute = function(attr) { 
	this._validAttributes.push(attr);
	return attr;
}

/**
 * Determines whether an attribute is on the validAttributes list.
 * @param {Attr} attr The attribute to check.
 * @type Boolean
 */
mozile.rng.Validation.prototype.isAttributeValid = function(attr) {
	for(var a=0; a < this._validAttributes.length; a++) {
		//alert(this._validAttributes[a].nodeName +" = "+ attr.nodeName +" ? "+ (this._validAttributes[a] == attr));
		// Old versions of Mozilla mess up attribute identity checks.
		if(mozile.browser.isMozilla && mozile.browser.mozillaVersion < 1.8) {
			if(this._validAttributes[a].nodeName == attr.nodeName) return true;
		}
		if(this._validAttributes[a] == attr) return true;
	}
	return false;
}

/**
 * Logs a message object to the messae array.
 * @param {mozile.rng.Node} RNG node The node which is logging the message.
 * @param {Node} node The current node being validated.
 * @param {String} message The message to log.
 * @param {String} type Optional. The message type.
 * @type Void
 */
mozile.rng.Validation.prototype.logMessage = function(rngNode, node, message, type) {
	if(!type) type = "Message";
	this._messages.push({"type": type, "rngNode": rngNode, "node": node, "message": message});
	//if(errorsCheck.checked) {
	//	var name = rngNode.getType();
	//	if(rngNode.getName) name += ":"+ rngNode.getName();
	//	alert(name +" "+ message);
	//}
}

/**
 * Logs a parsing error message object to the messae array, and sets isValid to false.
 * A parsing error indicates that the XML failed validation.
 * @param {mozile.rng.Node} RNG node The node which is logging the message.
 * @param {Node} node The current node being validated.
 * @param {String} message The message to log.
 * @type Void
 */
mozile.rng.Validation.prototype.logError = function(rngNode, node, message) { 
	this.logMessage(rngNode, node, message, "Error");
	this.isValid = false;
}

/**
 * Returns the first error message.
 * @type Object
 */
mozile.rng.Validation.prototype.getFirstError = function() {
	for(var m=0; m < this._messages.length; m++) {
		if(this._messages[m].type != "Message") return this._messages[m];
	}
	return null;
}

/**
 * Creates a new Validation object, and copies some of the properties of the current object to the new object.
 * @type mozile.rng.Validation
 */
mozile.rng.Validation.prototype.branch = function() {
	var validation = new mozile.rng.Validation();
	validation._currentParent = this._currentParent;
	return validation;
}

/**
 * Appends the messages from the given mozile.rng.Validation to this object's message list. If the given object is not valid, this object is set to isValid = false.
 * @param {mozile.rng.Validation} validation The node which is logging the message.
 * @param {Boolean} status Optional. THis parameter overrides the object's "isValid" property.
 * @type Void
 */
mozile.rng.Validation.prototype.append = function(validation, status) {
	this._messages = this._messages.concat(validation._messages);
	if(status) this.isValid = status;
	else if(!validation.isValid) this.isValid = false;
}

/**
 * Appends messages and then merges validAttributes lists from the given mozile.rng.Validation object with this object. 
 * If the given object allows text, this allowText is set to true. 
 * The currentElement and lastValidElement are set to match the given object.
 * @param {mozile.rng.Validation} validation The node which is logging the message.
 * @type Void
 */
mozile.rng.Validation.prototype.merge = function(validation) {
	this.append(validation);
	this._validAttributes = this._validAttributes.concat(validation._validAttributes);
	if(validation.isEmpty) this.isEmpty = true;
	if(validation.allowText) this.allowText = true;
	if(validation._currentElement) this._currentElement = validation._currentElement;
	if(validation._lastValidElement) this._lastValidElement = validation._lastValidElement;
}

/**
 * Prints a report of the validation operation.
 * @param {Boolean} errorsOnly Optional. When true, only errors will be reported.
 * @type String
 */
mozile.rng.Validation.prototype.report = function(errorsOnly) {
	mozile.require("mozile.util");
	mozile.require("mozile.xpath");

	var messages = new Array();
	var maxType = 8;
	var maxName = 0;
	var maxLocation = 0;
	var msg;
	var location;
	var rngName
	var lastNode;
	var lastLocation;
	for(var m=0; m < this._messages.length; m++) {
		msg = this._messages[m];
		if(errorsOnly && msg.type == "Message") continue;


		rngName = "validation";
		if(msg.rngNode) {
			if(msg.rngNode.getType()) rngName = mozile.util.pad(msg.rngNode.getType(), 11, true);
			//var rngName = msg.rngNode._element.getXPath();
			if(msg.rngNode.getName) rngName = rngName +" "+ msg.rngNode.getName();
		}

		
		location = "";
		if(msg.node) {
			if(msg.node == lastNode) location = lastLocation;
			else {
				location = mozile.xpath.getXPath(msg.node);
				location = location.replace(/xmlns:/g, "");
				//var depth = 0;
				//var parent = msg.node.parentNode;
				//while(parent) {
				//	depth++;
				//	parent = parent.parentNode;
				//}
				//var space = "";
				//for(var d=0; d < depth; d++) space += " ";
				//location = space + msg.node.nodeName;
				lastNode = msg.node;
				lastLocation = location;
				if(location.length > maxLocation) maxLocation = location.length;
			}
		}
		
		//if(msg.type.length > maxType) maxType = msg.type.length;
		if(rngName.length > maxName) maxName = rngName.length;
		
		messages.push([msg.type, rngName, location, msg.message]);
	}

	var output = new Array();
	//if(this.isValid) output.push("VALIDATION SUCCESSFUL");
	//else output.push("VALIDATION FAILED");

	var maxNum = Math.ceil( Math.log(messages.length+1) / Math.log(10) ) + 1;
	for(m=0; m < messages.length; m++) {
		msg = messages[m];
		output.push( mozile.util.pad((m+1) +".", maxNum) +" "+ mozile.util.pad(msg[0], maxType) +" "+ mozile.util.pad(msg[1], maxName) +"  "+ mozile.util.pad(msg[2], maxLocation) +"  "+ msg[3] ); 
	}

	return output.join(mozile.linesep);
}


/**
 * Represents any element from a RelaxNG grammar.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Node = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}

/**
 * An array of all cached properties. They can be cleared with the reset() method.
 * @private
 * @type Array
 */
mozile.rng.Node.prototype._cached = [this._mustHaveCache, this._mayHaveCache, this._descendantElementsCache, this._type];

/**
 * Returns the identifier "[object mozile.rng.Node]".
 * Works for any RNG node by capitalizing the type.
 * @type String
 */
mozile.rng.Node.prototype.toString = function() {
	var type = this.getType();
	type = type.charAt(0).toUpperCase() + type.substring(1);
	var name = "";
	if(this.getName) name = " '"+ this.getName() +"'";
	return "[object mozile.rng."+ type + name +"]";
}

/**
 * Gets the element's localName.
 * @type String
 */
mozile.rng.Node.prototype.getType = function() {
	if(!this._type) this._type = mozile.dom.getLocalName(this._element);
	return this._type;
}

/**
 * Gets the mozile.rng.Node's owner mozile.rng.Schema.
 * @type mozile.rng.Schema
 */
mozile.rng.Node.prototype.getSchema = function() {
	return this._schema;
}

/**
 * Gets the mozile.rng.Node's parent mozile.rng.Grammar.
 * @type mozile.rng.Grammar
 */
mozile.rng.Node.prototype.getGrammar = function() {
	var node = this;
	while(node) {
		if(node.getType() == "grammar") return node;
		node = node.getParentNode();
	}
	return null;
}

/**
 * Gets the element's parentNode.
 * @type mozile.rng.Node
 */
mozile.rng.Node.prototype.getParentNode = function() {
	if(this._parentNode) return this._parentNode;
	return null;
}

/**
 * Gets the element's first ancestor which is an mozile.rng.Element.
 * @type mozile.rng.Element
 */
mozile.rng.Node.prototype.getParentElement = function() {
	var parent = this.getParentNode();
	while(parent) {
		if(parent.getType() == "element") return parent;
		parent = parent.getParentNode();
	}
	return null;
}

/**
 * Gets the element's parentNode.
 * @type String
 */
mozile.rng.Node.prototype.getNextSibling = function() {
	var parent = this.getParentNode();
	if(!parent) return null;
	var siblings = parent.getChildNodes();
	for(var r=0; r < siblings.length; r++) {
		if(siblings[r] == this) {
			if(siblings[r+1]) return siblings[r+1];
		}
	}
	return null;
}

/**
 * Gets an array of the node's child nodes.
 * @type Array
 */
mozile.rng.Node.prototype.getChildNodes = function() {
	if(!this._childNodes) this._childNodes = new Array();
	return this._childNodes;
}

/**
 * Gets the child nodes at the given index.
 * @param {Integer} index
 * @type mozile.rng.Node
 */
mozile.rng.Node.prototype.getChildNode = function(index) {
	var children = this.getChildNodes();
	if(children && children.length && 
		children.length > 0 && index < children.length) 
			return children[index];
	else return null;
}

/**
 * Appends the child to this object's list of child nodes and sets its parentNode.
 * @param {mozile.rng.Node} child The child to be appended.
 * @type mozile.rng.Node
 */
mozile.rng.Node.prototype.appendChild = function(child) {
	if(child) {
		this.getChildNodes().push(child);
		child._parentNode = this;
	}
	return child;
}

/**
 * Removes the child from this object's list of child node, and unsets its parentNode.
 * @param {mozile.rng.Node} child The child to be removed.
 * @type mozile.rng.Node
 */
mozile.rng.Node.prototype.removeChild = function(child) {
	var children = new Array();
	for(var c=0; c < this.getChildNodes().length; c++) {
		if(this.getChildNodes()[c] == child) continue;
		children.push(this.getChildNodes()[c]);
	}
	if(children.length == this.getChildNodes().length)
		throw Error("mozile.rng.Node.removeChild(): The given node is not child of this node.");
	else this._childNodes = children;
	child._parentNode = null;
	return child;
}

/**
 * Gets an array of all the elements which descend from this node.
 * @param types The RNG element type(s) of search for. Can be a string or an array of strings. E.g. "element".
 * @param {Boolean} deep Optional. If the current node is an element and "deep" is true, then its children are included.
 * @type Array
 */
mozile.rng.Node.prototype.getDescendants = function(types, deep) {
	//if(this._descendantElementsCache) return this._descendantElementsCache;
	if(deep !== false) deep = true;
	
	var descendants = new Array();
	if(!deep && mozile.rng.checkType(types, this.getType())) {
		descendants.push(this);
	}
	else {
		var result;
		for(var c=0; c < this.getChildNodes().length; c++) {
			result = this.getChildNode(c).getDescendants(types, false);
			descendants = descendants.concat(result);
		}
	}
	//alert(this.getType() +" "+ this._name +" "+ deep +"\n"+ descendants);
	
	//this._descendantElementsCache = descendants;
	return descendants;
}

/**
 * True if the current element must contain at least one child of the given type.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Node.prototype.mustContain = function(type) {
	if(!this._mustHaveCache) this._mustHaveCache = new Object();
	if(this._mustHaveCache[type]) return this._mustHaveCache[type];

	var result = false;
	for(var c=0; c < this.getChildNodes().length; c++) {
		if(this.getChildNodes()[c].mustHave(type)) {
			result = true;
			break;
		}
	}

	this._mustHaveCache[type] = result;
	return result;
}

/**
 * True if the current element may contain at least one child of the given type.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Node.prototype.mayContain = function(type) {
	if(!this._mayHaveCache) this._mayHaveCache = new Object();
	if(this._mayHaveCache[type]) return this._mayHaveCache[type];

	var result = false;
	for(var c=0; c < this.getChildNodes().length; c++) {
		if(this.getChildNodes()[c].mayHave(type)) {
			result = true;
			break;
		}
	}

	this._mayHaveCache[type] = result;
	return result;
}

/**
 * True if the current element is of the given type or must contain at least one child of the given type.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Node.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	else return this.mustContain(type);
}

/**
 * True if the current element is of the given type or may contain at least one child of the given type.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Node.prototype.mayHave = function(type) {
	if(this.getType() == type) return true;
	else return this.mayContain(type);
}

/**
 * Resets all cached data in this node.
 * @type Void
 */
mozile.rng.Node.prototype.reset = function() {
	for(var i=0; i < this._cached.length; i++) {
		this._cached[i] = undefined;
	}
}

/**
 * Resets all cached data for this node and all children.
 * @type Void
 */
mozile.rng.Node.prototype.resetAll = function() {
	this.reset();
	for(var c=0; c < this.getChildNodes().length; c++) {
		this.getChildNodes()[c].reset;
	}
}

/**
 * Validates this RNG object.
 * @type mozile.rng.Validation
 */
mozile.rng.Node.prototype.selfValidate = function(validation) {
	if(!mozile.dom.getFirstChildElement(this._element))
		validation.logError(this, this._element, "This RNG element must have at least one child element.");
	return validation;
}

/**
 * Validates this RNG object and its children.
 * @type mozile.rng.Validation
 */
mozile.rng.Node.prototype.selfValidateAll = function(validation) {
	validation = this.selfValidate(validation);
	validation.logMessage(this, null, "Self validated.");

	for(var r=0; r < this.getChildNodes().length; r++) {
		if(!validation.isValid) break;
		validation = this.getChildNodes()[r].selfValidateAll(validation);
	}
		
	return validation;
}

/**
 * Validates the node. This is an abstract method, meant to be overriden by child classes.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Node.prototype.validate = function(node, validation) {
	validation.logError(this, node, "The validate() method is not defined yet for "+ this +".");
	return validation;
}


/**
 * Represents an RNG "grammar" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Grammar = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;

	/**
	 * A grammar can have a single start element.
	 * @private
	 * @type mozile.rng.Start
	 */
	this._start = null;

	/**
	 * A grammar can zero or more definitions. They are stored as an associative array with names as keys and mozile.rng.Define objects as values.
	 * @private
	 * @type Object
	 */
	this._definitions = new Object;
}
mozile.rng.Grammar.prototype = new mozile.rng.Node;
mozile.rng.Grammar.prototype.constructor = mozile.rng.Grammar;


/**
 * Gets the grammar's first ancestor which is an mozile.rng.Grammar.
 * @type mozile.rng.Grammar
 */
mozile.rng.Grammar.prototype.getParentGrammar = function() {
	var parent = this.getParentNode();
	while(parent) {
		if(parent.getType() == "grammar") return parent;
		parent = parent.getParentNode();
	}
	return null;
}

/**
 * Appends the given child to this object's list of children.
 * Also watches for a mozile.rng.Start object for this grammar.
 * @param {mozile.rng.Node} child The child to be appended.
 * @type mozile.rng.Node
 */
mozile.rng.Grammar.prototype.appendChild = function(child) {
	if(child && child.getType) {
		switch(child.getType()) {
			case "start":
				return this.setStart(child);
				break;
			case "define":
				return this.addDefinition(child);
				break;
			case "include":
			case "div":
				break;
			default:
				throw Error("RNG grammar element cannot have a child of type '"+ child.getType() +"'.");
				break;
		}
		this.getChildNodes().push(child);
		child._parentNode = this;
		
		//if(child.getType() == "include") child.include();
	}
	return child;
}

/**
 * Gets the start element. A grammar may have a single start element.
 * @type mozile.rng.Start
 */
mozile.rng.Grammar.prototype.getStart = function() {
	if(this._start) return this._start;
	else return null;
}

/**
 * Sets the start element. A grammar may have a single start element. If there is a previous start element, it is combined.
 * @type mozile.rng.Start
 */
mozile.rng.Grammar.prototype.setStart = function(start) {
	if(start.getType && start.getType() == "start") {
		if(this._start) this._start.combine(start);
		else {
			this._start = start;
			this.getChildNodes().push(start);
			start._parentNode = this;
		}
		return start;
	}
	else return null;
}

/**
 * Given a name, returns a definition object.
 * @param {String} name The name of the definition.
 * @type mozile.rng.Define
 */
mozile.rng.Grammar.prototype.getDefinition = function(name) {
	if(this._definitions[name]) return this._definitions[name];
	else return null;
}

/**
 * Adds the definition to the associative array of definitions.
 * @param {mozile.rng.Define} definition The definition to be added.
 * @type mozile.rng.Define
 */
mozile.rng.Grammar.prototype.addDefinition = function(definition) {
	if(this._definitions[definition.getName()]) {
		definition = this._definitions[definition.getName()].combine(definition);
	}
	else {
		this._definitions[definition.getName()] = definition;
		this.getChildNodes().push(definition);
		definition._parentNode = this;
	}
	return definition;
}

/**
 * True if the type is "grammar". Otherwise calls the mozile.rng.Start object's mustHave() method.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Grammar.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;

	var result = false;
	var start = this.getStart();
	if(start) result = start.mustHave(type);
	else throw Error("mozile.rng.Grammar.mustHave() requires getStart() to return a mozile.rng.Start object.");

	return result;
}

/**
 * True if the type is "grammar". Otherwise calls the mozile.rng.Start object's mayHave() method.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Grammar.prototype.mayHave = function(type) {
	if(this.getType() == type) return true;

	var result = false;
	var start = this.getStart();
	if(start) result = start.mayHave(type);
	else throw Error("mozile.rng.Grammar.mayHave() requires getStart() to return a mozile.rng.Start object.");

	return result;
}

/**
 * Validates the node by calling the grammar's start node's validation method.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation Optional. The validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Grammar.prototype.validate = function(node, validation) {
	var start = this.getStart();
	if(start) validation = start.validate(node, validation);
	else validation.logError(this, node, "RNG grammar element must have a start element in order to validate a node, but it does not.");
	return validation;
}


/**
 * Represents an RNG "start" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Start = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Start.prototype = new mozile.rng.Node;
mozile.rng.Start.prototype.constructor = mozile.rng.Start;

/**
 * Combines this mozile.rng.Start object with another mozile.rng.Start object.
 * @type mozile.rng.Start
 */
mozile.rng.Start.prototype.combine = mozile.rng.combine;

/**
 * Validates this RNG object. Start elements must bust be children of a grammar and have child elements. If they have a "combine" attribute, the value must be "choice" or "interleave".
 * @type mozile.rng.Validation
 */
mozile.rng.Start.prototype.selfValidate = function(validation) {
	if(!this._element.parentNode || mozile.dom.getLocalName(this._element.parentNode) != "grammar")
		validation.logError(this, this._element, "This RNG element must be the child of a grammar element.");

	if(!mozile.dom.getFirstChildElement(this._element))
		validation.logError(this, this._element, "This RNG element must have at least one child element.");
	
	var combine = this._element.getAttribute("combine");
	if(combine) {
		if(combine != "choice" && combine != "interleave")
			validation.logError(this, this._element, "This RNG 'start' element has an invalid 'combine' attribute value of '"+ combine +"'.");
	}

	return validation;
}


/**
 * Validates the node by calling the validation method of the first RNG child object.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation Optional. The validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Start.prototype.validate = function(node, validation) {
	var RNGChild = this.getChildNode(0);
	if(RNGChild) validation = RNGChild.validate(node, validation);
	return validation;
}



/**
 * Represents an RNG "element" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Element = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Element.prototype = new mozile.rng.Node;
mozile.rng.Element.prototype.constructor = mozile.rng.Element;


/**
 * An array of all cached properties. They can be cleared with the reset() method.
 * @private
 * @type Array
 */
mozile.rng.Element.prototype._cached = [this._mustHaveCache, this._mayHaveCache, this._descendantElementsCache, this._type, this._name, this._localName, this._prefix, this._namespace];

/**
 * Gets the full name of the element. Strips leading and trailing whitespace.
 * @type String
 */
mozile.rng.Element.prototype.getName = mozile.rng.getName;

/**
 * Gets the local part of the qualified name of the element.
 * @type String
 */
mozile.rng.Element.prototype.getLocalName = mozile.rng.getLocalName;

/**
 * Gets the prefix part of the qualified name of the element.
 * @type String
 */
mozile.rng.Element.prototype.getPrefix = mozile.rng.getPrefix;

/**
 * Gets the namespace for the element.
 * If the element has an "ns" attribute, that is returned.
 * If the element's name has a prefix, the namespace for that prefix is looked up and returned.
 * The element's ancestor elements are checked for an "ns" attribute.
 * If no namespace is found, null is returned.
 * <a href="http://www.oasis-open.org/committees/relax-ng/spec.html#IDAEBZR">Specification</a>
 * @type String
 */
mozile.rng.Element.prototype.getNamespace = function() {
	if(this._namespace) return this._namespace;
	if(this._element.getAttribute("ns")) this._namespace = this._element.getAttribute("ns");
	else if(this.getPrefix()) this._namespace = mozile.dom.lookupNamespaceURI(this._element, this.getPrefix());
	else {
		var parent = this.getParentNode();
		while(parent && parent._element) {
			if(parent._element.getAttribute("ns")) {
				this._namespace = parent._element.getAttribute("ns");
				break;
			}
			parent = parent.getParentNode();
		}
	}
	if(!this._namespace) this._namespace = null;
	return this._namespace;
}

/**
 * True if this is an element. False otherwise since the element is the basic unit of validation.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Element.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * True if this is an element. False otherwise since the element is the basic unit of validation.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Element.prototype.mayHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * Validates the node.
 * Operates in stages, first checking that the node itself is valid, then checking the contents of the node.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation Optional. The validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Element.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Element.validate() requires a node.");

	if(!validation) validation = new mozile.rng.Validation();

	validation = this._validateElement(node, validation);
	if(!validation.isValid) return validation;

	validation.logMessage(this, node, "Validating element...");
	var result = new mozile.rng.Validation();
	result.setCurrentParent(node);


	result = this._validateChildElements(node, result);
	if(result.isValid) result = this._validateText(node, result);
	if(result.isValid) result = this._validateAttributes(node, result);

	if(result.isValid) result.logMessage(this, node, "Element is valid.");
	else result.logError(this, node, "Element is not valid.");

	validation.append(result);
	validation.setCurrentElement(node);

	return validation;
}

/**
 * Validates the node type, localName, and namespace for the given node, and makes sure that this mozile.rng.Element has child nodes.
 * @private
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Element.prototype._validateElement = function(node, validation) {
	if(!node) throw Error("mozile.rng.Element._validateElement() requires a node.");
	if(!validation) throw Error("mozile.rng.Element._validateElement() requires an mozile.rng.Validation object.");

	if(node.nodeType != mozile.dom.ELEMENT_NODE) validation.logError(this, node, "Not an element.");
	if(!validation.isValid) return validation;

	if(mozile.dom.getLocalName(node) != this.getLocalName()) validation.logError(this, node, "Names do not match. '"+ mozile.dom.getLocalName(node) +"' != '"+ this.getLocalName() +"'");
	if(!validation.isValid) return validation;

	var ns = this.getNamespace();
	if(ns) {
		if(node.namespaceURI != ns) validation.logError(this, node, "Namespaces do not match. '"+ node.namespaceURI +"' != '"+ ns +"'");
	}
	else {
		if(node.namespaceURI) validation.logError(this, node, "This element has the namespace '"+ node.namespaceURI +"'but the RNG element has no namespace.");
	}
	if(!validation.isValid) return validation;

	return validation;
}

/**
 * Validates nodes in a sequence. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.Element.prototype._validateSequence = mozile.rng.validateSequence;

/**
 * Validates the child elements of this node.
 * If there are no children, each RNG child is checked to see if this is allowed. 
 * An attribute node or a text node will be used as a child, if available.
 * If there are no children at all, the method ensures that the RNG element has a single child element and it is either a "text" or an "empty" element.
 * If there are any children, each child is checked in sequence. While this happens, information about text and attributes will be added to the mozile.rng.Validation object. For that reason, this method must be run before _validateText and _validateAttributes.
 * @private
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Element.prototype._validateChildElements = function(node, validation) {
	if(!node) throw Error("mozile.rng.Element._validateChildElements() requires a node.");
	if(!validation) throw Error("mozile.rng.Element._validateChildElements() requires an mozile.rng.Validation object.");

	var RNGChildren = this.getChildNodes();
	var child = mozile.dom.getFirstChildElement(node);
	var r=0;
	
	// If there are no child elements, make sure that's acceptable.
	if(!child) {
		for(r=0; r < RNGChildren.length; r++) {
			if(RNGChildren[r].mustHave("element")) {
				validation.logError(this, node, "This element must have child elements, but it does not.");
				return validation;
			}
		}
		if(node.firstChild) validation.logMessage(this, node, "Element has no child elements.");
	}
	
	
	if(!child) {
		if(node.attributes.length > 0) child = node.attributes[0];
		
		// If there are no attributes, make sure that's acceptable.
		else {
			for(r=0; r < RNGChildren.length; r++) {
				if(RNGChildren[r].mustHave("attribute")) {
					validation.logError(this, node, "This element must have attributes, but it does not.");
					return validation;
				}
			}
			//validation.logMessage(this, node, "Element has no attributes.");
		}
	}
	
	if(!child && node.firstChild) {
		child = node.firstChild;
	}

	if(!child) validation.logMessage(this, node, "Element has no child nodes or attributes.");

	// Validate each child element in sequence.
	else {
		validation = this._validateSequence(child, validation);

		if(validation.isValid) {
			var overflow;
			if(mozile.dom.getFirstChildElement(node) && !validation.getCurrentElement())
				overflow = mozile.dom.getFirstChildElement(node);
			if(validation.getCurrentElement() && mozile.dom.getNextSiblingElement(validation.getCurrentElement()) )
				overflow = mozile.dom.getNextSiblingElement(validation.getCurrentElement());
			if(overflow) {
				validation.logError(this, node, "There are elements which are not matched by any RNG rules: '"+ overflow.nodeName +"'.");
		  }
		}

		if(mozile.dom.getFirstChildElement(node) && validation.isValid) validation.logMessage(this, node, "All child elements are valid.");
	}

	return validation;
}

/**
 * Validates the child text nodes for this node.
 * If there are any non-whitespace text nodes, then isEmpty must be false and allowText must be true, otherwise the element is invalid.
 * @private
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Element.prototype._validateText = function(node, validation) {
	if(!node) throw Error("mozile.rng.Element._validateText() requires a node.");
	if(!validation) throw Error("mozile.rng.Element._validateText() requires an mozile.rng.Validation object.");

	var child;
	for(var r=0; r < node.childNodes.length; r++) {
		child = node.childNodes[r];
		if(child.nodeType != mozile.dom.TEXT_NODE) continue;
		if(mozile.dom.isWhitespace(child)) continue;
		if(validation.isEmpty) validation.logError(this, node, "This element contains text, but it is supposed to be empty.");
		if(!validation.allowText) validation.logError(this, node, "This element contains text but that is not allowed.");
		//else validation.logMessage(this, node, "Text is allowed in this element.");
		if(validation.isValid) {
			validation.logMessage(this, node, "Element has text content.");
			break;
		}
	}
	return validation;
}

/**
 * Validates the attributes for this node.
 * This method expects the validation object to contain a list of valid attributes. If there are any attributes which are not on that list (and are not ignored) then validation fails.
 * @private
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Element.prototype._validateAttributes = function(node, validation) {
	var attr;
	for(var r=0; r < node.attributes.length; r++) {
		attr = node.attributes[r];
		if(mozile.dom.isIgnored(attr)) continue;
		if(validation.isEmpty) validation.logError(this, node, "This element contains attributes, but it is supposed to be empty.");
		if(validation.isAttributeValid(attr)) continue;
		validation.logError(this, node, "This element contains an invalid attribute: "+ attr.nodeName);
	}
	if(validation.isValid && node.attributes.length > 0) validation.logMessage(this, node, "All attributes are valid.");
	return validation;
}




/**
 * Represents an RNG "attribute" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Attribute = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Attribute.prototype = new mozile.rng.Node;
mozile.rng.Attribute.prototype.constructor = mozile.rng.Attribute;


/**
 * An array of all cached properties. They can be cleared with the reset() method.
 * @private
 * @type Array
 */
mozile.rng.Attribute.prototype._cached = [this._mustHaveCache, this._mayHaveCache, this._descendantElementsCache, this._type, this._name, this._localName, this._prefix, this._namespace];

/**
 * Gets the name of the element. Strips leading and trailing whitespace.
 * @type String
 */
mozile.rng.Attribute.prototype.getName = mozile.rng.getName;

/**
 * Gets the local part of the qualified name of the element.
 * @type String
 */
mozile.rng.Attribute.prototype.getLocalName = mozile.rng.getLocalName;

/**
 * Gets the prefix part of the qualified name of the element.
 * @type String
 */
mozile.rng.Attribute.prototype.getPrefix = mozile.rng.getPrefix;

/**
 * Gets the namespace for the element.
 * If the element has an "ns" attribute, that is returned.
 * If the element's name has a prefix, the namespace for that prefix is looked up and returned. The special case where the prefix is "xml" is ignored.
 * If no namespace is found, null is returned.
 * <a href="http://www.oasis-open.org/committees/relax-ng/spec.html#IDAEBZR">Specification</a>
 * @type String
 */
mozile.rng.Attribute.prototype.getNamespace = function() {
	if(this._namespace) return this._namespace;
	if(this._element.getAttribute("ns")) this._namespace = this._element.getAttribute("ns");
	else if(this.getPrefix()) {
		if(this.getPrefix() == "xml") this._namespace = null;
		else this._namespace = mozile.dom.lookupNamespaceURI(this._element, this.getPrefix());
	}
	if(!this._namespace) this._namespace = null;
	return this._namespace;
}

/**
 * Validates this RNG object. Attributes don't need children.
 * @type mozile.rng.Validation
 */
mozile.rng.Attribute.prototype.selfValidate = function(validation) {
	return validation;
}

/**
 * True if this is an attribute. False otherwise.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Attribute.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * True if this is an attribute. False otherwise.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Attribute.prototype.mayHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * Checks the node's parent for this attribute, and validates the attribute's value.
 * If this element has a namespace, then it looks for an attribute with that namespace and the localName.
 * Otherwise it looks for an attribute with localName.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Attribute.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Attribute.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.Attribute.validate() requires an mozile.rng.Validation object.");

	//var element;
	//if(node.ownerElement) element = node.ownerElement;
	//else element = node.parentNode;
	//alert("Validating Attribute "+ this.getName() +" "+ element);
	var element = validation.getCurrentParent();
	if(!element) throw Error("mozile.rng.Attribute.validate() requires the Validation object to have a 'current parent' element.");
	
	if(element) {
		var attr;
		var ns = this.getNamespace();
		if(ns) {
			if(element.getAttributeNodeNS) attr = element.getAttributeNodeNS(ns, this.getLocalName());
			else {
				// IE doesn't support getAttributeNodeNS
				var prefix = mozile.dom.lookupPrefix(element, ns);
				if(prefix) attr = element.getAttributeNode(prefix +":"+ this.getLocalName());
				//alert(prefix+":"+this.getLocalName() +" -> "+ typeof(attr));
			}
		}
		else attr = element.getAttributeNode(this.getLocalName());
		
		// TODO: validate the attribute's value

		if(attr) {
			validation.addAttribute(attr);
			validation.logMessage(this, element, "Attribute "+ attr.nodeName +" validated.");
		}
		else validation.logError(this, element, "Attribute "+ this.getName() +" not found.");
	}
	else validation.logError(this, node, "The node has no parent node.");

	return validation;
}


/**
 * Represents an RNG "text" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Text = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Text.prototype = new mozile.rng.Node;
mozile.rng.Text.prototype.constructor = mozile.rng.Text;


/**
 * True if the type is "text". False otherwise since text cannot have child nodes.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Text.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * True if the type is "text". False otherwise since text cannot have child nodes.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Text.prototype.mayHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * Validates this RNG object. Text elements are not allowed to have children.
 * @type mozile.rng.Validation
 */
mozile.rng.Text.prototype.selfValidate = function(validation) {
	if(mozile.dom.getFirstChildElement(this._element))
		validation.logError(this, this._element, "This RNG element must not have any child elements.");
	return validation;
}

/**
 * Validates the node.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Text.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Text.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.Text.validate() requires an mozile.rng.Validation object.");

	if(!validation.allowText) validation.logMessage(this, node.parentNode, "This element is allowed to contain text.");
	validation.allowText = true;
	return validation;
}



/**
 * Represents an RNG "empty" element.
 * <p>An empty element indicates that the element can have no child nodes of any kind, including attributes and non-whitespace text.
 * <a href="http://www.oasis-open.org/committees/relax-ng/spec.html#empty-pattern">Specification</a>
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Empty = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Empty.prototype = new mozile.rng.Node;
mozile.rng.Empty.prototype.constructor = mozile.rng.Empty;

/**
 * True if the type is "empty". False otherwise since empty can't have children.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Empty.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * True if the type is "empty". False otherwise since empty can't have children.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Empty.prototype.mayHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * Validates this RNG object. Empty elements are not allowed to have children.
 * @type mozile.rng.Validation
 */
mozile.rng.Empty.prototype.selfValidate = function(validation) {
	if(mozile.dom.getFirstChildElement(this._element))
		validation.logError(this, this._element, "This RNG element must not have any child elements.");
	return validation;
}

/**
 * Validates the node.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Empty.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Empty.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.Empty.validate() requires an mozile.rng.Validation object.");
	validation.isEmpty = true;
	validation.logMessage(this, node.parentNode, "This element must be empty.");
	return validation;
}


/**
 * Represents an RNG "group" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Group = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Group.prototype = new mozile.rng.Node;
mozile.rng.Group.prototype.constructor = mozile.rng.Group;


/**
 * Validates this RNG object. Groups must have at least two child elements.
 * @type mozile.rng.Validation
 */
mozile.rng.Group.prototype.selfValidate = function(validation) {
	if(mozile.dom.getChildElements(this._element).length < 2)
		validation.logError(this, this._element, "This RNG element must have at least two child elements.");
	return validation;
}

/**
 * Validates nodes in a sequence. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.Group.prototype._validateSequence = mozile.rng.validateSequence;

/**
 * Validates the node.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Group.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Group.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.Group.validate() requires an mozile.rng.Validation object.");
	
	validation = this._validateSequence(node, validation);

	if(validation.isValid) validation.logMessage(this, validation.getCurrentElement(), "Group is valid.");	
	return validation;
}



/**
 * Represents an RNG "optional" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Optional = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Optional.prototype = new mozile.rng.Node;
mozile.rng.Optional.prototype.constructor = mozile.rng.Optional;

/**
 * True if the type is "optional". False otherwise since any children are optional.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Optional.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * Validates nodes in a sequence. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.Optional.prototype._validateSequence = mozile.rng.validateSequence;

/**
 * Validates the node.
 * Creates a new validation object and tries to validate all children.
 * If the validation succeeds the validation object is merged and returned.
 * If the validation fails, the validation object is returned but not merged.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Optional.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Optional.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.Optional.validate() requires an mozile.rng.Validation object.");
	
	var result = validation.branch();
	result = this._validateSequence(node, result);

	if(result.isValid) {
		validation.count = 1;
		validation.merge(result);
		validation.logMessage(this, validation.getCurrentElement(), "Option is present.");
	}
	else {
		if(mozile.rng.debug) validation.append(result, true); 
		validation.logMessage(this, node, "Option is not present.");
	}
	
	return validation;
}


/**
 * Represents an RNG "zeroOrMore" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.ZeroOrMore = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.ZeroOrMore.prototype = new mozile.rng.Node;
mozile.rng.ZeroOrMore.prototype.constructor = mozile.rng.ZeroOrMore;

/**
 * True if the type is "zeroOrMore". False otherwise since any children are optional.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.ZeroOrMore.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	else return false;
}

/**
 * Validates the node.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.ZeroOrMore.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.ZeroOrMore.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.ZeroOrMore.validate() requires an mozile.rng.Validation object.");
	
	validation = this._validateMany(node, validation);
	validation.logMessage(this, validation.getCurrentElement(), "ZeroOrMore matched "+ validation.count +" times.");

	return validation;
}

/**
 * Validates nodes in a sequence. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.ZeroOrMore.prototype._validateSequence = mozile.rng.validateSequence;

/**
 * Validates the set of the RNG children as many times as possible. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.ZeroOrMore.prototype._validateMany = mozile.rng.validateMany;


/**
 * Represents an RNG "oneOrMore" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.OneOrMore = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.OneOrMore.prototype = new mozile.rng.Node;
mozile.rng.OneOrMore.prototype.constructor = mozile.rng.OneOrMore;

/**
 * Validates the node.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.OneOrMore.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.OneOrMore.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.OneOrMore.validate() requires an mozile.rng.Validation object.");

	validation = this._validateMany(node, validation);
	if(validation.count == 0) validation.logError(this, validation.getCurrentElement(), "OneOrMore did not match any nodes.");
	validation.logMessage(this, validation.getCurrentElement(), "OneOrMore matched "+ validation.count +" times.");

	return validation;
}

/**
 * Validates nodes in a sequence. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.OneOrMore.prototype._validateSequence = mozile.rng.validateSequence;

/**
 * Validates the set of the RNG children as many times as possible. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.OneOrMore.prototype._validateMany = mozile.rng.validateMany;



/**
 * Represents an RNG "choice" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Choice = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Choice.prototype = new mozile.rng.Node;
mozile.rng.Choice.prototype.constructor = mozile.rng.Choice;

/**
 * True if the type is "choice" or if all children mustHave the given type.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Choice.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	if(!this._mustHaveCache) this._mustHaveCache = new Object();
	if(this._mustHaveCache[type]) return this._mustHaveCache[type];

	var result = true;
	for(var c=0; c < this.getChildNodes().length; c++) {
		if(!this.getChildNodes()[c].mustHave(type)) {
			result = false;
			break;	
		}
	}

	this._mustHaveCache[type] = result;
	return result;
}

/**
 * Validates this RNG object. Choices must have at least two child elements.
 * @type mozile.rng.Validation
 */
mozile.rng.Choice.prototype.selfValidate = function(validation) {
	if(mozile.dom.getChildElements(this._element).length < 2)
		validation.logError(this, this._element, "This RNG element must have at least two child elements.");
	return validation;
}

/**
 * Validates the node.
 * If this allowText is false and this choice contains an mozile.rng.Text child, then the mozile.rng.Text is immediately validated. Afterward any mozile.rng.Text children are ignored when trying to match an element.
 * For each RNG child object a new mozile.rng.Validation object is created and the node is validated. The first valid result is returned.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Choice.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Choice.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.Choice.validate() requires an mozile.rng.Validation object.");

	var RNGChildren = this.getChildNodes();
	var r = 0;
	for(r=0; r < RNGChildren.length; r++) {
		if(validation.allowText) break;
		if(RNGChildren[r].getType() == "text") validation = RNGChildren[r].validate(node, validation);
	}
	
	var result;
	for(r=0; r < RNGChildren.length; r++) {
		if(RNGChildren[r].getType() == "text" && node.nodeType == mozile.dom.ELEMENT_NODE) continue;
		result = validation.branch();
		result = RNGChildren[r].validate(node, result);

		if(result.isValid) break;
		else if(mozile.rng.debug) validation.append(result, true); 
	}

	if(result && result.isValid) {
		validation.merge(result);
		validation.logMessage(this, validation.getCurrentElement(), "Choice number "+ (r+1) +" selected.");
	}
	else validation.logError(this, node, "All choices failed to validate.");

	return validation;
}



/**
 * Represents an RNG "define" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Define = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Define.prototype = new mozile.rng.Node;
mozile.rng.Define.prototype.constructor = mozile.rng.Define;


/**
 * Gets the full name of the element. Strips leading and trailing whitespace.
 * @type String
 */
mozile.rng.Define.prototype.getName = mozile.rng.getName;

/**
 * Combines this mozile.rng.Define object with another mozile.rng.Define object.
 * @type mozile.rng.Define
 */
mozile.rng.Define.prototype.combine = mozile.rng.combine;

/**
 * Validates this RNG object. Definitions must bust be children of grammar, include, or div elements, and have child elements and names.
 * @type mozile.rng.Validation
 */
mozile.rng.Define.prototype.selfValidate = function(validation) {
	if(!this._element.parentNode)
		validation.logError(this, this._element, "This RNG element must be the child of a 'grammar', 'include', or 'div' element.");

	var parentName = mozile.dom.getLocalName(this._element.parentNode);
	if (parentName != "grammar" && parentName != "include" && parentName != "div")
		validation.logError(this, this._element, "This RNG element must be the child of a 'grammar', 'include', or 'div' element.");

	if(!mozile.dom.getFirstChildElement(this._element))
		validation.logError(this, this._element, "This RNG element must have at least one child element.");
	if(!this.getName())
		validation.logError(this, this._element, "This RNG element must have a 'name' attribute.");
	
	var combine = this._element.getAttribute("combine");
	if(combine) {
		if(combine != "choice" && combine != "interleave")
			validation.logError(this, this._element, "This RNG 'define' element has an invalid 'combine' attribute value of '"+ combine +"'.");
	}

	return validation;
}

/**
 * Validates nodes in a sequence. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.Define.prototype._validateSequence = mozile.rng.validateSequence;

/**
 * Validates the node.
 * Validates each node in sequence.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Define.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Define.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.Define.validate() requires an mozile.rng.Validation object.");
	
	validation = this._validateSequence(node, validation);

	if(validation.isValid) validation.logMessage(this, validation.getCurrentElement(), "Definition is valid.");	
	return validation;
}



/**
 * Represents an RNG "ref" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Ref = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Ref.prototype = new mozile.rng.Node;
mozile.rng.Ref.prototype.constructor = mozile.rng.Ref;

/**
 * An array of all cached properties. They can be cleared with the reset() method.
 * @private
 * @type Array
 */
mozile.rng.Ref.prototype._cached = [this._type, this._name, this._grammar, this._definition];

/**
 * Gets the full name of the element. Strips leading and trailing whitespace.
 * @type String
 */
mozile.rng.Ref.prototype.getName = mozile.rng.getName;

/**
 * Gets the definition matching this reference. Caches values for faster lookup.
 * @type mozile.rng.Define
 */
mozile.rng.Ref.prototype.getDefinition = function() {
	if(!this._grammar) this._grammar = this.getGrammar()
	if(this._grammar) {
		if(!this._definition) this._definition = this.getGrammar().getDefinition(this.getName());
		if(this._definition) return this._definition;
	}
	return null;
}

/**
 * Gets an array of all the elements which descend from this node.
 * @param types The RNG element type(s) of search for. Can be a string or an array of strings. E.g. "element".
 * @param {Boolean} deep Optional. If the current node is an element and "deep" is true, then its children are included.
 * @type Array
 */
mozile.rng.Ref.prototype.getDescendants = function(types, deep) {
	if(!deep && mozile.rng.checkType(types, this.getType())) return new Array(this);
	else if(this.getDefinition()) return this.getDefinition().getDescendants(types, deep);
	else return new Array();
}

/**
 * True if the type is "ref". Otherwise the definition is checked.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Ref.prototype.mustHave = function(type) {
	if(this.getType() == type) return true;
	if(this.getDefinition()) return this.getDefinition().mustHave(type);
	else return false;
}

/**
 * True if the type is "ref". Otherwise the definition is checked.
 * @param {String} type The type to be matched.
 * @type Boolean
 */
mozile.rng.Ref.prototype.mayHave = function(type) {
	if(this.getType() == type) return true;
	if(this.getDefinition()) return this.getDefinition().mayHave(type);
	else return false;
}

/**
 * Validates this RNG object. Refs may not have children and they must have names.
 * @type mozile.rng.Validation
 */
mozile.rng.Ref.prototype.selfValidate = function(validation) {
	if(mozile.dom.getFirstChildElement(this._element))
		validation.logError(this, this._element, "This RNG element must not have any child elements.");
	if(!this.getName())
		validation.logError(this, this._element, "This RNG element must have a 'name' attribute.");
	return validation;
}

/**
 * Validates the node.
 * Checks that this reference has a definiton, and calls the definition's validation method.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Ref.prototype.validate = function(node, validation) {
	if(this.getDefinition()) {
		validation.logMessage(this, node, "Reference followed.");
		validation = this.getDefinition().validate(node, validation);
	}
	else validation.logError(this, node, "This reference does not have a matching definition.");
	return validation;
}



/**
 * Represents an RNG "data" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Data = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Data.prototype = new mozile.rng.Node;
mozile.rng.Data.prototype.constructor = mozile.rng.Data;

/**
 * Validates this RNG object. Data elements do not need children but they need a 'type'.
 * @type mozile.rng.Validation
 */
mozile.rng.Data.prototype.selfValidate = function(validation) {
	if(!this._element.getAttribute("type"))
		validation.logError(this, this._element, "This RNG element must have an 'type' attribute.");
	return validation;
}



/**
 * Represents an RNG "param" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Param = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Param.prototype = new mozile.rng.Node;
mozile.rng.Param.prototype.constructor = mozile.rng.Param;


/**
 * Represents an RNG "value" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Value = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Value.prototype = new mozile.rng.Node;
mozile.rng.Value.prototype.constructor = mozile.rng.Value;

/**
 * Validates this RNG object. Value elements must have non-whitespace text.
 * @type mozile.rng.Validation
 */
mozile.rng.Value.prototype.selfValidate = function(validation) {
	for(var r=0; r < this._element.childNodes.length; r++) {
		if(this._element.childNodes[r].nodeType == mozile.dom.TEXT_NODE &&
			!mozile.dom.isWhitespace(this._element.childNodes[r])) {
			return validation;
		}	
	}

	validation.logError(this, this._element, "This RNG element must contain non-whitespace text.");
	return validation;
}


/**
 * Represents an RNG "include" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Include = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Include.prototype = new mozile.rng.Node;
mozile.rng.Include.prototype.constructor = mozile.rng.Include;

/**
 * Validates this RNG object. Includes must be childen of a grammar. They do not need children but they need an 'href'.
 * @type mozile.rng.Validation
 */
mozile.rng.Include.prototype.selfValidate = function(validation) {
	if(!this._element.parentNode || mozile.dom.getLocalName(this._element.parentNode) != "grammar")
		validation.logError(this, this._element, "This RNG element must be the child of a grammar element.");

	if(!this._element.getAttribute("href"))
		validation.logError(this, this._element, "This RNG element must have an 'href' attribute.");
	return validation;
}

/**
 * Includes definitions from the target grammar into the current grammar.
 * Converts a relative href to a URI relative to the element's owner document, if necessary.
 * Loads the file for the included schema and makes sure its document element is a "grammar" element.
 * Tries to include all "define", "start", and "include" elements from the top level of the new file into this object's grammar.
 * The "define" elements can be overridden by children of this Include object.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Include.prototype.include = function(validation) {
	if(!validation) throw Error("mozile.rng.Include.include() requires an mozile.rng.Validation object.");

	var href = this._element.getAttribute("href");
	var schema = this.getSchema();
	var grammar = this.getGrammar();
	if(!href || !schema || !grammar) {
		validation.logError(this, this._element, "mozile.rng.Include.include() requires an 'href', a schema, and a grammar, but one or more was missing.");
		return validation;
	}
	
	// Get an absolute URI.
	var root;
	if(this.filepath) root = this.filepath;
	else if(schema.filepath) root = schema.filepath;
	//else if(mozile.root) root = mozile.root;
	var URI = mozile.getAbsolutePath(href, root);
	var filepath = mozile.getDirectory(URI);
	//mozile.debug.debug("mozile.rng.Include.include", root +" "+ href +" "+ URI);
	//alert(root +" "+ href +" "+ URI);
	
	// Load the file.
	validation.logMessage(this, this._element, "Including RNG content from '"+ URI +"'.");
	var rngDoc = schema.load(URI);
	if(!rngDoc || !rngDoc.documentElement || 
		mozile.dom.getLocalName(rngDoc.documentElement) != "grammar") {
		validation.logError(this, null, "mozile.rng.Include.include() could not load schema at '"+ URI +"'.");
		//throw Error("Bad URI: "+ URI);
		return validation;
	}


	// Include the children of the new file.
	var children = rngDoc.documentElement.childNodes;
	for(var c=0; c < children.length; c++) {
		var child = null;
		switch(mozile.dom.getLocalName(children[c])) {
			case "start":
				child = schema.parseElement(children[c], validation);
				break;
			case "define":
				// Check to see if the definition should be overridden.
				for(var i=0; i < this._element.childNodes.length; i++) {
					var includeChild = this._element.childNodes[i];
					if(includeChild.nodeType != mozile.dom.ELEMENT_NODE) continue;
					var includeChildName = includeChild.getAttribute("name");
					if(mozile.dom.getLocalName(includeChild) == "define" &&
						includeChildName == children[c].getAttribute("name") ) {
						validation.logMessage(this, includeChild, "Overriding defintition of '"+ includeChildName +"'.");
						child = schema.parseElement(includeChild, validation);
						break;
					}
				}
				if(!child) child = schema.parseElement(children[c], validation);
				break;
			case "include":	
				child = schema.parseElement(children[c], validation);
				child.filepath = filepath;
				break;
			default:
				child = null;
				break;
		}
		if(child) grammar.appendChild(child);
	}
	return validation;
}



/**
 * Represents an RNG "interleave" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Interleave = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Interleave.prototype = new mozile.rng.Node;
mozile.rng.Interleave.prototype.constructor = mozile.rng.Interleave;

/**
 * Validates this RNG object. Groups must have at least two child elements.
 * @type mozile.rng.Validation
 */
mozile.rng.Interleave.prototype.selfValidate = function(validation) {
	if(mozile.dom.getChildElements(this._element).length < 2)
		validation.logError(this, this._element, "This RNG element must have at least two child elements.");
	return validation;
}

/**
 * Validates the node.
 * @type mozile.rng.Validation
 */
mozile.rng.Interleave.prototype.validate = mozile.rng.validateInterleave;






/**
 * Represents an RNG "div" element.
 * @constructor
 * @param {Element} element The RNG element to represent.
 */
mozile.rng.Div = function(element, schema) {
	/**
	 * The element associated with this RNG node.
	 * @private
	 * @type Element
	 */
	this._element = element;

	/**
	 * The schema to which this RNG node belongs.
	 * @private
	 * @type mozile.rng.Schema
	 */
	this._schema = schema;
}
mozile.rng.Div.prototype = new mozile.rng.Node;
mozile.rng.Div.prototype.constructor = mozile.rng.Div;


/**
 * Validates this RNG object. Divs must bust be children of a grammar.
 * @type mozile.rng.Validation
 */
mozile.rng.Div.prototype.selfValidate = function(validation) {
	if(!this._element.parentNode || 
		mozile.dom.getLocalName(this._element.parentNode) != "grammar" )
		validation.logError(this, this._element, "This RNG element must be the child of a 'grammar' element.");
	return validation;
}

/**
 * Validates nodes in a sequence. 
 * @private
 * @type mozile.rng.Validation
 */
mozile.rng.Div.prototype._validateSequence = mozile.rng.validateSequence;

/**
 * Validates the node.
 * @param {Node} node The node to be validated.
 * @param {mozile.rng.Validation} validation The mozile.rng.Validation object for this validation operation.
 * @type mozile.rng.Validation
 */
mozile.rng.Div.prototype.validate = function(node, validation) {
	if(!node) throw Error("mozile.rng.Div.validate() requires a node.");
	if(!validation) throw Error("mozile.rng.Div.validate() requires an mozile.rng.Validation object.");
	
	validation = this._validateSequence(node, validation);

	if(validation.isValid) validation.logMessage(this, validation.getCurrentElement(), "Div is valid.");	
	return validation;
}




/*
mozile.rng.Mixed
mozile.rng.ParentRef
mozile.rng.ExternalRef
mozile.rng.List
mozile.rng.NotAllowed
mozile.rng.Except
mozile.rng.NSName
mozile.rng.Div
*/

/**
 * Editing tools.
 * @type Object
 */
mozile.edit = new Object();
// JSDoc hack
mozile.edit.prototype = new mozile.Module;

/**
 * Indicates whether Mozile is currently allowed to edit the document.
 * A value of "true" means the document can be edited.
 * A value of "false" means that editing is disabled.
 * @type Boolean
 */
mozile.edit.editable = true;

/**
 * Indicates Mozile's current editing status.
 * A value of "true" means the document is being edited.
 * A value of "false" means that editing is currently disabled.
 * @type Boolean
 */
mozile.edit.status = false;

/**
 * Flag for editing direction.
 * @type Integer
 */
mozile.edit.NEXT = 1;

/**
 * Flag for editing direction.
 * @type Integer
 */
mozile.edit.PREVIOUS = -1;

/**
 * An associative array of all the Command objects (including subclasses) in this document. Keys are command names, and values are the command objects.
 */
mozile.edit.allCommands = new Object();

/**
 * An associative array of keyCodes and their standard names. Used to convert keyCode to strings in accelerators.
 * See: http://www.xulplanet.com/references/objref/KeyboardEvent.html
 * TODO: Make sure this is not Mozilla-centric.
 * @type Object
 */
mozile.edit.keyCodes = {
	8:  "Backspace",
	9:  "Tab",
	12: "Clear",
	13: "Return",
	14: "Enter",
	19: "Pause",
	27: "Escape",
	32: "Space",
	33: "Page-Up",
	34: "Page-Down",
	35: "End",
	36: "Home",
	37: "Left",
	38: "Up",
	39: "Right",
	40: "Down",
	45: "Insert",
	46: "Delete",
	112: "F1",
	113: "F2",
	114: "F3",
	115: "F4",
	116: "F5",
	117: "F6",
	118: "F7",
	119: "F8",
	121: "F9",
	122: "F10",
	123: "F11",
	123: "F12"
}


/**
 * Searches for an ancestor which mas been marked by Mozile as editable.
 * Containers themselves should not be edited, but the nodes they contain can be.
 * Elements with the "contentEditable" attribute set to true are marked as editable when found.
 * @param {Element} element The element to check.
 * @type Element
 */
mozile.edit.getContainer = function(element) {
	if(!element || !element.nodeType) return null;
	if(element.nodeType != mozile.dom.ELEMENT_NODE) element = element.parentNode;
	if(!element || !element.nodeType) return null;

	var doc = element.ownerDocument;
	while(element && element.nodeType &&
		element.nodeType == mozile.dom.ELEMENT_NODE) {
		if(mozile.edit.isEditableElement(element)) return element;

		switch(element.getAttribute("contentEditable")) {
			case "true":
				mozile.editElement(element);
				return element;
			case "false":
				// TODO: Protect element?
				return null;
		}
		element = element.parentNode;
	}

	return null;
}

/**
 * Detect whether this node is inside an editable container element.
 * @param {Node} node The node to check.
 * @type Boolean
 */
mozile.edit.isEditable = function(node) {
	if(!node) return false;
	var container = mozile.edit.getContainer(node)
	if(container && container != node) return true;
	else return false;
}

/**
 * Determines whether this element has been marked as editable by Mozile.
 * @param {Element} element The element to check.
 * @type Boolean
 */
mozile.edit.isEditableElement = function(element) {
	if(element && mozile.edit.getMark(element, "editable")) return true;
	return false;
}

/**
 * Sets the editing status for the document.
 * In browsers that do not support contentEditable, the document.designMode is turned on or off.
 * If the status is already set to the desired value, nothing is changed.
 * @param {Boolean} status The desired editing status.
 * @type Boolean
 */
mozile.edit.setStatus = function(status) {
	status = Boolean(status);
	if(mozile.edit.status != status) {
		mozile.edit.status = status;
		if(mozile.useDesignMode == true && 
			typeof(document.documentElement.contentEditable) == "undefined") {
			document.designMode = (status) ? "on" : "off";
			//alert("Design Mode set to "+ document.designMode);
		}
		//alert("Editing status changed to "+ status);
	}
	return mozile.edit.status;
}

/**
 * Enables editing of the current document.
 * @type Void
 */
mozile.edit.enable = function() {
	mozile.edit.editable = true;
	return mozile.edit.editable;
}

/**
 * Disables editing of the current document.
 * @type Void
 */
mozile.edit.disable = function() {
	mozile.edit.editable = false;
	return mozile.edit.editable;
}

/**
 * Starts editing the current document.
 * @type Void
 */
mozile.edit.start = function() {
	return mozile.edit.setStatus(true);
}

/**
 * Stops editing the current document.
 * @type Void
 */
mozile.edit.stop = function() {
	return mozile.edit.setStatus(false);
}

/**
 * Sets a property of a special "mozile" object for an element, which stores data for Mozile to use.
 * Note: this is restricted to elements because of limitations with IE6.
 * Warning: Won't work for XML element in IE6.
 * @param {Element} element The element to mark.
 * @param {String} key The name of the property to set. Must be a valid JavaScript property name.
 * @param value The value for the new property.
 * @return The value, if set. Otherwise null.
 */
mozile.edit.setMark = function(element, key, value) {
	if(!element || element.nodeType == undefined) return null;
	if(element.nodeType != mozile.dom.ELEMENT_NODE) return null;
	if(!key || typeof(key) != "string") return null;
	
	// Catch errors thrown by IE6 in the case of XML elements.
	try {
		if(element.mozile == undefined || typeof(element.mozile) != "object")
			element.mozile = new Object();
		element.mozile[key] = value;
		return value;
	} catch(e) {
		return null;
	}
}

/**
 * Gets a property of a special "mozile" object belonging to an element. 
 * @param {Element} element The element to get check.
 * @param {String} key The name of the property to set. Must be a valid JavaScript property name.
 * @return The value, if set. Otherwise undefined.
 */
mozile.edit.getMark = function(element, key) {
	if(!element || element.nodeType == undefined) return undefined;
	if(element.nodeType != mozile.dom.ELEMENT_NODE) return undefined;
	if(!key || typeof(key) != "string") return undefined;
	if(element.mozile == undefined || !element.mozile) return undefined;
	if(element.mozile[key] == undefined) return undefined;
	return element.mozile[key];
}

/**
 * Gets the mozile.rng.Element object which corresponds to the given element.
 * Currently returns the first RNG object of type "element" with a name matching the given element's nodeName (changed to lower case).
 * TODO: Should be smarter.
 * @param {Node} node The node to find the RNG rule for.
 * @type mozile.rng.Element
 */
mozile.edit.lookupRNG = function(node) {
	if(!node) return null;
	var element = node;
	if(node.nodeType != mozile.dom.ELEMENT_NODE) element = node.parentNode;

	if(!mozile.schema) return null;
	var name = mozile.dom.getLocalName(element);
	if(name && mozile.dom.isHTML(node)) name = name.toLowerCase();
	var matches = mozile.schema.getNodes("element", name);
	if(matches.length > 0) return matches[0];
	else return null;
}


/**
 * Search through the children of the given node (following any references) for MES definitions.
 * @param {mozile.edit.Command} container The Command object to attach new commands to.
 * @parse {Node} node A node in the RNG schema to search for new commands.
 * @type Void
 */
mozile.edit.parseMES = function(container, node) {
	if(node.nodeType != mozile.dom.ELEMENT_NODE) return;
	var command, define;

	for(var i=0; i < node.childNodes.length; i++) {
		var child = node.childNodes[i];
		switch(mozile.dom.getNamespaceURI(child)) {
			// Mozile namespace case
			case mozile.xml.ns.mes:
				switch(mozile.dom.getLocalName(child)) {
					case "ref":  // Follow references
						define = mozile.edit.followMESRef(child);
						if(define) mozile.edit.parseMES(container, define);
						break;
					case "command":
						command = mozile.edit.generateCommand(child);
						if(command) container.addCommand(command);
						break;
					case "group":				
						command = mozile.edit.generateCommand(child);
						if(command) {
							container.addCommand(command);
							// Get more commands.
							if(command._commands.length == 0) 
								mozile.edit.parseMES(command, child);
						}
						break;			
				}
				break;

			// RNG namespace case
			case mozile.xml.ns.rng:
				// Follow RNG references.
				if(child.nodeName == "ref") {
					var name = child.getAttribute("name");
					if(!name) continue;
					define = mozile.schema._root.getDefinition(name);
					mozile.edit.parseMES(container, define._element);
				}
				break;
		}
	}
}

/**
 * Follow an MES reference and return an MES define element.
 * Currently only works for define elements immediately under the root of the document that the ref element belongs to.
 * @param {Element} element The "ref" element to follow.
 * @type Element
 */
mozile.edit.followMESRef = function(element) {
	var define = mozile.edit.getMark(element, "define");
	if(define && define.nodeType && define.nodeType == mozile.dom.ELEMENT_NODE)
		return define;

	var name = element.getAttribute("name");
	if(!name) return null;

	var doc = element.ownerDocument.documentElement;
	define = null;
	var child;
	for(var i=0; i < doc.childNodes.length; i++) {
		child = doc.childNodes[i];
		if(mozile.dom.getNamespaceURI(child) == mozile.xml.ns.mes &&
			mozile.dom.getLocalName(child) == "define" &&
			child.getAttribute("name") == name) {
			define = child;
			break;
		}
	}
	
	if(define) {
		mozile.edit.setMark(element, "define", define);
		return define;
	}
	else return null;
}

/**
 * For each RNG Element in the given schema, add all of the appropriate commands.
 * @param {mozile.rng.Schema} schema The schema to generate commands for.
 * @type Void
 */
mozile.edit.generateCommands = function(schema) {
	var elements = schema.getNodes("element");
	for(var i=0; i < elements.length; i++) {
		elements[i].addCommand(mozile.edit.navigateLeftRight);

		// Common text editing commands.
		if(elements[i].mayContain("text")) {
			elements[i].addCommand(mozile.edit.insertText);
			elements[i].addCommand(mozile.edit.removeText);
		}
		
		// Common rich editing commands
		if(mozile.edit.remove) elements[i].addCommand(mozile.edit.remove);

		// Add other commands, using the extended RNG schema information.
		mozile.edit.parseMES(elements[i], elements[i]._element);
	}
}

/**
 * Generate a Mozile command from an RNG Element.
 * @param {mozile.rng.Element} rng The RNG element to generate commands for.
 * @type mozile.rng.Command
 */
mozile.edit.generateCommand = function(node) {
	var name = node.getAttribute("name");
	if(!name) return null;
	
	// Check for a command with this name.
	if(mozile.edit.allCommands[name]) return mozile.edit.allCommands[name];

	// Create Command or CommandGroup object
	var command;
	if(mozile.dom.getLocalName(node) == "command") {
		var className = node.getAttribute("class");
		if(className && mozile.edit[className]) {
			//alert("Class found: "+ className);
			eval("command = new mozile.edit."+ className +"(name)");
		}
		else command = new mozile.edit.Command(name);

		// Parse child elements.
		var child = node.firstChild;
		while(child) {
			if(child.nodeType == mozile.dom.ELEMENT_NODE) {
				switch(mozile.dom.getLocalName(child)) {
					case "element":
						var element = mozile.dom.getFirstChildElement(child);
						if(element) command.element = element;
						break;
					case "script":
						command.script = child;
						break;
				}
			}
			child = child.nextSibling;
		}
	}

	else if(mozile.dom.getLocalName(node) == "group") {
		command = new mozile.edit.CommandGroup(name);
	}
	else return null;

	// Assign properties.
	command.node = node;
	var properties = ["priority", "label", "image", "tooltip", "accel", "makesChanges", "watchesChanges", "element", "text", "remove", "nested", "direction", "target", "collapse", "copyAttributes", "styleName", "styleValue"];
	for(var i=0; i < properties.length; i++) {
		var property = properties[i];
		if(node.getAttribute(property)) {
			var value = node.getAttribute(property);
			// TODO: Convert numeric values?
			if(value.toLowerCase() == "true") value = true;
			else if(value.toLowerCase() == "false") value = false;
			command[property] = value;
		}
	}
	
	// Prepare any accelerators.
	if(command.accel) {
		command.accels = mozile.edit.splitAccelerators(command.accel);
	}
	
	// Add a direction property.
	if(command.target && !command.direction) {
		command.direction = null;
	}

	// Evaluate any scripts.
	if(command.script) {		
		child = command.script.firstChild;
		while(child) {
			if(child.nodeType == mozile.dom.TEXT_NODE ||
				child.nodeType == mozile.dom.CDATA_SECTION_NODE) {
				command.evaluate(child.data);
			}
			child = child.nextSibling;
		}
	}

	return command;
}



/**** Editing Commands ****/




/**
 * Check an event against an array of "accelerator" strings (i.e. a specified key combination) or array of strings.
 * @param {Event} event The event to check.
 * @param {Array} accelerator An array of strings denoting the key combination(s).
 * @type Boolean
 */
mozile.edit.checkAccelerators = function(event, accelerators) {
	if(!event) return false;
	if(typeof(accelerators) != "object" || !accelerators.length) return false;
	for(var i=0; i < accelerators.length; i++) {
		if(mozile.edit.checkAccelerator(event, accelerators[i])) return true;
	}
	return false;
}

/**
 * Check an event against an "accelerator" string (i.e. a specified key combination) or array of strings.
 * Examples of accelerators: "Command-D", "Command-Shift-D". "Command" is translated as "Control" under Windows and Linux, and "Meta" under Mac OS.
 * <p>Note: The sequence is important. The order must be "Command-Meta-Control-Alt-Shift-UpperCaseCharacter". (Comparison is done using lower case. Not all combinations will work on all platforms.)
 * @param {Event} event The event to check.
 * @param {String} accelerator A string denoting the key combination(s).
 * @type Boolean
 */
mozile.edit.checkAccelerator = function(event, accelerator) {
	if(!event) return false;
	if(typeof(accelerator) != "string") return false;
	//if(event.type.indexOf("key")==0) mozile.debug.debug("", [event.type, event.keyCode, mozile.edit.keyCodes[event.keyCode]].join("\n"));

	if(mozile.browser.isIE) {
		if(event.type != "keydown") {
			if(event.type != "keypress") return false;
			if(event.keyCode && !mozile.edit.keyCodes[event.keyCode]) return false;
		}
	}
	else if(event.type != "keypress") return false;

	if(event.accel == undefined) event.accel = mozile.edit.generateAccelerator(event);
	if(event.accel.toLowerCase() == accelerator.toLowerCase()) return true;
	else return false;
}


/**
 * Takes an event and returns a representation of the event as an accelerator string.
 * @param {Event} event The event to generate the accelerator from.
 * @type String
 */
mozile.edit.generateAccelerator = function(event) {
	if(!event) return "";
	var accel = "";

	if(event.metaKey)  accel = accel + "Meta-";
	if(event.ctrlKey)  accel = accel + "Control-";
	if(event.altKey)   accel = accel + "Alt-";
	if(event.shiftKey) accel = accel + "Shift-";

	if(event.keyCode && mozile.edit.convertKeyCode(event.keyCode)) {
		accel = accel + mozile.edit.convertKeyCode(event.keyCode);
	}
	// Special case for "Space"
	else if(event.charCode == 32) accel = accel + "Space";
	else accel = accel + String.fromCharCode(event.charCode).toUpperCase();
	
	var command = "Control";
	if(mozile.os.isMac) command = "Meta";
	accel = accel.replace(command, "Command");

	return accel;
}

/**
 * Splits a space-separated list of accelerator strings, and cleans them.
 * @param {String} accelerators A space-separated list of accelerators.
 * @type Array
 */
mozile.edit.splitAccelerators = function(accelerators) {
	var accels = new Array();
	var split = accelerators.split(/\s/);
	var accel;
	for(var i=0; i < split.length; i++) {
		accel = split[i];
		accel = accel.replace(/\s+/g, "");
		if(accel) accels.push(accel);
	}
	return accels;
}

/**
 * Takes an accelerator string and returns an object with easy-to-use properties.
 * @param {String} accelerator The accelerator string to check. See @see #mozile.edit.checkAccelerator
 * @type Object
 * @return The returned object has Boolean properties for: command, meta, ctrl, alt, and shift, and integer "charCode", and a string "character" and a string "abbr".
 */
mozile.edit.parseAccelerator = function(accelerator) {
	// Remove everything after any white-space character.
	accelerator = accelerator.replace(/\s.*/, "");

	var accel = {
		command: false,
		meta: false,
		ctrl: false,
		alt: false,
		shift: false,
		charCode: 0,
		character: "",
		abbr: ""
	}
	if(accelerator.indexOf("Command")  > -1) accel.command = true;
	if(accelerator.indexOf("Meta")     > -1) accel.meta    = true;
	if(accelerator.indexOf("Control")  > -1) accel.ctrl    = true;
	if(accelerator.indexOf("Alt")      > -1) accel.alt     = true;
	if(accelerator.indexOf("Shift")    > -1) accel.shift   = true;
	accel.character = accelerator.substring(accelerator.lastIndexOf("-")+1);
	
	// TODO: Use images instead?
	if(mozile.os.isMac) {
		if(accel.ctrl) accel.abbr += "\u2303";
		if(accel.alt) accel.abbr += "\u2325";
		if(accel.shift) accel.abbr += "\u21E7";
		if(accel.command) accel.abbr += "\u2318";
		accel.abbr += accel.character;
	}
	else {
		if(accel.command) accel.abbr += "Ctrl+";
		if(accel.alt) accel.abbr += "Alt+";
		if(accel.shift) accel.abbr += "Shift+";
		accel.abbr += accel.character;
	}
	
	return accel;
}

/**
 * Converts a key code to a key name. E.g. 14 becomes "Enter".
 * @param {Integer} keyCode The key code to convert.
 * @type String
 */
mozile.edit.convertKeyCode = function(keyCode) {
	if(mozile.edit.keyCodes[keyCode]) return mozile.edit.keyCodes[keyCode];
	else return null;
}



/**** Support Methods for Commands ****/

/**
 * Adds a command to the list of global commands for this document.
 * @param {mozile.edit.Command} command The command to be added.
 * @type mozile.edit.Command
 */
mozile.edit.addCommand = function(command) {
	if(!command) return command;
	if(!this._commands) this._commands = new Array();
	if(!this._priority) this._priority = new Array();
	this._commands.push(command);
	this._priority.push(command);
	this._priority.sort(mozile.edit.compareCommands);
	return command;
}

/**
 * Compares a pair commands by their "priority" attribute. Higher values come first.
 * Designed to be used by JavaScript's arrat.sort() method.
 * @param {mozile.edit.Command} command1
 * @param {mozile.edit.Command} command2
 * @type Number
 */
mozile.edit.compareCommands = function(command1, command2) {
	if(command1.priority == undefined || Number(command1.priority) == NaN) 
		command1.priority = 0;
	if(command2.priority == undefined || Number(command2.priority) == NaN) 
		command2.priority = 0;
	return command2.priority - command1.priority;
}

/**
 * Gets a command from the list of all commands.
 * @param {String} name The name of the command.
 * @type mozile.edit.Command
 */
mozile.edit.getCommand = function(name) {
	//return "Get command: "+ name +"\n"+ mozile.edit.allCommands[name];
	if(mozile.edit.allCommands[name]) return mozile.edit.allCommands[name];
	else return null;
}

/**
 * Takes an event object and uses it to try and trigger all of the document-wide commands in the mozile.edit._commands array.
 * @param {mozile.edit.Command} command The command to be added.
 * @type mozile.edit.State
 */
mozile.edit.handleEvent = function(event) {
	//alert(this.getName() +" is handling event "+ event +" with commands "+ this._commands);
	if(!this._priority) return null;
	
	var state;
	for(var i=0; i < this._priority.length; i++) {
		state = this._priority[i].trigger(event);
		if(state) return state;
	}
	return null;
}

/**
 * Adds a command to the list default commands for this document.
 * @param {mozile.edit.Command} command The command to be added.
 * @type mozile.edit.Command
 */
mozile.edit.addDefaultCommand = function(command) {
	if(!mozile.edit._defaultCommands) mozile.edit._defaultCommands = new Array();
	mozile.edit._defaultCommands.push(command)
	return command;
}

/**
 * Takes an event object and uses it to try and trigger all of the document-wide default commands in the mozile.edit._defaultCommands array.
 * @param {mozile.edit.Command} command The command to be added.
 * @type mozile.edit.State
 */
mozile.edit.handleDefault = function(event) {
	if(!mozile.edit._defaultCommands) return null;
	
	var state;
	for(var i=0; i < mozile.edit._defaultCommands.length; i++) {
		state = mozile.edit._defaultCommands[i].trigger(event);
		if(state) return state;
	}
	return null;
}



/**** RelaxNG Element Extensions ****/

/**
 * Add command functionality to the RNG system.
 * @type Void
 */
mozile.edit.extendRNG = function() {

/**
 * Creates a new element using the information from this RNG Element object.
 * @param {Element} parent Optional. If a parent element is given the new node will be appended as a child and the parent will be returned.
 * @type Element
 */
mozile.rng.Element.prototype.create = function(parent) {
	var node = mozile.dom.createElement(this.getName());
	// TODO: Create children.
	if(parent) {
		parent.appendChild(node);
		return parent;
	}
	else return node;
}

/**
 * Adds a command to the list of commands for this element.
 * @param {mozile.edit.Command} command The command to be added.
 * @type mozile.edit.Command
 */
mozile.rng.Element.prototype.addCommand = mozile.edit.addCommand;

/**
 * Takes an event object and uses it to try and trigger all of the commands associated with this RNG Element.
 * @param {Event} event The event to handle.
 * @type mozile.edit.State
 */
mozile.rng.Element.prototype.handleEvent = mozile.edit.handleEvent;

} // End of mozile.edit.extendRNG() method.


// If the RNG module has been loaded, extend its functionality.
if(mozile.rng) mozile.edit.extendRNG();





/**** Command State Object ****/

/**
 * The State object is a container for information used to execute and unexecute a command. It is associated with a particular command execution.
 * @param {mozile.edit.Command} command A reference to the command which this state belongs to.
 * @param {Selection} selection Optional. The current selection, or a range to be stored as the current selection. If the value is "false" then no selection is stored.
 * @constructor
 */
mozile.edit.State = function(command, selection) {
	/**
	 * A reference to the command that created this state.
	 * @type mozile.edit.Command
	 */
	this.command = command;
	
	/**
	 * A reference to the current selection object.
	 * @type mozile.dom.Selection
	 */
	this.selection = null;

	if(selection !== false) {
		if(!selection) selection = mozile.dom.selection.get();
		this.selection = { before: selection.store() };
	}

	/**
	 * Indicates that the command can be undone.
	 * @type Boolean
	 */
	this.reversible = true;

	/**
	 * Indicates that the command should be cancelled by mozile.event.handle() once complete.
	 * @type Boolean
	 */
	this.cancel = true;

	/**
	 * Specified the kind of change that this state has made.
	 * See mozile.edit.Command.respond for possible values.
	 * @type String
	 */
	this.changesMade = command.makesChanges;

	/**
	 * Indicates that the command has been executed.
	 * @type Boolean
	 */
	this.executed = false;
}

/**
 * Returns a description of this object.
 * @type String
 */
mozile.edit.State.prototype.toString = function() {
	return "[object mozile.edit.State]";
}

/**
 * Used to sanity check command arguments. If given a node, it returns an XPath. If given a string, it tries to make sure it's an XPath. Given anything else, it will return null;
 * @param input A node or XPath string.
 * @type String
 */
mozile.edit.State.prototype.storeNode = function(input) {
	if(!input) return null;

	if(typeof(input) == "string") {
		if(input.indexOf("/") != 0) return null;
		return input;
	}

	else {
		var xpath = mozile.xpath.getXPath(input);
		if(xpath) return xpath;
		else return null;
	}

	return null;
}


/**** Command Object ****/

/**
 * Commands are objects capable of making undoable changes to the document, and aware of the context in which those changes can be made.
 * @param {String} name The command's name.
 * @constructor
 */
mozile.edit.Command = function(name) { 
	/**
	 * The name for this command.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command does not contain more commands.
	  * @type Boolean
	  */
	 this.group = false;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * Can be "none", "state", "text", or "node". Each includes all the previous types.
	  * @type String
	  */
	 this.makesChanges = "node";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * Can be any of the values of makesChange.
	  * @type String
	  */
	 this.watchesChanges = "node";
	 
	 // Register this command on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}


/**
 * Returns a description of this object.
 * @type String
 */
mozile.edit.Command.prototype.toString = function() {
	return "[object mozile.edit.Command '"+ this.name +"']";
}

/**
 * Evaluates the given JavaScript code string in the context of this object instance. The method can be used to customize an instance of the Command class.
 * @param {String} code The JavaScript code to be evaluated.
 * @type Void
 */
mozile.edit.Command.prototype.evaluate = function(code) {
	eval(code);
}

/**
 * Determines whether this command should respond to a change of a given type.
 * Compares the given change type (usually the value of the last command's makesChanges property), and ompares it with this command's watchesChanges property.
 * @param {String} change The type of change. Can be "state", "text", or "node". Each includes all the previous types. Can also be "none", which means all changes are ignored.
 * @type Boolean
 */
mozile.edit.Command.prototype.respond = function(change) {
	if(!change || typeof(change) != "string") return false;

	if(change == "none") return false;
	switch(this.watchesChanges) {
		case "none":   return false;
		case "state":  if(change == "state")  return true;
		case "text":   if(change == "text")   return true;
		case "node":   if(change == "node")   return true;
	}

	return false;
}

/**
 * Indicates that the command is available to be used.
 * @param {Event} event Optional. The current event object.
 * @type Boolean
 */
mozile.edit.Command.prototype.isAvailable = function(event) {
	return true;
}

/**
 * Indicates that the command is currently active.
 * In the case of a "strong" command it would me that the cursor is inside a "strong" element, and the command's button should indicate that fact.
 * @param {Event} event Optional. The current event object.
 * @type Boolean
 */
mozile.edit.Command.prototype.isActive = function(event) {
	return false;
}

/**
 * Tests to see if the command should be executed. Returns true if the event matches the accelerator.
 * @param {Event} event Optional. The event object which caused this command to be tested.
 * @type Boolean
 */
mozile.edit.Command.prototype.test = function(event) {
	if(event) {
		if(this.accels) return mozile.edit.checkAccelerators(event, this.accels);
		else if(this.accel) {
			this.accels = mozile.edit.splitAccelerators(this.accel);
			return mozile.edit.checkAccelerator(event, this.accel);
		}
		else return false;
	}
	return true;
}

/**
 * Creates a "state" object with all of the information needed to execute the command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.Command.prototype.prepare = function(event) {
	var state = new mozile.edit.State(this);

	// Some commands may need additional information from the user.
	// If so, define a method named "prompt", which takes the event and the state and returns a Boolean true when it succeeds.
	if(this.prompt) {
		if(!this.prompt(event, state)) return null;
	}

	return state;
}

/**
 * Executes the command, but only if the test is successful.
 * @param {Event} event The event object which may trigger the command.
 * @type mozile.edit.State
 */
mozile.edit.Command.prototype.trigger = function(event) {
	if(this.test(event)) {
		return this.execute(this.prepare(event), true);
	}
	return null;
}

/**
 * Used by commands to call other commands. Executes the command if the test is successful. If the command is executed, its new state is added to the original state's actions array.
 * @param {mozile.edit.State} state The state object of the calling command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @param a* Other optional arguments, which will be sent to the perpare() method.
 * @type mozile.edit.State
 */
mozile.edit.Command.prototype.request = function(state, fresh, a1, a2, a3, a4, a5, a6, a7, a8, a9, a10) {
	var test = this.test(null, a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	if(!test) return null;
	
	var newState = this.prepare(null, a1, a2, a3, a4, a5, a6, a7, a8, a9, a10);
	if(!newState) return null;
	newState = this.execute(newState, fresh);
	if(!newState.executed) return null;
	
	if(!state.actions) state.actions = new Array();
	state.actions.push(newState);
	return newState;
}

/**
 * Executes the command and returns a "state" object which stores the information necessary to unexecute the command.
 * <p>This method is meant to be overridden by instances and subclasses.
 * @param {mozile.edit.State} state A state object with the information necessary for executing this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type mozile.edit.State
 */
mozile.edit.Command.prototype.execute = function(state, fresh) {
	mozile.debug.inform("mozile.edit.Command.execute", "Command '"+ this.name +"' executed with state "+ state);
	state.executed = true;
	return state;
}

/**
 * Reverses the operation of the execute command.
 * This is an "undo" operation which should leave the document exactly as it was before the command was originally executed.
 * <p>This method is meant to be overridden by instances and subclasses.
 * @param {mozile.edit.State} state The state object returned by the execute() method. It will contain enough information to unexecute the command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type mozile.edit.State
 */
mozile.edit.Command.prototype.unexecute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.after);

	// Unexecute actions in reverse order.
	if(state.actions) {
		for(var i = state.actions.length - 1; i >= 0; i--) {
			state.actions[i] = state.actions[i].command.unexecute(state.actions[i], fresh);
			if(state.actions[i].executed) mozile.debug.inform(this.name +".unexecute", "Child command "+ i +" failed to unexecute.");
		}
	}

	selection.restore(state.selection.before);
	state.executed = false;
	return state;
}


/**** CommandGroup Object ****/

/**
 * CommandGroups contain other commands.
 * @param {String} name The group's name.
 * @constructor
 */
mozile.edit.CommandGroup = function(name) { 
	/**
	 * The name for this command group.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command contains more commands.
	  * @type Boolean
	  */
	 this.group = true;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.makesChanges = "none";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.watchesChanges = "none";
	 
	 /**
	  * An array of commands belonging to this group.
	  * @private
	  * @type Array
	  */
	 this._commands = new Array();

	 // Register this command group on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}
mozile.edit.CommandGroup.prototype = new mozile.edit.Command;
mozile.edit.CommandGroup.prototype.constructor = mozile.edit.CommandGroup;

/**
 * Returns a description of this object.
 * @type String
 */
mozile.edit.CommandGroup.prototype.toString = function() {
	return "[object mozile.edit.CommandGroup '"+ this.name +"']";
}

/**
 * Adds a command to the list of commands for this CommandGroup.
 * @param {mozile.edit.Command} command The command to be added.
 * @type mozile.edit.Command
 */
mozile.edit.CommandGroup.prototype.addCommand = mozile.edit.addCommand;

/**
 * Takes an event object and uses it to try and trigger all of the commands associated with this CommandGroup.
 * @param {Event} event The event to handle.
 * @type mozile.edit.State
 */
mozile.edit.CommandGroup.prototype.trigger = mozile.edit.handleEvent;







/**** Undo / Redo System ****/

/**
 * An array of undo states.
 * @private
 * @type Array
 */
mozile.edit._undoStack = new Array();

/**
 * The index of the last executed state in the undo stack.
 * @private
 * @type Integer
 */
mozile.edit._undoIndex = -1;

/**
 * The current state.
 * @type mozile.edit.State
 */
mozile.edit.currentState = null;

/**
 * Displays the contents of the undo stack.
 * @type Void
 */
mozile.edit.dumpUndoStack = function() {
	var entries = new Array("Undo Stack [ "+ mozile.edit._undoIndex +" / "+ mozile.edit._undoStack.length +" ]");
	for(var i=0; i < mozile.edit._undoStack.length; i++) {
		var picked = "  ";
		if(i == mozile.edit._undoIndex) picked = "> "
		entries.push(picked + i +". "+ mozile.edit._undoStack[i].command.name);
	}
	return entries.join("\n");
}

/**
 * Records the result of a command in such a way that it can be undone.
 * @param {mozile.edit.State} state The state to record. Expected to be the result of a Command's execute method.
 * @type Void
 */
mozile.edit.done = function(state) {
	if(!state || !state.reversible) return;
	mozile.edit._undoStack = mozile.edit._undoStack.slice(0, mozile.edit._undoIndex + 1);
	mozile.edit._undoStack.push(state);
	mozile.edit._undoIndex = mozile.edit._undoStack.length - 1;
	mozile.edit.setCurrentState();
}

/**
 * Sets the current state. Uses the undoIndex to find the state in the undoStack.
 * @type Void
 */
mozile.edit.setCurrentState = function() {
	mozile.edit.currentState = mozile.edit._undoStack[mozile.edit._undoIndex];
}





/**** Global Commands ****/


/**
 * Shows document source.
 * @type mozile.edit.Command
 */
mozile.edit.save = new mozile.edit.Command("Save");
mozile.edit.save.accel = "Command-S";
mozile.edit.save.image = "silk/page_save";
mozile.edit.save.makesChanges = "none";
mozile.edit.save.watchesChanges = "state";
mozile.edit.addCommand(mozile.edit.save);

/**
 * Indicates that the command is available to be used.
 * @param {Event} event Optional. The current event object.
 * @type Boolean
 */
mozile.edit.save.isAvailable = function(event) {
	if(!mozile.save) return false;
	if(mozile.save.isSaved()) return false;
	else return true;
}

/**
 * Dumps the page source to a new window.
 * @type Object
 */
mozile.edit.save.execute = function(state, fresh) {
	mozile.save.save();
	
	state.reversible = false;
	state.executed = true;
	return state;
}

/**
 * Shows document source.
 * @type mozile.edit.Command
 */
mozile.edit.source = new mozile.edit.Command("Source");
//mozile.edit.source.accel = "Command-S";
mozile.edit.source.image = "silk/html";
mozile.edit.source.makesChanges = "none";
mozile.edit.source.watchesChanges = "none";
mozile.edit.addCommand(mozile.edit.source);

/**
 * Dumps the page source to a new window.
 * @type Object
 */
mozile.edit.source.execute = function(state, fresh) {
	if(mozile.save && mozile.gui){
		var content = mozile.save.getContent(document);
		content = mozile.save.cleanMarkup(content);
		mozile.gui.display("<h3>Page Source</h3>\n<pre>"+ content +"</pre>");
	}
	
	state.reversible = false;
	state.executed = true;
	return state;
}


/**
 * Shows debugging information.
 * @type mozile.edit.Command
 */
mozile.edit.debug = new mozile.edit.Command("Debug");
mozile.edit.debug.accel = "Command-D";
mozile.edit.debug.image = "silk/bug";
mozile.edit.debug.makesChanges = "none";
mozile.edit.debug.watchesChanges = "none";
mozile.edit.addCommand(mozile.edit.debug);

/**
 * Displays information used for testing and development.
 * @type Object
 */
mozile.edit.debug.execute = function(state, fresh) {
	mozile.debug.show();
	state.reversible = false;
	state.executed = true;
	return state;
}


/**** Undo ****/

/**
 * Reverses the action of the last command in the global undo stack.
 * @type mozile.edit.Command
 */
mozile.edit.undo = new mozile.edit.Command("Undo");
mozile.edit.undo.accel = "Command-Z";
mozile.edit.undo.image = "silk/arrow_undo";
mozile.edit.undo.makesChanges = "node";
mozile.edit.undo.watchesChanges = "state";
mozile.edit.addCommand(mozile.edit.undo);

/**
 * True if there are states to undo and the event matches the accelerator.
 * @param {Event} event Optional. The event object to be tested.
 * @type Boolean
 */
mozile.edit.undo.test = function(event) {
	if(mozile.edit._undoIndex < 0) return false;
	if(event) {
		return mozile.edit.checkAccelerator(event, this.accel);
	}
	return true;
}

/**
 * Indicates that the command is available to be used.
 * @param {Event} event Optional. The current event object.
 * @type Boolean
 */
mozile.edit.undo.isAvailable = function(event) {
	if(mozile.edit._undoIndex < 0) return false;
	else return true;
}

/**
 * Prepares a state object for the undo command.
 * @param {Event} event The event object to be converted into a state.
 * @param {Boolean} repeated Optional. Indicates that the undo operation is being repeated. Planned to be used to set the "freshness" of commands.
 * @type mozile.edit.State
 */
mozile.edit.undo.prepare = function(event, repeated) {
	var state = new mozile.edit.State(this, false); // don't store the selection
	
	state.repeated = false;
	if(repeated) state.repeated = repeated;
	if(event) state.repeated = event.repeat;

	state.reversible = false;
	return state;
}

/**
 * Undo the previous action on the global undo stack.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.undo.execute = function(state, fresh) {
	var undoState = mozile.edit._undoStack[mozile.edit._undoIndex];
	if(undoState) {
		// TODO: use "state.repeated" for freshness
		undoState.command.unexecute(undoState, false); 
		mozile.edit._undoIndex--;
		mozile.edit.setCurrentState();
		state.changesMade = undoState.changesMade;
	}
	state.executed = true;
	return state;
}


/**** Redo ****/

/**
 * Executes the current command in the global undo stack.
 * @type mozile.edit.Command
 */
mozile.edit.redo = new mozile.edit.Command("Redo");
mozile.edit.redo.accel = "Command-Shift-Z";
mozile.edit.redo.image = "silk/arrow_redo";
mozile.edit.redo.makesChanges = "node";
mozile.edit.redo.watchesChanges = "state";
mozile.edit.addCommand(mozile.edit.redo);

/**
 * True if there are states to redo and the event matches the accelerator.
 * @param {Event} event The event object to be tested.
 * @type Boolean
 */
mozile.edit.redo.test = function(event) {
	if(mozile.edit._undoIndex + 1 >= mozile.edit._undoStack.length) return false;
	if(event) {
		return mozile.edit.checkAccelerator(event, this.accel);
	}
	return true;
}

/**
 * Indicates that the command is available to be used.
 * @param {Event} event Optional. The current event object.
 * @type Boolean
 */
mozile.edit.redo.isAvailable = function(event) {
	if(mozile.edit._undoIndex + 1 >= mozile.edit._undoStack.length) return false;
	else return true;
}

/**
 * Prepares a state object for the redo command.
 * @param {Event} event The event object to be converted into a state.
 * @param {Boolean} repeated Optional. Indicates that the undo operation is being repeated. Planned to be used to set the "freshness" of commands.
 * @type mozile.edit.State
 */
mozile.edit.redo.prepare = function(event, repeated) {
	var state = new mozile.edit.State(this, false); // don't store the selection

	state.repeated = false;
	if(repeated) state.repeated = repeated;
	if(event) state.repeated = event.repeat;

	state.reversible = false;
	return state;
}

/**
 * Redo the previous action on the global undo stack.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.redo.execute = function(state, fresh) {
	var redoState = mozile.edit._undoStack[mozile.edit._undoIndex + 1];
	if(redoState) {
		mozile.edit._undoIndex++;
		// TODO: use "state.repeated" for freshness
		redoState.command.execute(redoState, false); 
		mozile.edit.setCurrentState();
		state.changesMade = redoState.changesMade;
	}
	state.executed = true;
	return state;
}


/**** Clipboard System ****/

/**
 * The local clipboard. Contains document fragments created using the copy and cut commands.
 * @type DocumentFragment
 */
mozile.edit.clipboard = null;

/**
 * Updates the local clipboard with data from the system clipboard, when possible.
 * @type Void
 */
mozile.edit.updateClipboard = function() {
	// TODO: Implement.
}


/**** Copy ****/

/**
 * Copies the current selection to the clipboard.
 * @type mozile.edit.Command
 */
mozile.edit.copy = new mozile.edit.Command("Copy");
mozile.edit.copy.accel = "Command-C";
mozile.edit.copy.image = "silk/page_copy";
mozile.edit.addCommand(mozile.edit.copy);

/**
 * Prepares a state object for the command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.copy.prepare = function(event) {
	var state = new mozile.edit.State(this, false); // don't store the selection
	state.reversible = false;
	state.cancel = false;
	return state;
}

/**
 * Copies the selected content. If rich editing is enabled, use range.toString() for text and range.cloneContents() for nodes. If rich editing is not enabled, always use range.toString().
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.copy.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	var range = selection.getRangeAt(0);
	
	if(range.commonAncestorContainer.nodeType == mozile.dom.TEXT_NODE ||
		!mozile.edit.rich) {
		mozile.edit.clipboard = range.toString();
	}
	else mozile.edit.clipboard = range.cloneContents();

	state.executed = true;
	return state;
}


/**** Cut ****/

/**
 * Copies the current selection to the clipboard, then removes it.
 * @type mozile.edit.Command
 */
mozile.edit.cut = new mozile.edit.Command("Cut");
mozile.edit.cut.accel = "Command-X";
mozile.edit.cut.image = "silk/cut";
mozile.edit.addCommand(mozile.edit.cut);

/**
 * True if there is a non-collapsed selection and the event matches the accelerator.
 * @param {Event} event The event object to be tested.
 * @type Boolean
 */
mozile.edit.cut.test = function(event) {
	if(event) {
		if(!event.editable) return false;
		if(!mozile.edit.checkAccelerator(event, this.accel)) return false;
		if(!mozile.edit.rich) {
			if(event.node) return false;
			if(event.node.nodeType != mozile.dom.TEXT_NODE) return false;
		}
	}

	return true;
}

/**
 * Copies the selected content. If rich editing is enabled, use range.toString() for text and range.cloneContents() for nodes. If rich editing is not enabled, always use range.toString().
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.cut.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	var range = selection.getRangeAt(0);
	state.actions = new Array();
	
	if(range.commonAncestorContainer.nodeType == mozile.dom.TEXT_NODE ||
		!mozile.edit.rich) {
		mozile.edit.clipboard = range.toString();
		mozile.edit.removeText.request(state, fresh);
	}
	else {
		mozile.edit.clipboard = range.cloneContents();
		mozile.edit.remove.request(state, fresh);
	}
	
	state.selection.after = selection.store();
	state.executed = true;
	return state;
}

/**
 * Restores removes content;
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.cut.unexecute = function(state, fresh) {
	var selection = mozile.dom.selection.get();

	// Unexecute all actions in reverse order.
	for(var i = state.actions.length - 1; i >= 0; i--) {
		state.actions[i] = state.actions[i].command.unexecute(state.actions[i], fresh);
		if(state.actions[i].executed) throw("Error: mozile.edit.paste.unexecute Child command unexecute failed at action "+ i +".");
	}

	selection.restore(state.selection.before);
	selection.collapseToEnd();
	state.executed = false;
	return state;
}



/**** Paste ****/

/**
 * Copies the current selection to the clipboard.
 * @type mozile.edit.Command
 */
mozile.edit.paste = new mozile.edit.Command("Paste");
mozile.edit.paste.accel = "Command-V";
mozile.edit.paste.image = "silk/page_paste";
mozile.edit.addCommand(mozile.edit.paste);

/**
 * True if there is a non-collapsed selection and the event matches the accelerator.
 * @param {Event} event The event object to be tested.
 * @type Boolean
 */
mozile.edit.paste.test = function(event) {
	if(!mozile.edit.clipboard) return false;

	if(event) {
		if(!event.editable) return false;
		if(!mozile.edit.checkAccelerator(event, this.accel)) return false;
		if(!mozile.edit.rich) {
			if(typeof(mozile.edit.clipboard) != "string") return false;
			if(event.node) return false;
			if(event.node.nodeType != mozile.dom.TEXT_NODE) return false;
		}
	}

	return true;
}

/**
 * Prepares a state object for the command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.paste.prepare = function(event) {
	var state = new mozile.edit.State(this);
	
	if(typeof(mozile.edit.clipboard) == "string") {
		state.content = mozile.edit.clipboard;
	}
	else state.content = mozile.edit.clipboard.cloneNode(true);
	
	state.reversible = true;
	return state;
}

/**
 * Pastes the content of the clipboard.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.paste.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	var range = selection.getRangeAt(0);
	state.actions = new Array();
	
	if(!selection.isCollapsed) {
		if(mozile.edit.remove) mozile.edit.remove.request(state, fresh);
		else mozile.edit.removeText.request(state, fresh);
	}

	// Text case	
	if(typeof(state.content) == "string") {
		mozile.edit.insertText.request(state, fresh, mozile.edit.NEXT, state.content);
	}

	// Rich case
	else {
		var previousSibling = null;
		var nextSibling = null;
		// Split text node
		if(selection.focusNode.nodeType == mozile.dom.TEXT_NODE) {
			mozile.edit.splitNode.request(state, fresh);
			previousSibling = selection.focusNode.previousSibling;
			nextSibling = selection.focusNode;
		}
		else {
			previousSibling = selection.focusNode.childNodes[selection.focusOffset - 1];
			nextSibling = selection.focusNode.childNodes[selection.focusOffset];
		}
	
		var parentNode = null;
		if(!previousSibling) parentNode = selection.focusNode.parentNode;
		var clone;
		
		// Insert first node (if there is more than one node).
		if(state.content.childNodes.length > 1) {
			clone = state.content.firstChild.cloneNode(true);
			if(previousSibling && previousSibling.nodeType == mozile.dom.TEXT_NODE &&
				clone.nodeType == mozile.dom.TEXT_NODE) {
				selection.collapse(previousSibling, previousSibling.data.length);
				mozile.edit.insertText.request(state, fresh, null, clone.data);
			}
			else {
				mozile.edit.insertNode.request(state, fresh, parentNode, previousSibling, clone);
				previousSibling = clone;
				parentNode = null;
			}
		}

		// Insert middle nodes.
		for(var i=1; i < state.content.childNodes.length - 1; i++) {
			clone = state.content.childNodes[i].cloneNode(true);
			mozile.edit.insertNode.request(state, fresh, parentNode, previousSibling, clone);
			previousSibling = clone;
			parentNode = null;
		}
		
		// Insert last node.
		clone = state.content.lastChild.cloneNode(true);
		if(nextSibling && nextSibling.nodeType == mozile.dom.TEXT_NODE &&
			clone.nodeType == mozile.dom.TEXT_NODE) {
				selection.collapse(nextSibling, 0);
				mozile.edit.insertText.request(state, fresh, null, clone.data);
		}
		else {
			mozile.edit.insertNode.request(state, fresh, parentNode, previousSibling, clone);
			var IP = mozile.edit.getInsertionPoint(clone, mozile.edit.PREVIOUS);
			if(IP) IP.select();
			else {
				IP = mozile.edit.getInsertionPoint(nextSibling, mozile.edit.NEXT);
				if(IP) IP.select();
				else mozile.debug.debug("mozile.edit.paste.execute", "Nowhere to collapse to.");
			}
		}
	}

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}

/**
 * Removes pasted content.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.paste.unexecute = function(state, fresh) {
	var selection = mozile.dom.selection.get();

	// Unexecute all actions in reverse order.
	for(var i = state.actions.length - 1; i >= 0; i--) {
		state.actions[i] = state.actions[i].command.unexecute(state.actions[i], fresh);
		if(state.actions[i].executed) throw("Error: mozile.edit.paste.unexecute Child command unexecute failed at action "+ i +".");
	}

	selection.restore(state.selection.before);
	selection.collapseToEnd();
	state.executed = false;
	return state;
}



/**
 * A command used for testing purposes only.
 * @type mozile.edit.Command
 */
mozile.edit.test = new mozile.edit.Command("Test");
mozile.edit.test.accel = "Escape";
mozile.edit.addCommand(mozile.edit.test);

/**
 * Displays information used for testing and development.
 * @type Object
 */
mozile.edit.test.execute = function(state, fresh) {
	mozile.require("mozile.util");
	var output = new Array();
	output.push("Debugging Information:");

	output.push("Undo: "+ mozile.edit._undoIndex +" / "+ mozile.edit._undoStack.length);
	
	var selection = mozile.dom.selection.get();
	output.push("Selection:\n"+ mozile.util.dumpValues(selection.store()));

	var element = selection.focusNode;
	if(element.nodeType != mozile.dom.ELEMENT_NODE) element = element.parentNode;
	var rng = mozile.edit.lookupRNG(element);
	if(rng) {
		if(rng.getName()) output.push("RNG: "+ rng +" "+ rng.getName());
		else output.push("RNG: "+ rng);
		output.push("Text? "+ rng.mayContain("text"));
	}
	else output.push("No matching RNG object.");

	alert(output.join("\n"));
	state.reversible = false;
	state.executed = true;
	return state;
}


/**
 * A tweaking command. For testing only
 * @type mozile.edit.Command
 */
mozile.edit.tweak = new mozile.edit.Command("Tweak");
mozile.edit.tweak.accel = "Command-E";
mozile.edit.addCommand(mozile.edit.tweak);

/**
 * Displays information used for testing and development.
 * @type Object
 */
mozile.edit.tweak.execute = function(state, fresh) {
	if(mozile.browser.isIE) {
		var selection = mozile.dom.selection.get();
		var range = selection.getRangeAt(0);
		range._range.move("character", 1);
		selection.removeAllRanges();
		selection.addRange(range);
	}
	
	state.reversible = false;
	state.executed = true;
	return state;
}





/**** General Commands ****/
mozile.require("mozile.edit.InsertionPoint");

/**
 * Inserts text into a text node.
 * @type mozile.edit.Command
 */
mozile.edit.navigateLeftRight = new mozile.edit.Command("NavigateLeftRight");
mozile.edit.navigateLeftRight.priority = 15;
mozile.edit.navigateLeftRight.accel = "Left Right";
mozile.edit.navigateLeftRight.accels = 
	mozile.edit.splitAccelerators(mozile.edit.navigateLeftRight.accel);
mozile.edit.navigateLeftRight.makesChanges = "none";
mozile.edit.navigateLeftRight.watchesChanges = "none";

/**
 * Prepares a state object for the insert text command.
 * @param {Event} event Optional. The event object to be converted into a state.
 * @param {Boolean} direction Optional. The direction to move in. Defautls to next.
 * @param {Boolean} extend Optional. When true the selection is extended. Otherwise it si collapsed.
 * @type mozile.edit.State
 */
mozile.edit.navigateLeftRight.prepare = function(event, direction, extend) {
	var state = new mozile.edit.State(this, false); // Don't store selection	
	
	state.direction = mozile.edit.NEXT;
	if(direction) state.direction = direction;
	else if(event && event.keyCode == 37) state.direction = mozile.edit.PREVIOUS;

	state.extend = false;
	if(extend) state.extend = extend;
	else if(event) state.extend = event.shiftKey;

	state.reversible = false; // This command is not undoable.
	return state;
}

/**
 * Moves the cursor to the next insertion point. If a range is selected, the range is collapsed.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.navigateLeftRight.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	
	// Move the selection.
	if(selection.isCollapsed || state.extend) {
		var IP = selection.getInsertionPoint();
		IP.seek(state.direction);
		if(state.extend) IP.extend();
		else IP.select();
		//alert(IP +"\n"+ mozile.util.dumpValues(selection.store()));
	}

	// Collapse the selection.
	else {
		if(state.direction == mozile.edit.NEXT) selection.collapseToEnd();
		else selection.collapseToStart();
	}

	state.executed = true;
	return state;
}




/**
 * Used to move thr cursor through the document.
 * @param {String} name The group's name.
 * @constructor
 */
mozile.edit.Navigate = function(name) { 
	/**
	 * The name for this command group.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command does not contain more commands.
	  * @type Boolean
	  */
	 this.group = false;
	 
	 /**
	  * Indicates that any contents of the selection should be removed before inserting.
	  * @type Boolean
	  */
	 this.remove = true;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.makesChanges = "none";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.watchesChanges = "none";
	 
	 /**
	  * Indicates the direction for the navigation. @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.direction = "next";
	 
	 /**
	  * Indicates how the selection should be collapsed. Can be null, "start" or "end".
	  * @type String
	  */
	 this.collapse = null;

	 // Register this command group on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}
mozile.edit.Navigate.prototype = new mozile.edit.Command;
mozile.edit.Navigate.prototype.constructor = mozile.edit.Navigate;


/**
 * Prepares a state object for the Split command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.Navigate.prototype.prepare = function(event) {
	var state = new mozile.edit.State(this);

	var target = mozile.edit._getTarget(event, this.target, this.direction);
	state.target = state.storeNode(target);

	// Some commands may need additional information from the user.
	// If so, define a method named "prompt", which takes the event and the state and returns a Boolean true when it succeeds.
	if(this.prompt) {
		if(!this.prompt(event, state)) return null;
	}

	state.reversible = false; // This command is not undoable.
	return state;
}

/**
 * Inserts a new element at the selection.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.Navigate.prototype.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();

	var target = mozile.xpath.getNode(state.target);
	var direction = mozile.edit.NEXT;
	if(this.direction == "previous") direction = mozile.edit.PREVIOUS;

	// Move the selection.
	var IP = mozile.edit.getInsertionPoint(target, direction);
	if(IP) {
		IP.select();
		IP = mozile.edit.getInsertionPoint(target, -1 * direction);
		if(IP) IP.extend();
		if(this.collapse == "start") selection.collapseToStart();
		else if(this.collapse == "end") selection.collapseToEnd();
		
		// Scroll to the new selection.
		var x = mozile.dom.getX(target);
		var y = mozile.dom.getY(target);
		var pX = window.pageXOffset;
		var pY = window.pageYOffset;
		//alert([x, y, pX, pY, window.innerWidth, window.innerHeight,
		//	pX + window.innerWidth, pY + window.innerHeight].join("\n"));
		if(x < pX || x > (pX + window.innerWidth) ||
			y < pY || y > (pY + window.innerHeight) ) {
			window.scroll(x-50, y-100);
		}
	}
	//else alert("Could not get IP in "+ state.target);

	state.executed = true;
	return state;
}


/**
 * Inserts text into a text node.
 * @type mozile.edit.Command
 */
mozile.edit.insertText = new mozile.edit.Command("InsertText");
mozile.edit.insertText.priority = 10;
mozile.edit.insertText.makesChanges = "text";
mozile.edit.insertText.watchesChanges = "none";

/**
 * True if the event was a keypress of a non-control and non-arrow character.
 * @param {Event} event Optional. The event object to be tested.
 * @param {Integer} direction Optional. The direction of the insertion. Defaults to next.
 * @param {String} content Optional. The content string to be inserted.
 * @param {Text} node Optional. A text node which will have its data replace with "content".
 * @type Boolean
 */
mozile.edit.insertText.test = function(event, direction, content, node) {
	if(event) {
		if(event.type != "keypress") return false;
		if(event.ctrlKey || event.metaKey) return false;
		if(!mozile.os.isMac && event.altKey) return false;

		// Special case: spaces
		// Don't insert a second consecutive space unless there is an alternateSpace
		if(!node && event.charCode == 32 && !mozile.alternateSpace) {
			var range = event.range;
			if(!range) range = mozile.dom.selection.get().getRangeAt(0);
			if(range.startContainer.nodeType == mozile.dom.TEXT_NODE) {
				if(range.startContainer.data.charAt(range.startOffset-1) == " ") {
					return false;
				}
			}
		}

		// Accept non-control characters
		if(event.charCode && event.charCode >= 32) {
			if(mozile.browser.isSafari && 
				event.charCode >= 63232 && 
				event.charCode <= 63235) return false;
			return true;
		}
		return false;
	}
	else {
		if(typeof(content) != "string") return false;
		//if(content.length < 1) return false;
	}

	return true;
}

/**
 * Prepares a state object for the insert text command.
 * @param {Event} event The event Optional. object to be converted into a state.
 * @param {Integer} direction Optional. The direction of the insertion. Defaults to next.
 * @param {String} content Optional. The content string to be inserted.
 * @param {Text} node Optional. A text node which will have its data replace with "content".
 * @type mozile.edit.State
 */
mozile.edit.insertText.prepare = function(event, direction, content, node) {
	var state = new mozile.edit.State(this);

	state.direction = mozile.edit.NEXT;
	if(direction) state.direction = direction;

	state.content = " ";
	if(content) state.content = content;
	else if(event) state.content = String.fromCharCode(event.charCode);
	
	state.node = state.storeNode(node);

	state.remove = false;

	// Handle special cases with spaces.
	var selection = null;
	if(event && event.selection) selection = event.selection;
	else selection = mozile.dom.selection.get();
	if(mozile.alternateSpace && !state.node && state.content == " ") {
		var range = selection.getRangeAt(0);
		var alt = mozile.alternateSpace;
		var text = range.startContainer;
		var offset = range.startOffset;

		// Get the next and previous characters. Determine if the preceeding character is an alternateSpace.
		var nextChar = null;
		if(range.endContainer.nodeType == mozile.dom.TEXT_NODE)
			nextChar = range.endContainer.data.charAt(range.endOffset);
		var previousChar = null;
		var previousAlt = false;
		if(text.nodeType == mozile.dom.TEXT_NODE) {
			previousChar = text.data.charAt(offset-1);
			var data = text.data.substring(0, offset);
			if(offset && data.lastIndexOf(alt) + alt.length == offset) {
				previousAlt = true;
				previousChar = text.data.charAt(offset - alt.length - 1);
			}
		}

		// Set the content, based on the variables we've collected.
		//alert(previousAlt +" '"+ previousChar +"' '"+ nextChar +"'");
		if(previousAlt) {
			if(previousChar && previousChar != " ") {
				state.remove = true;
				state.content = " " + alt;
			}
			else if(!nextChar || nextChar == " ") state.content = alt;
			// else insert the space
		}
		else if(!nextChar || nextChar == " ") state.content = alt;
		else if(!previousChar || previousChar == " ") state.content = alt;
		//alert("'"+state.content+"'");
	}

	return state;
}

/**
 * Inserts text at the current selection.
 * If the selection is not collapsed, the range is removed first. 
 * Uses the DOM Text.insertData() method.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.insertText.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	state.emptyToken = false;
	state.actions = new Array();
	
	// Remove a non-collapsed range.
	if(!state.node && !selection.isCollapsed) {
		if(mozile.edit.remove) mozile.edit.remove.request(state, fresh, state.direction, null, true);
		else mozile.edit.removeText.request(state, fresh, state.direction);
	}
	
	// If a node was given, insert the data there.
	if(state.node) {
		var node = mozile.xpath.getNode(state.node);
		state.changedNode = node;
		state.oldData = node.data;
		node.data = state.content;
		if(state.direction == mozile.edit.NEXT) {
			selection.collapse(node, 0);
			selection.extend(node, node.data.length);
		}
		else {
			selection.collapse(node, node.data.length);
			selection.extend(node, 0);
		}
		state.selection.after = selection.store();
	}
	
	// If this node is an empty token, replace the token with the data.
	else if(mozile.edit.isEmptyToken(selection.focusNode)) {
		state.emptyToken = true;
		selection.focusNode.data = state.content;
		selection.collapse(selection.focusNode, selection.focusNode.data.length);
		state.selection.after = selection.store();
	}
	
	// If this is not a text node, insert one.
	else if(selection.focusNode.nodeType != mozile.dom.TEXT_NODE) {
		state.newNode = document.createTextNode(state.content);
		if(selection.focusOffset == 0) mozile.dom.prependChild(state.newNode, selection.focusNode);
		else selection.focusNode.insertBefore(state.newNode, selection.focusNode.childNodes[selection.focusOffset]);
		selection.collapse(state.newNode, state.newNode.data.length);
		state.selection.after = selection.store();
	}

	// Otherwise insert the text into this text node.
	else {	
		var text = selection.focusNode;
		var offset = selection.focusOffset;
		if(state.remove) {
			mozile.edit.removeText.request(state, fresh, -1 * state.direction, mozile.alternateSpace);
			offset -= mozile.alternateSpace.length;
		}
		text.insertData(offset, state.content);
		var newOffset = offset + (state.direction * state.content.length);
		//if(!mozile.browser.isIE) selection.collapse(selection.focusNode, newOffset);
		selection.collapse(text, newOffset);
		if(state.actions.length == 0) 
			state.selection.after = selection.store(state.selection.before, newOffset);
		else state.selection.after = selection.store();
	}

	state.executed = true;
	return state;
}

/**
 * Removes inserted text and restores any removed range.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.insertText.unexecute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.after);

	// Remove empty token, new node, or text.
	if(state.changedNode) state.changedNode.data = state.oldData;
	else if(state.emptyToken) selection.focusNode.data = mozile.emptyToken;
	else if(state.newNode) state.newNode.parentNode.removeChild(state.newNode);
	else selection.focusNode.deleteData(selection.focusOffset - state.content.length, state.content.length);

	// Unexecute any other actions in reverse order.
	for(var i = state.actions.length - 1; i >= 0; i--) {
		state.actions[i] = state.actions[i].command.unexecute(state.actions[i], fresh);
		if(state.actions[i].executed) throw("Error: mozile.edit.inertText.unexecute Child command unexecute failed at action "+ i +".");
	}
	
	
	selection.restore(state.selection.before);
	state.executed = false;
	return state;
}




/**
 * Removes text from a node.
 * @type mozile.edit.Command
 */
mozile.edit.removeText = new mozile.edit.Command("RemoveText");
mozile.edit.removeText.priority = 10;
mozile.edit.removeText.makesChanges = "text";
mozile.edit.removeText.watchesChanges = "none";

/**
 * True if the selection is text, if the event was a keypress of the backspace key or delete key, and if the operation won't take the selection out of the current node.
 * @param {Event} event Optional. The event object to be tested.
 * @param {Integer} direction Optional. The direction for the removal to use. Defaults to previous.
 * @param {String} content Optional. The content removed.
 * @type Boolean
 */
mozile.edit.removeText.test = function(event, direction, content) {
	var dir;
	if(event) {
		if(mozile.edit.checkAccelerator(event, "Backspace")) {
			dir = mozile.edit.PREVIOUS;
		}
		if(mozile.edit.checkAccelerator(event, "Delete")) {
			dir = mozile.edit.NEXT;
		}
		if(!dir) return false;	
	}
	
	if(!dir) dir = mozile.edit.PREVIOUS;
	if(direction) dir = direction;

	var selection;
	if(event && event.selection) selection = event.selection;
	if(!selection) selection = mozile.dom.selection.get();
	if(!selection) return false;

	var node;
	if(event && event.node) node = event.node;
	if(!node) {
		var range;
		if(event && event.range) range = event.range;
		if(!range) range = selection.getRangeAt(0);
		if(!range) return false;
		node = range.commonAncestorContainer;
		if(event) {
			event.selection = selection;
			event.range = range;
			event.node = node;
		}
	}
	if(!node) return false;
	if(node.nodeType != mozile.dom.TEXT_NODE) return false;

	// Make sure the removal will stay within the text node.
	if(event) {
		if(mozile.edit.remove) {
			if(node.data.length <= 2) return false;
			if(!selection.isCollapsed) return false;
		}
		else if(!selection.isCollapsed) return true;

		if(dir == mozile.edit.PREVIOUS &&
			selection.focusOffset > 0) return true;
		if(dir == mozile.edit.NEXT &&
			selection.focusOffset < node.data.length) return true;
		return false;
	}
	else return true;
}


/**
 * Prepares a state object for the remove text command.
 * @param {Event} event The event Optional. object to be converted into a state.
 * @param {Integer} direction Optional. The direction for the removal to use. Defaults to previous.
 * @param {String} content Optional. The content removed.
 * @type mozile.edit.State
 */
mozile.edit.removeText.prepare = function(event, direction, content) {
	var state = new mozile.edit.State(this);
	
	state.direction = mozile.edit.PREVIOUS;
	if(direction) state.direction = direction;
	else if(event && mozile.edit.convertKeyCode(event.keyCode) == "Delete")
		state.direction = mozile.edit.NEXT;

	state.content = null;
	if(content) state.content = content;

	return state;
}

/**
 * Removes text at the current selection. Stores the removed text so that the operation can be undone. 
 * If the selection is collapsed, the state.direction is used to deleted the next character in that direction.
 * If the selection is not collapsed, all text inside the range is removed.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.removeText.execute = function(state, fresh) {
	//alert("Removing Text...");
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	if(!state.direction) state.direction = mozile.edit.PREVIOUS;
	
	if(selection.isCollapsed) {
		var firstOffset = selection.focusOffset;
		var secondOffset = selection.focusOffset;
		if(!state.content) {
			var IP = selection.getInsertionPoint();
			IP.seek(state.direction);
			if(state.direction == mozile.edit.PREVIOUS) firstOffset = IP.getOffset();
			else secondOffset = IP.getOffset();
			state.content = selection.focusNode.data.substring(firstOffset, secondOffset);
		}
		else {
			if(state.direction == mozile.edit.PREVIOUS)
				firstOffset -= state.content.length;
			else secondOffset += state.content.length;
		}
		selection.focusNode.deleteData(firstOffset, state.content.length);
		selection.collapse(selection.focusNode, firstOffset);
		state.selection.after = selection.store(state.selection.before, firstOffset);
	}

	else {
		var range = selection.getRangeAt(0);
		// Hack to compensate for a problem translating IE TextRanges to Ranges in the case of adjacent text nodes.
		if(mozile.browser.isIE && range.startContainer != range.endContainer) {
			if(range.endOffset == 0) range.setEnd(range.startContainer, range.startContainer.data.length);
			else range.setStart(range.endContainer, 0);
		}
		//alert(mozile.util.dumpValues(range.store()));
		state.content = range.startContainer.data.substring(range.startOffset, range.endOffset);
		range.startContainer.deleteData(range.startOffset, range.endOffset - range.startOffset);
		selection.collapse(range.startContainer, range.startOffset);
		state.selection.after = selection.store(state.selection.before, range.startOffset);
	}

	state.executed = true;
	//alert("Removed Text");
	return state;
}

/**
 * Restores removed text.
 * @param {mozile.edit.State} state The state information needed to unexecute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.removeText.unexecute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.after);

	if(!selection || !selection.focusNode) throw("Error: mozile.edit.removeText.unexecute no selection.focusNode");
	selection.focusNode.insertData(selection.focusOffset, state.content);

	//var newOffset = selection.focusOffset;
	//if(state.direction == mozile.edit.PREVIOUS) newOffset += state.content.length;
	//Optimize for IE? if(!mozile.browser.isIE || mozile.edit.NEXT) selection.collapse(selection.focusNode, newOffset);
	//selection.collapse(selection.focusNode, newOffset);
	selection.restore(state.selection.before);
	state.executed = false;
	return state;
}




/**** Detection Methods ****/


/**
 * A temporary hack to check whether a node is a block level element or not.
 * TODO: Replace with an RNG based method.
 * @param {Node} node
 * @type Boolean
 */
mozile.edit.isBlock = function(node) {
	if(!node) return false;
	if(node.nodeType != mozile.dom.ELEMENT_NODE) return false;
	var display = mozile.dom.getStyle(node, "display");
	switch(display) {
		// TODO: Include more cases
		case "block":
		case "list-item":
			return true;
	}
	return false;
}


/**
 * Returns the node if it is a block, or the first ancestor which is a block.
 * @param {Node} node
 * @type Element
 */
mozile.edit.getParentBlock = function(node) {
	while(node) {
		if(mozile.edit.isBlock(node)) return node;
		else node = node.parentNode;
	}
	return null;
}


/**
 * Checks a node to see if it is editable.
 * A text node is editable only if the RNG object associated with its parent element allows text nodes.
 * An element is editable as long as it is allowed to have child elements.
 * All other node types are not editable.
 * @param {Node} node The node to check.
 * @type Boolean
 */
mozile.edit.isNodeEditable = function(node) {
	if(!node) return false;
	if(node.nodeType == mozile.dom.TEXT_NODE) {
		var rng = mozile.edit.lookupRNG(node);
		if(rng) return rng.mayContain("text");
		else return true;
	}
	else if(node.nodeType == mozile.dom.ELEMENT_NODE) {
		var rng = mozile.edit.lookupRNG(node);
		if(rng) return rng.mayContain("text");
		else {
			mozile.debug.debug("mozile.edit.isNodeEditable", "No RNG Element for element named '"+ node.nodeName +"'.");
			// TODO: Decide on default behaviour when no RNG Element is found.
			return true;
		}
	}
	return false;
}


/**
 * A temporary hack to check whether a node can have child nodes.
 * TODO: Replace with an RNG based method.
 * @param {Node} node
 * @type Boolean
 */
mozile.edit.isChildless = function(node) {
	if(node.nodeType == mozile.dom.COMMENT_NODE) return true;
	if(node.nodeType != mozile.dom.ELEMENT_NODE) return false;
	
	var rng = mozile.edit.lookupRNG(node);
	if(rng) {
		if(rng.mayContain("element")) return false;
		else return true;
	}
	else return false;
}

/**
 * Creates an empty token node.
 * @type Text
 */
mozile.edit.createEmptyToken = function() {
	return document.createTextNode(mozile.emptyToken);
}

/**
 * Determines whether a node is an "empty token" instance. That is, is it a text node which contains only the mozile.emptyToken character(s)?
 * @param {Node} node The text node to check.
 * @type Boolean
 */
mozile.edit.isEmptyToken = function(node) {
	if(node && node.nodeType == mozile.dom.TEXT_NODE &&
		node.data == mozile.emptyToken) return true;
	else return false;
}

/**
 * Determines whether a text node ends with an empty token instance.
 * @param {Node} node The text node to check.
 * @param {Integer} offset Optional. An offset within the text node. The method will look for an empty token immediately after this offset. If none is given, the method searches for any empty token. 
 * @type Boolean
 */
mozile.edit.containsEmptyToken = function(node, offset) {
	if(!node || node.nodeType != mozile.dom.TEXT_NODE) return false;
	if(offset == undefined || Number(offset)) {
		if(node.data.indexOf(mozile.emptyToken) > -1) return true;
		else return false;
	}
	else {
		var data = node.data.substring(offset);
		if(data.indexOf(mozile.emptyToken) == 0) return true;
		else return false;
	}
}

/**
 * Determines whether a node is empty: it contains no non-white-space text and no empty tokens inside any of its children.
 * @param {Node} node The node to check.
 * @type Boolean
 */
mozile.edit.isEmpty = function(node) {
	switch(node.nodeType) {
		case mozile.dom.TEXT_NODE:
			if(node.data.match(/\S/)) return false;
			if(mozile.edit.isEmptyToken(node)) return false;
			return true;

		case mozile.dom.ELEMENT_NODE:
			var children = node.childNodes;
			var i=0;
			// Check text nodes.
			for(i=0; i < children.length; i++) {
				if(children[i].nodeType == mozile.dom.TEXT_NODE &&
					!mozile.edit.isEmpty(children[i]) ) 
					return false;
			}
			// Check element nodes.
			for(i=0; i < children.length; i++) {
				if(children[i].nodeType == mozile.dom.ELEMENT_NODE &&
					!mozile.edit.isEmpty(children[i]) ) 
					return false;
			}
			return true;

		default: 
			return true;
	}
}



/**** Support Methods ****/

/**
 * Finds the name of the command's element.
 * @private
 * @param {mozile.edit.Command} command The calling command.
 * @type Element
 */
mozile.edit._getElementName = function(command) {
	var elementName;
	if(typeof(command.element) == "string") elementName = command.element;
	else if(command.element && command.element.cloneNode)
		elementName = mozile.dom.getLocalName(command.element);
	elementName = elementName.toLowerCase();
	return elementName;
}

/**
 * A function shared by several commands gets the current selection, range, and commonAncestorContainer.
 * @private
 * @param {Event} event Optional. The current event.
 * @type Element
 */
mozile.edit._getNode = function(event) {
	var node;
	if(event && event.node) node = event.node;
	if(!node) {
		var selection;
		if(event && event.selection) selection = event.selection;
		if(!selection) selection = mozile.dom.selection.get();
		if(!selection) return false;
		var range;
		if(event && event.range) range = event.range;
		if(!range) range = selection.getRangeAt(0);
		if(!range) return false;
		node = range.commonAncestorContainer;
		if(event) {
			event.selection = selection;
			event.range = range;
			event.node = node;
		}
	}

	if(!node) return null;
	else return node;
}


/**
 * A function shared by several commands which finds a target element.
 * The technique is to generate test function from the given target string, and then to iterate through nodes in the given direction, testing each node with the new test function until a positive result is found.
 * @private
 * @param {Event} event Optional. The current event.
 * @param target A string specifying the target, or a function which returns the target. Strings can be: "any", "text", "element", "block" or "localname tagName".
 * @param {String} direction Optional. A string specifying the direction to search in. Can be "ancestor" (the default), "descendant", "next", or "previous".
 * @type Element
 */
mozile.edit._getTarget = function(event, target, direction) {
	var node = mozile.edit._getNode(event);
	if(!node) return null;
	var test, result;
	
	// Set the direction.
	if(!direction) direction = "ancestor";
	if(direction != "ancestor" && direction != "descendant" &&
		direction != "next" && direction != "previous") {
		mozile.debug.debug("mozile.edit._getTarget", "Invalid direction '"+ direction +"'.");
		return null;
	}

	// Check the target. Either get a result or generate a test function.
	if(typeof(target) == "function") {
		result = target(event, null);
	}
	else if(typeof(target) == "string") {
		// "Any" case
		if(target.toLowerCase() == "any") {
			test = function(node) {
				if(node) return true;
				else return false;
			}
		}

		// Text case
		else if(target.toLowerCase() == "text") {
			test = function(node) {
				if(node.nodeType == mozile.dom.TEXT_NODE) return true;
				else return false;
			}
		}

		// Element case
		else if(target.toLowerCase() == "element") {
			test = function(node) {
				if(node.nodeType == mozile.dom.ELEMENT_NODE) return true;
				else return false;
			}
		}
		
		// Block case
		else if(target.toLowerCase() == "block") {
			test = function(node) {
				if(mozile.edit.isBlock(node)) return true;
				else return false;
			}
		}
		
		// LocalName case
		else if(target.toLowerCase().indexOf("localname") == 0) {
			var name = target.substring(9);
			var match = name.match(/\W*(\w+)\W*/);
			if(match && match[1]) {
				name = match[1].toLowerCase();
				test = function(node) {
					var localName = mozile.dom.getLocalName(node);
					if(localName && localName.toLowerCase() == name) return true;
					else return false;
				}
			}
			else return null;
		}
		
		// Unknown case
		else return null;
	}
	else return null;
	
	// Run the test function. Iterate in the given direction.
	var treeWalker;
	if(test && !result) {
		if(direction != "ancestor" && !treeWalker) {
			var root = document.documentElement;
			if(direction == "descendant") {
				root = node;
				if(root.nodeType != mozile.dom.ELEMENT_NODE) root = root.parentNode;
				direction = "next";
			}
			treeWalker = document.createTreeWalker(root, mozile.dom.NodeFilter.SHOW_ALL, null, false);
			treeWalker.currentNode = node;
		}
		
		var startNode = node;
		while(node) {
			if(direction == "next") node = treeWalker.nextNode();
			else if(direction == "previous") {
				node = treeWalker.previousNode();
				if(mozile.dom.isAncestorOf(node, startNode)) continue;
			}
			//alert(mozile.xpath.getXPath(node));
			// TODO: We want to make sure that the node is editable,
			// but I'm not sure this is thr right way to do it.
			if(node && test(node) && mozile.edit.isEditable(node) &&
				mozile.edit.getInsertionPoint(node, mozile.edit.NEXT)) {
				result = node;
				break;
			}
			if(direction == "ancestor") node = node.parentNode;
		}
	}

	if(result) return result;
	else return null;
}




/** 
 * An insertion point is the pair of either a) a text node and an offset within the text, or b) an element and an offset among its child nodes. This corresponds to the method used to denote points by the Selection and Range objects. 
 * @constructor
 * @param {Node} node
 * @param {Integer} offset The offset within the node.
 */
mozile.edit.InsertionPoint = function(node, offset) {
	/**
	 * Stores the current node.
	 * @private
	 */ 
	this._node = node;

	/**
	 * Stores the current offset.
	 * @private
	 */
	this._offset = offset;
}

// Define some regular expressions.
/**
 * Matches whitespace at the beginning of a string.
 * @private
 * @type RegExp
 */
mozile.edit.InsertionPoint.prototype._matchLeadingWS = /^(\s*)/;

/**
 * Matches whitespace at the end of a string.
 * @private
 * @type RegExp
 */
mozile.edit.InsertionPoint.prototype._matchTrailingWS = /(\s*)$/;

/**
 * Matches any non-whitespace character.
 * @private
 * @type RegExp
 */
mozile.edit.InsertionPoint.prototype._matchNonWS = /\S/;

/**
 * Gets the current node.
 * @type Node
 */
mozile.edit.InsertionPoint.prototype.getNode = function() { return this._node; }

/**
 * Gets the offset in the current node.
 * @type Integer
 */
mozile.edit.InsertionPoint.prototype.getOffset = function() { 
	if(this._offset < 0) this._offset = 0;
	// TODO: Handle too-long case.
	return this._offset; 
}

/**
 * Returns a string representation of the IP.
 * @type Integer
 */
mozile.edit.InsertionPoint.prototype.toString = function() { 
	return "Insertion Point: "+ mozile.xpath.getXPath(this._node) +" "+ this._offset;
}


/**
 * Collapses the selection to the current IP.
 * @type Void
 */
mozile.edit.InsertionPoint.prototype.select = function() {
	try {
		var selection = mozile.dom.selection.get();
		selection.collapse(this.getNode(), this.getOffset());
	} catch(e) {
		mozile.debug.debug("mozile.edit.InsertionPoint.prototype.select", "Bad collapse for IP "+ mozile.xpath.getXPath(this.getNode()) +" "+ this.getOffset() +"\n"+ mozile.dumpError(e));
	}
}

/**
 * Extends the selection to the IP.
 * @type Void
 */
mozile.edit.InsertionPoint.prototype.extend = function() {
	try {
		var selection = mozile.dom.selection.get();
		selection.extend(this.getNode(), this.getOffset());
	} catch(e) { 
		mozile.debug.debug("mozile.edit.InsertionPoint.prototype.extend", "Bad extend for IP "+ mozile.xpath.getXPath(this.getNode()) +" "+ this.getOffset() +"\n"+ mozile.dumpError(e));
	}
}

/**
 * Sets the node and offset to the next insertion point.
 * @type Void
 */
mozile.edit.InsertionPoint.prototype.next = function() {
	this.seek(mozile.edit.NEXT);
}

/**
 * Sets the node and offset to the previous insertion point.
 * @type Void
 */
mozile.edit.InsertionPoint.prototype.previous = function() {
	this.seek(mozile.edit.PREVIOUS);
}

/**
 * Sets the node and offset to the next insertion point.
 * <p>If the node is not a text node, or the offset and direction will mean that the IP leaves the node, then seekNode is returned instead.
 * <p>Otherwise we are inside a text node and have to worry about the XML white space rules. We want to treat adjacent whitespace as a single character. So we measure the length of the whitespace after the offset (if any). Then "moveBy" is set based on the length of the result and the CSS white-space mode. If the length takes the offset to the end of the node, seekNode is called.
 * @param {Integer} direction A coded integer. Can be NEXT (1) or PREVIOUS (-1).
 * @type Void
 */
mozile.edit.InsertionPoint.prototype.seek = function(direction) {
	var node = this.getNode();
	var offset = this.getOffset();
	if(!node || typeof(offset) == "undefined") return false;

	if(node.nodeType != mozile.dom.TEXT_NODE ||
		(direction == mozile.edit.PREVIOUS && offset == 0) ||
		(direction == mozile.edit.NEXT && offset == node.data.length) ||
		(direction == mozile.edit.NEXT && mozile.edit.isEmptyToken(node)) ) {
		return this.seekNode(direction);
	}
	else offset = offset + direction;
	if(!node || typeof(offset) == "undefined") return false;

	// Move to the leftmost position in an empty token.
	if(mozile.edit.isEmptyToken(node)) {
		this._offset = 0;
		return true;
	}

	// Measure the length of the white-space and the distance to the first alternateSpace token. 
	var content = node.data;
	var substring, result, altSpaceIndex;
	if(direction == mozile.edit.NEXT) {
		substring = content.substring(this.getOffset());
		result = substring.match(this._matchLeadingWS);
		if(mozile.alternateSpace) 
			altSpaceIndex = substring.indexOf(mozile.alternateSpace);
	}
	else {
		substring = content.substring(0, this.getOffset());
		result = substring.match(this._matchTrailingWS);
		if(mozile.alternateSpace) {
			altSpaceIndex = substring.length;
			altSpaceIndex -= substring.lastIndexOf(mozile.alternateSpace) + 1;
		}
	}
	// Use the smallest length as wsLength.
	var wsLength = result[0].length;
	if(Number(altSpaceIndex) != NaN && altSpaceIndex > -1 &&
		altSpaceIndex < wsLength) {
		wsLength = altSpaceIndex;
	}

	// Skip over white space, if necessary.	
	var moveBy = 0;
	if(wsLength < 2) moveBy = direction;
	else if(mozile.dom.getStyle(node.parentNode, "white-space").toLowerCase() == "pre") moveBy = direction;
	else if(wsLength < substring.length) moveBy = wsLength * direction;
	else if(wsLength == substring.length) {
		return this.seekNode(direction);
	}
	else throw Error("Unhandled case in InsertionPoint.seek()");

	this._node = node;
	this._offset = this.getOffset() + moveBy;
	return true;
}

/**
 * Seeks the next node which allows text to be inserted.
 * <p>The method involves a treeWalker, but unfortunately the setup section is complicated. The chief cause of the complexity is that, if the node is a first child, then we do not want the parentNode but the parentNode's previousNode.
 * <p>The method is to build a treeWalker, set the currentNode to this.getNode(), and move through the tree working as follows:
 * <ul>
 *   <li>Seeks the next text node, unless it finds two consecutive non edtiable elements (comments are optional), in which case the insertion point is inserted between them.
 * </ul>
 * @param {Integer} direction A coded integer. Can be NEXT (1) or PREVIOUS (-1).
 * @param {Boolean} extraStep Optional. Defaults to "true". When "false" the method will not move an extra offset step.
 * @type Void
 */
mozile.edit.InsertionPoint.prototype.seekNode = function(direction, extraStep) {
	if(extraStep !== false) extraStep = true;
	var node = this.getNode();
	if(!node) return false;

	var startNode = node;

	// Setup
	var container = mozile.edit.getContainer(node);
	if(!container) container = document.documentElement;
	var treeWalker = document.createTreeWalker(container, mozile.dom.NodeFilter.SHOW_ALL, null, false);
	// Setup text node
	if(node.nodeType == mozile.dom.TEXT_NODE) {
		treeWalker.currentNode = node;
		if(direction == mozile.edit.NEXT) node = treeWalker.nextNode();
		// Make sure the previousNode isn't just the parent node.
		else {
			var tempNode = node;
			node = treeWalker.previousNode();
			while(node && node.firstChild == tempNode) {
				tempNode = node;
				node = treeWalker.previousNode();
			}
		}
	}
	// Setup element
	else {
		node = node.childNodes[this.getOffset()];
		if(node) {
			startNode = node;
			treeWalker.currentNode = node;
		}
		// Handle offset at the end of a non-empty element.
		else if(this.getOffset() > 0 &&
		  this.getOffset() == this.getNode().childNodes.length &&		
		  this.getNode().childNodes[this.getOffset() - 1]) {
		  node = this.getNode().childNodes[this.getOffset() - 1];
		  startNode = node;
		  treeWalker.currentNode = node;
		}
		// Handle empty node case.
		else {
			node = this.getNode(); 
			startNode = node;
			treeWalker.currentNode = node;
			if(direction == mozile.edit.NEXT) node = treeWalker.nextNode();
			// Make sure the previousNode isn't just the parent node.
			else {
				var tempNode = node;
				node = treeWalker.previousNode();
				while(node && node.firstChild == tempNode) {
					tempNode = node;
					node = treeWalker.previousNode();
				}
			}
		}
	}
	if(!node) {
		mozile.debug.debug("mozile.edit.InsertionPoint.prototype.seekNode", "Lost node.");
		return false;
	}

	// Seek the next eligbile node.
	var IP;	
	var offset = null;
	var lastNode = node;
	var skippedOne = false;
	while(node) {
		//alert(mozile.xpath.getXPath(node));
		
		// Get an Insertion Point.
		IP = mozile.edit.getInsertionPoint(node, direction);
		if(IP) {
			this._node = IP.getNode();
			this._offset = IP.getOffset();
			// When entering inline elements or passing comments move an extra step.
			if( extraStep &&
				(mozile.edit.getParentBlock(node) == mozile.edit.getParentBlock(startNode) && mozile.edit.isNodeEditable(lastNode)) ||
				(lastNode.nodeType == mozile.dom.COMMENT_NODE && node == IP.getNode()) )
				this._offset = this._offset + direction;
			if(mozile.edit.isEmptyToken(IP.getNode())) this._offset = 0;
			return true;
		}

		// If this node allows text insertion, look ahead to the next sibling.
		else if(node.nodeType == mozile.dom.ELEMENT_NODE && 
			mozile.edit.isNodeEditable(node.parentNode) ) {
			var nextNode = node;
			while(nextNode) {
				if(direction == mozile.edit.NEXT) nextNode = nextNode.nextSibling;
				else nextNode = nextNode.previousSibling;
				if(nextNode && nextNode.nodeType == mozile.dom.COMMENT_NODE) continue;
				else break;
			}
			
			// If it does not have an insertion point, then make an IP here.
			IP = mozile.edit.getInsertionPoint(nextNode, direction);
			if(!IP) {
				var newNode = node;
				if(direction == mozile.edit.PREVIOUS) {
					if(nextNode.parentNode == this._node && !skippedOne) {
						skippedOne = true; 
						lastNode = node;
						node = nextNode;
						treeWalker.currentNode = node;
						continue;
					}
					newNode = nextNode;
				}
				this._node = newNode.parentNode;
				this._offset = mozile.dom.getIndex(newNode) + 1;
				//if(direction == mozile.edit.NEXT) this._offset++;
				return true;
			}
		}

		lastNode = node;
		if(direction == mozile.edit.NEXT) node = treeWalker.nextNode();
		else node = treeWalker.previousNode();
	}
	
	return false;
}








/**
 * Create a new insertion point using the selection's focus.
 * @type mozile.edit.InsertionPoint
 */
mozile.dom.Selection.prototype.getInsertionPoint = function() {
	if(!this.focusNode || this.focusOffset == null) return null;
	else return new mozile.edit.InsertionPoint(this.focusNode, this.focusOffset);
}

// Copy this method to the IE Selection object.
if(mozile.dom.InternetExplorerSelection) {
	/**
	 * Create a new insertion point using the selection's focus.
	 * @type mozile.edit.InsertionPoint
	 */
	mozile.dom.InternetExplorerSelection.prototype.getInsertionPoint = mozile.dom.Selection.prototype.getInsertionPoint;
}


/**
 * Get the first insertion point in the given node and the given direction.
 * If the direction is "next" then the first point is returned. If the direction is "previous" then the last point us returned.
 * @param {Node} node The node to search for an insertion point.
 * @param {Integer} direction A coded integer. Can be NEXT (1) or PREVIOUS (-1).
 * @param {Boolean} force Optional. When true the fact that the given node is not editable is ignored.
 * @type mozile.edit.InsertionPoint
 */
mozile.edit.getInsertionPoint = function(node, direction, force) {
	if(!node) return false;
	var offset, IP;
	
	if(mozile.edit.isNodeEditable(node) || force) {
		if(node.nodeType == mozile.dom.TEXT_NODE) {
			if(direction == mozile.edit.NEXT) offset = 0;
			else offset = node.data.length;
			return new mozile.edit.InsertionPoint(node, offset);
		}
		
		// Try to dig into this node.
		if(direction == mozile.edit.NEXT) IP = mozile.edit.getInsertionPoint(node.firstChild, direction);
		else IP = mozile.edit.getInsertionPoint(node.lastChild, direction);
		if(IP) return IP;

		// Set the IP at the beginning or the end.
		if(direction == mozile.edit.NEXT) offset = 0;
		else offset = node.childNodes.length;
		return new mozile.edit.InsertionPoint(node, offset);
	}

	return null;
}




/**
 * Indicates the rich editing commands have been defined.
 * @type Boolean
 */
mozile.edit.rich = true;


/**
 * Removes a node from the document.
 * @type mozile.edit.Command
 */
mozile.edit.insertNode = new mozile.edit.Command("InsertNode");

/**
 * Always false. This command is never triggered, but called directly by other commands.
 * @param {Event} event Optional. The event object to be tested.
 * @param {Node} parentNode Optional. If the node is to be inserted at the beginning of a parent node, provide the node here.
 * @param {Node} previousSibling Optional. If the node is to be inserted after a previous node, provide the node here.
 * @param {Node} content Optional. The node to be inserted.
 * @type Boolean
 */
mozile.edit.insertNode.test = function(event, parentNode, previousSibling, content) {
	if(event) {
		return false;
	}
	
	if(!parentNode && !previousSibling) return false;
	if(!content) return false;
	
	return true;
}

/**
 * Prepares a state object for the insert node command.
 * @param {Event} event Optional. The event object to be converted into a state.
 * @param {Node} parentNode Optional. If the node is to be inserted at the beginning of a parent node, provide the node here.
 * @param {Node} previousSibling Optional. If the node is to be inserted after a previous node, provide the node here.
 * @param {Node} content Optional. The node to be inserted.
 * @type mozile.edit.State
 */
mozile.edit.insertNode.prepare = function(event, parentNode, previousSibling, content) {
	var state = new mozile.edit.State(this, false); // don't need the selection
	state.location = {parentNode: null, previousSibling: null};
	
	state.location.parentNode = state.storeNode(parentNode);
	state.location.previousSibling = state.storeNode(previousSibling);
	
	state.content = null;
	if(content) state.content = content;
	
	return state;
}

/**
 * Inserts a node at the given location.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.insertNode.execute = function(state, fresh) {
	if(!state.content) throw("Error [mozile.edit.insertNode.execute]: No content provided.");
	var location = {previousSibling: null, parentNode: null};

	// Get "before" nodes
	if(state.location.previousSibling) {
		location.previousSibling = mozile.xpath.getNode(state.location.previousSibling);
	}
	else if(state.location.parentNode) {
		location.parentNode = mozile.xpath.getNode(state.location.parentNode);
	}
	else throw("Error [mozile.edit.insertNode.execute]: No previous sibling or parentNode provided.");

	// Insert the node.
	if(location.previousSibling) mozile.dom.insertAfter(state.content, location.previousSibling);
	else if(location.parentNode) mozile.dom.prependChild(state.content, location.parentNode);

	state.executed = true;
	return state;
}

/**
 * Removes an inserted node.
 * @param {mozile.edit.State} state The state information needed to unexecute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.insertNode.unexecute = function(state, fresh) {
	if(state.content.parentNode) {
		state.content.parentNode.removeChild(state.content);
	}
	else mozile.debug.debug("mozile.edit.insertNode.unexecute", "No parent for state.content "+ state.content);

	state.executed = false;
	return state;
}





/**
 * Removes a node from the document.
 * @type mozile.edit.Command
 */
mozile.edit.removeNode = new mozile.edit.Command("RemoveNode");

/**
 * Always false. This command is never triggered, but called directly by other commands.
 * @param {Event} event Optional. The event object to be tested.
 * @param {String} content Optional. The content removed.
 * @type Boolean
 */
mozile.edit.removeNode.test = function(event, content) {
	if(event) {
		return false;
	}
	
	if(!content) return false;
	if(!content.parentNode) return false;
	
	return true;
}

/**
 * Prepares a state object for the remove node command.
 * @param {Event} event The event object to be converted into a state.
 * @param {String} content Optional. The content removed.
 * @type mozile.edit.State
 */
mozile.edit.removeNode.prepare = function(event, content) {
	var state = new mozile.edit.State(this, false); // don't need the selection
	
	state.content = null;
	if(content) state.content = content;
	else if(event) state.content = mozile.edit._getNode(event);
	
	return state;
}

/**
 * Removes the node at the current selection. Stores the removed node so that the operation can be undone. Does not manipulate the selection.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.removeNode.execute = function(state, fresh) {
	var target = state.content;
	//alert([target, target.previousSibling, target.parentNode, target.data].join("\n"));
	var parentNode = target.parentNode;
	if(!parentNode) throw("Error [mozile.edit.removeNode.execute]: No parent node for node '"+ target +"'.");

	var previousSibling = target.previousSibling;
	if(previousSibling && !state.previousSibling)
		state.previousSibling = mozile.xpath.getXPath(previousSibling);
	else if(!state.parentNode)
		state.parentNode = mozile.xpath.getXPath(parentNode);

	parentNode.removeChild(target);

	state.executed = true;
	return state;
}

/**
 * Restores a removed node.
 * @param {mozile.edit.State} state The state information needed to unexecute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.removeNode.unexecute = function(state, fresh) {
	if(state.previousSibling) {
		var previousSibling = mozile.xpath.getNode(state.previousSibling);
		if(!previousSibling) throw("Error [mozile.edit.removeNode.unexecute]: Could not find previousSibling '"+ state.previousSibling +"'.");
		mozile.dom.insertAfter(state.content, previousSibling);
	}

	else if(state.parentNode) {
		var parentNode = mozile.xpath.getNode(state.parentNode);
		mozile.dom.prependChild(state.content, parentNode);
	}

	else mozile.debug.inform("mozile.edit.removeNode.unexecute", "No parent or previousSibling.");

	state.executed = false;
	return state;
}


/**
 * Removes text and nodes from a range.
 * @type mozile.edit.Command
 */
mozile.edit.remove = new mozile.edit.Command("Remove");

/**
 * True if the event was a keypress of the delete key.
 * @param {Event} event The event object to be tested.
 * @param {Integer} direction Optional. The direction for the removal to use. Defaults to previous.
 * @param {String} content Optional. The content removed.
 * @param {Boolean} preserve Optional. When true the container will not be removed, event if it's empty.
 * @type Boolean
 */
mozile.edit.remove.test = function(event, direction, content, preserve) {
	if(event) {
		if(mozile.edit.checkAccelerators(event, ["Backspace", "Delete"])) return true;
		return false;	
	}
	
	return true;
}

/**
 * Prepares a state object for the remove command.
 * @param {Event} event The event object to be converted into a state.
 * @param {Integer} direction Optional. The direction for the removal to use. Defaults to previous.
 * @param {String} content Optional. The content removed.
 * @param {Boolean} preserve Optional. When true the container will not be removed, event if it's empty.
 * @type mozile.edit.State
 */
mozile.edit.remove.prepare = function(event, direction, content, preserve) {
	var state = new mozile.edit.State(this);

	state.direction = mozile.edit.PREVIOUS;
	if(direction) state.direction = direction;
	else if(event && mozile.edit.convertKeyCode(event.keyCode) == "Delete")
		state.direction = mozile.edit.NEXT;

	state.content = " ";
	if(content) state.content = content;

	state.preserve = false;
	if(preserve === true) state.preserve = true;

	return state;
}

/**
 * General remove command.
 * There are many cases to handle, which makes the method complex.
 * <ul>
 * 	 <li>If the selection is collapsed:
 *   <ul>
 *     <li>If the selection is within a text node, removeText is called.
 *     <li>If the selection is at the edge of a text node and moving out of it,
 *     <ul>
 *       <li>If there is an adjacent childless node, it is removed,
 *       <li>If the next text node is inside another block, the blocks are merged,
 *       <li>Otherwise the selection is moved into the next node and one character is removed with removeText
 *     </ul>
 *     <li>If the node is empty it is removed
 *     <li>If the parentNode is empty, it is removed
 *     <li>Remaining text nodes are merged (normalized).
 *   </ul>
 *   <li>If the selection is not collapsed:
 *   <ul>
 *     <li>If the selection is within a text node, removeText is called,
 *     <li>Otherwise the _removeRange() method is called.
 *   </ul>
 * </ul>
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.remove.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	state.actions = new Array();

	var newState, IP;
	var collapsed = selection.isCollapsed;

	// Collapsed case is tricky.
	if(selection.isCollapsed) {
		var node = selection.focusNode;
		// Text case.
		if(node && node.nodeType == mozile.dom.TEXT_NODE &&
			!mozile.edit.isEmptyToken(node) ) {
			// Just remove the text.
			if( (state.direction == mozile.edit.PREVIOUS &&
					selection.focusOffset > 0) ||
				(state.direction == mozile.edit.NEXT && 
					selection.focusOffset < node.data.length) ) {
				mozile.edit.removeText.request(state, fresh, state.direction);
			}
			// Move to next insertion point.
			else {
				IP = selection.getInsertionPoint();
				IP.seek(state.direction);
				IP.extend();
			}
		}

		// Element case.
		else {		
			IP = selection.getInsertionPoint();
			IP.seek(state.direction);
			IP.extend();
		}
	}
	
	// Non-collapsed case is simple.
	//mozile.require("mozile.util");
	//alert(mozile.util.dumpValues(selection.store()));
	if(!selection.isCollapsed) {
		if(selection.getRangeAt(0).commonAncestorContainer.nodeType == mozile.dom.TEXT_NODE) {
			mozile.edit.removeText.request(state, fresh, state.direction);
		}
		else mozile.edit._removeRange(state, fresh, state.direction);
	}

	// Remove the empty container, or ensure it's non-empty.
	//mozile.debug.debug("", mozile.xpath.getXPath(selection.focusNode) +"\n"+ state.preserve +" "+ mozile.edit.isEmpty(selection.focusNode));
	if(state.preserve) {
		mozile.edit._ensureNonEmpty(state, fresh, selection.focusNode);
	}
	else {
		mozile.edit._removeEmpty(state, fresh, selection.focusNode,
			mozile.edit.getParentBlock(selection.focusNode));
	}

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}





/**
 * Moves a node from one location in the document to another.
 * @type mozile.edit.Command
 */
mozile.edit.moveNode = new mozile.edit.Command("MoveNode");

/**
 * Always false. This command is always called instead of triggered.
 * @param {Event} event Optional. The event object to be tested.
 * @param {Node} destinationParentNode Optional. The new parentNode for the target.
 * @param {Node} destinationPreviousSibling Optional. The new previousSibling for the target.
 * @param {Node} target Optional. The node to move.
 * @type Boolean
 */
mozile.edit.moveNode.test = function(event, destinationParentNode, destinationPreviousSibling, target) {
	if(event) {
		return false;
	}
	
	if(!destinationParentNode && !destinationPreviousSibling) return false;
	if(!target) return false;

	return true;
}

/**
 * Prepares a state object for the move node command.
 * @param {Event} event Optional. The event object to be converted into a state.
 * @param {Node} destinationParentNode Optional. The new parentNode for the target.
 * @param {Node} destinationPreviousSibling Optional. The new previousSibling for the target.
 * @param {Node} target Optional. The node to move.
 * @type mozile.edit.State
 */
mozile.edit.moveNode.prepare = function(event, destinationParentNode, destinationPreviousSibling, target) {
	var state = new mozile.edit.State(this);
	state.destination = {parentNode: null, previousSibling: null};
	
	state.destination.parentNode = state.storeNode(destinationParentNode);
	state.destination.previousSibling = state.storeNode(destinationPreviousSibling);
	state.target = state.storeNode(target);
	
	return state;
}

/**
 * Removes a node from one location and inserts it at another. XPath locations are stored so that the node can be moved back. (The node itself is not stored.)
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.moveNode.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	
	// Get some selection data.
	var anchorNode = selection.anchorNode;
	var anchorOffset = selection.anchorOffset;
	var focusNode = selection.focusNode;
	var focusOffset = selection.focusOffset;
	
	var target = mozile.xpath.getNode(state.target);
	if(!target) throw("Error: mozile.edit.moveNode.execute No target node.");

	// Move the node
	mozile.edit.removeNode.request(state, fresh, target);
	mozile.edit.insertNode.request(state, fresh, 
		state.destination.parentNode, state.destination.previousSibling, target);

	// Restore the selection.
	selection.collapse(anchorNode, anchorOffset);
	if(focusNode != anchorNode || focusOffset != anchorOffset) {
		selection.extend(focusNode, focusOffset);
	}

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}





/**
 * Merges two nodes, appending the children of the second node to the first node.
 * @type mozile.edit.Command
 */
mozile.edit.mergeNodes = new mozile.edit.Command("MergeNodes");

/**
 * Always false. This command is always called instead of triggered.
 * @param {Event} event Optional. The event object to be tested.
 * @param from Optional. A Node or an XPath for the node to be merged into the "to" node.
 * @param to Optional. A Node or an XPath for the node which will gain the content of the "from" node.
 * @type Boolean
 */
mozile.edit.mergeNodes.test = function(event, from, to) {
	if(event) {
		return false;
	}
	
	if(!from) return false;
	if(!to) return false;
	
	return true;
}

/**
 * Prepares a state object for the merge nodes command.
 * @param {Event} event Optional. The event object to be converted into a state.
 * @param from Optional. A Node or an XPath for the node to be merged into the "to" node.
 * @param to Optional. A Node or an XPath for the node which will gain the content of the "from" node.
 * @type mozile.edit.State
 */
mozile.edit.mergeNodes.prepare = function(event, from, to) {
	var state = new mozile.edit.State(this);

	state.from = state.storeNode(from);
	state.to = state.storeNode(to);
	
	return state;
}

/**
 * Merges two nodes.
 * <p>In the case of text, nodes are merged using the DOM Text.appendData() method.
 * In the case of elements, child nodes are moved to the "to" node using moveNode, and the "from" element is removed using removeNode.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.mergeNodes.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	state.actions = new Array();

	// Get target nodes
	var fromNode = mozile.xpath.getNode(state.from);
	var toNode = mozile.xpath.getNode(state.to);
	
	// Get some selection data.
	var anchorNode = selection.anchorNode;
	var anchorOffset = selection.anchorOffset;
	var focusNode = selection.focusNode;
	var focusOffset = selection.focusOffset;
	
	mozile.edit._removeEmptyTokens(state, fresh, toNode);
	mozile.edit._removeEmptyTokens(state, fresh, fromNode);
	var firstNode, secondNode;

	// Normalize adjacent text nodes
	if(fromNode.nodeType == mozile.dom.TEXT_NODE &&
		toNode.nodeType == mozile.dom.TEXT_NODE) {
		if(fromNode.nextSibling == toNode) {
			firstNode = fromNode;
			secondNode = toNode;
		}
		else if(toNode.nextSibling == fromNode) {
			firstNode = toNode;
			secondNode = fromNode;
		}
		if(firstNode && secondNode) {
			// Fix selection.
			var offset = firstNode.data.length;
			if(anchorNode == secondNode) {
				anchorNode = firstNode;
				anchorOffset += offset;
				//alert("Anchor "+ mozile.xpath.getXPath(anchorNode) +" "+ anchorOffset);
			}
			if(focusNode == secondNode) {
				focusNode = firstNode;
				focusOffset += offset;
				//alert("Focus "+ mozile.xpath.getXPath(focusNode) +" "+ focusOffset);
			}
			
			// Merge nodes.
			mozile.edit.insertText.request(state, fresh,
				null, firstNode.data + secondNode.data, firstNode);
			mozile.edit.removeNode.request(state, fresh, secondNode);

			// Restore the selection.
			selection.collapse(anchorNode, anchorOffset);
			if(focusNode != anchorNode || focusOffset != anchorOffset) {
			  selection.extend(focusNode, focusOffset);
			}

			state.selection.after = selection.store();
			state.executed = true;
			return state;
		}
		else throw("Error [mozile.edit.mergeNodes.execute]: Cannot merge text non-adjacent nodes: "+ state.from +" "+ state.to);
	}

	// Move all children from fromNode to the end of toNode.
	firstNode = toNode.lastChild;
	secondNode = fromNode.firstChild
	while(fromNode.firstChild) {
		mozile.edit.moveNode.request(state, fresh, toNode, toNode.lastChild, fromNode.firstChild);
	}

	// Remove fromNode
	mozile.edit.removeNode.request(state, fresh, fromNode);
	
	// Restore the selection.
	var IP;
	// Collapse to the anchor.
	if(mozile.dom.isAncestorOf(document.documentElement, anchorNode)) {
		selection.collapse(anchorNode, anchorOffset);
	}
	else {
		IP = mozile.edit.getInsertionPoint(toNode, mozile.edit.NEXT);
		if(IP) IP.select();
	}
	// Extend to the focus.
	if(mozile.dom.isAncestorOf(document.documentElement, focusNode)) {
		if(focusNode != selection.anchorNode || 
			focusOffset != selection.anchorOffset) {
			selection.extend(focusNode, focusOffset);
		}
	}
	else {
		IP = mozile.edit.getInsertionPoint(toNode, mozile.edit.PREVIOUS);
		if(IP) IP.extend();
	}
	
	// Normalize last node of toNode with first node of fromNode
	if(firstNode && firstNode.nodeType == mozile.dom.TEXT_NODE &&
	  secondNode && secondNode.nodeType == mozile.dom.TEXT_NODE) {
	  this.request(state, fresh, firstNode, secondNode);
	}

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}




/**
 * Splits a node by cloning it, inserting the clone after the original, and moving all the original's children after the cursor into the clone.
 * @type mozile.edit.Command
 */
mozile.edit.splitNode = new mozile.edit.Command("SplitNode");

/**
 * Always false. This command is never triggered, but called directly by other commands instead.
 * @param {Event} event The event object to be tested.
 * @param {Node} target Optional. The first node in the new split element.
 * @param {Node} offset Optional. An integer offset within a text node where the text node will be split.
 * @param {Boolean} after Optional. When true the node will be split after the target rather than before it.
 * @type Boolean
 */
mozile.edit.splitNode.test = function(event, target, offset, after) {
	if(event) {
		return false;
	}
	if(!target) return false;
	if(!target.nodeType) return false;
	if(target.nodeType != mozile.dom.TEXT_NODE &&
		target.nodeType != mozile.dom.ELEMENT_NODE) 
		return false;
	
	return true;
}

/**
 * Prepares a state object for the split nodes command.
 * @param {Event} event Optional. The event object to be converted into a state.
 * @param {Node} target Optional. The first node in the new split element.
 * @param {Node} offset Optional. An integer offset within a text node where the text node will be split.
 * @param {Boolean} after Optional. When true the node will be split after the target rather than before it.
 * @type mozile.edit.State
 */
mozile.edit.splitNode.prepare = function(event, target, offset, after) {
	var state = new mozile.edit.State(this);

	target = state.storeNode(target);
	state.target = target;
	
	state.offset = null;
	if(offset) state.offset = offset;

	state.after = false;
	if(after === true) state.after = true;

	return state;
}

/**
 * Splits a node in two. 
 * <ul>
 *  <li>If the focusNode is a text node, it is split at the focusOffset,
 *  <li>A clone of the focusNode or focusNode's parent is created,
 *  <li>The clone is inserted after the focusNode
 *  <li>All children after the focusOffset are moved into the clone.
 * </ul>
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.splitNode.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	state.actions = new Array();

	var target = mozile.xpath.getNode(state.target);
	var oldContainer, newContainer;

	// Get some selection data.
	var anchorNode = selection.anchorNode;
	var anchorOffset = selection.anchorOffset;
	var focusNode = selection.focusNode;
	var focusOffset = selection.focusOffset;
	
	//alert("Anchor "+ mozile.xpath.getXPath(anchorNode) +" "+ anchorOffset +"\n"+
	//	"Focus "+ mozile.xpath.getXPath(focusNode) +" "+ focusOffset);
		
	// Split a text node.
	if(target.nodeType == mozile.dom.TEXT_NODE && state.offset != undefined) {
		state.splitNode = target;
		oldContainer = target;
		newContainer = target.splitText(state.offset);

		// Fix the selection.
		if(anchorNode == target && anchorOffset >= state.offset) {
			anchorNode = newContainer;
			anchorOffset -= state.offset;
		}
		if(focusNode == target && focusOffset >= state.offset) {
			focusNode = newContainer;
			focusOffset -= state.offset;
		}
	}

	// Split an element.
	else if(target.nodeType == mozile.dom.TEXT_NODE ||
		target.nodeType == mozile.dom.ELEMENT_NODE) {
		// Count previous siblings (not including the target node).
		var i = 0;
		if(state.after) i++;
		var node = target;
		while(node) {
			i++;
			node = node.previousSibling;
		}
	
		// Create the new node.	
		var oldContainer = target.parentNode;
		var newContainer = oldContainer.cloneNode(false);
		mozile.edit.insertNode.request(state, fresh, 
			null, oldContainer, newContainer);
		var newContainerPath = mozile.xpath.getXPath(newContainer);
		
		// Move nodes
		while(oldContainer.childNodes.length >= i) {
			mozile.edit.moveNode.request(state, fresh, 
				newContainerPath, null, oldContainer.lastChild);
		}
		
		if(mozile.edit.isBlock(oldContainer))
			mozile.edit._ensureNonEmpty(state, fresh, oldContainer);
		if(mozile.edit.isBlock(newContainer))
			mozile.edit._ensureNonEmpty(state, fresh, newContainer);
	}

	// Restore the selection.
	selection.collapse(anchorNode, anchorOffset);
	if(focusNode != selection.anchorNode || 
		focusOffset != selection.anchorOffset) {
		selection.extend(focusNode, focusOffset);
	}

	state.oldContainer = oldContainer;
	state.newContainer = newContainer;
	state.selection.after = selection.store();
	state.executed = true;
	return state;
}

/**
 * Merges split nodes.
 * Unexecutes all move operations, removes the cloned node, and normalizes the focusNode if it was split.
 * @param {mozile.edit.State} state The state information needed to unexecute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.splitNode.unexecute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.after);

	// Unexecute actions in reverse order.
	for(var i = state.actions.length - 1; i >= 0; i--) {
		state.actions[i] = state.actions[i].command.unexecute(state.actions[i], fresh);
		if(state.actions[i].executed) throw("Error: mozile.edit.splitNode.unexecute Child command unexecute failed at action "+ i +".");
	}

	// Normalize text node.
	if(state.splitNode) {
		state.splitNode.appendData(state.newContainer.data);
		if(state.newContainer.parentNode) {
			state.newContainer.parentNode.removeChild(state.newContainer);
		}
		//else mozile.debug.debug("mozile.edit.splitNode.unexecute", "New container has no parent: "+ state.newContainer.nodeName);
	}

	selection.restore(state.selection.before);
	state.executed = false;
	return state;
}





/**
 * Splits mutliple nodes.
 * @type mozile.edit.Command
 */
mozile.edit.splitNodes = new mozile.edit.Command("SplitNodes");

/**
 * True when given an "enter" key keypress.
 * @param {Event} event The event object to be tested.
 * @param {Node} target Optional. The first node in the new split element.
 * @param {Node} offset Optional. An integer offset within a text node where the text node will be split.
 * @param {Node} limitNode Optional. The first node which will not be split. Defaults to the parent block's parent.
 * @param {Boolean} shallow Optional. Indicates that the method should not climb all the way to the limitNode.
 * @type Boolean
 */
mozile.edit.splitNodes.test = function(event, target, offset, limitNode, shallow) {
	if(event) {
		return false;
	}
	
	return true;
}

/**
 * Prepares a state object for the split nodes command.
 * @param {Event} event The event object to be converted into a state.
 * @param {Node} target Optional. The first node in the new split element.
 * @param {Node} offset Optional. An integer offset within a text node where the text node will be split.
 * @param {Node} limitNode Optional. The first node which will not be split. Defaults to the parent block's parent.
 * @param {Boolean} shallow Optional. Indicates that the method should not climb all the way to the limitNode.
 * @type mozile.edit.State
 */
mozile.edit.splitNodes.prepare = function(event, target, offset, limitNode, shallow) {
	var state = new mozile.edit.State(this);

	if(!target) {	
		var selection = mozile.dom.selection.get();
		var range = selection.getRangeAt(0);
		if(range.startContainer.nodeType == mozile.dom.TEXT_NODE)
			target = range.startContainer;
		else target = range.startContainer.childNodes[range.startOffset];
	}
	state.target = state.storeNode(target);

	state.offset = null;
	if(offset != undefined && offset != null) state.offset = offset;
	// If the command was triggered by an event, and no offset was given, 
	// indicate that the focusOffset should be used.
	else if(event) state.offset = "focusOffset";

	var limit = null;
	if(limitNode && mozile.dom.isAncestorOf(limitNode, target)) {
		limit = limitNode;
	}
	else limit = mozile.edit.getParentBlock(target).parentNode;
	state.limitNode = state.storeNode(limit);
	
	state.shallow = false;
	if(shallow === true) state.shallow = true;

	return state;
}

/**
 * Splits multiple nodes by calling splitNode repeatedly.
 * If the selection is not collapsed, the range is removed first.
 * Starting at the focusNode, each ancestor is split until a block level element is reached. 
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.splitNodes.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	state.actions = new Array();

	//alert(state.target +" "+ offset +"\n"+ state.limitNode);
	var newState;
	var node = mozile.xpath.getNode(state.target);
	var offset = state.offset;
	if(offset == "focusOffset") offset = selection.focusOffset;
	var limitNode = mozile.xpath.getNode(state.limitNode);
	var after = false;
	//alert(mozile.xpath.getXPath(node) +" "+ offset +"\n"+ mozile.xpath.getXPath(limitNode));
	
	// Try to avoid splitting by climbing toward the root.
	// The insertion point changes, but the visible cursor position does not.
	// Don't climb any higher than the limitNode's grandchildren.
	// Climb to the left.
	if(offset == null || offset == 0) {
		offset = null;
		while(node) {
			//alert("Climbing left "+ mozile.xpath.getXPath(node) +" "+ offset);
			if(node == limitNode) break;
			if(!node.parentNode) break;
			if(node.parentNode == limitNode) break;
			if(node != node.parentNode.firstChild) break;
			if(state.shallow) {
				if(!node.parentNode.parentNode) break;
				if(node.parentNode.parentNode == limitNode) break;
			}
			node = node.parentNode;
		}
	}
	// Climb to the right.
	else if(node.data && offset == node.data.length) {
		offset = null;
		if(node.nextSibling) node = node.nextSibling;
		while(node) {
			//alert("Climbing right "+ mozile.xpath.getXPath(node) +" "+ offset);
			if(node == limitNode) break;
			if(!node.parentNode) break;
			if(node.parentNode == limitNode) break;
			if(node != node.parentNode.lastChild) break;
			if(!node.parentNode.nextSibling) break;
			if(state.shallow) {
				if(!node.parentNode.parentNode) break;
				if(node.parentNode.parentNode == limitNode) break;
			}
			node = node.parentNode.nextSibling;
		}
		// Special case: this is the last child of its parent.
		if(node == node.parentNode.lastChild) after = true;
	}
	
	// Split until the limit node is reached.
	while(node) {
		//alert("Splitting "+ mozile.xpath.getXPath(node) +" "+ offset);
		if(node == limitNode) break;
		if(offset == null && node.parentNode == limitNode) break;
		if(!node.parentNode) break;

		newState = mozile.edit.splitNode.request(state, fresh, node, offset, after);

		if(newState && newState.newContainer) {
			node = newState.newContainer;
			offset = null;
		}
	}
	
	if(newState) {
		if(newState.oldContainer) state.oldContainer = newState.oldContainer;
		if(newState.newContainer) state.newContainer = newState.newContainer;
	}
	else {
		state.oldContainer = node.previousSibling;
		state.newContainer = node;
	}

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}





/**
 * Splits a target element using splitNodes.
 * @param {String} name The group's name.
 * @constructor
 */
mozile.edit.Split = function(name) { 
	/**
	 * The name for this command group.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command does not contain more commands.
	  * @type Boolean
	  */
	 this.group = false;
	 
	 /**
	  * Indicates that any contents of the selection should be removed before inserting.
	  * @type Boolean
	  */
	 this.remove = true;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.makesChanges = "node";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.watchesChanges = "node";
	 
	 /**
	  * Determines what node is the target of the split. Can be a string: "element", "block", or "localName [name]"; or a function.
	  * @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.target = "block";
	 
	 /**
	  * Determines what direction to search for the target in. 
	  * @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.direction = "ancestor";

	 // Register this command group on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}
mozile.edit.Split.prototype = new mozile.edit.Command;
mozile.edit.Split.prototype.constructor = mozile.edit.Split;


/**
 * Prepares a state object for the Split command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.Split.prototype.prepare = function(event) {
	var state = new mozile.edit.State(this);

	var target = mozile.edit._getTarget(event, this.target, this.direction);
	state.limit = state.storeNode(target.parentNode);

	// Some commands may need additional information from the user.
	// If so, define a method named "prompt", which takes the event and the state and returns a Boolean true when it succeeds.
	if(this.prompt) {
		if(!this.prompt(event, state)) return null;
	}

	return state;
}

/**
 * Inserts a new element at the selection.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.Split.prototype.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	state.actions = new Array();
	
	if(!selection.isCollapsed) mozile.edit.remove.request(state, fresh);
	var limit = mozile.xpath.getNode(state.limit);
	var newState = mozile.edit.splitNodes.request(state, fresh, selection.focusNode, selection.focusOffset, limit, true);
	var IP = mozile.edit.getInsertionPoint(newState.newContainer, mozile.edit.NEXT);
	if(IP) IP.select();
	//alert(newState);
	
	state.newContainer = newState.newContainer;
	state.oldContainer = newState.oldContainer;

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}



/**
 * A command used to insert a new element.
 * @param {String} name The group's name.
 * @constructor
 */
mozile.edit.Insert = function(name) { 
	/**
	 * The name for this command group.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command does not contain more commands.
	  * @type Boolean
	  */
	 this.group = false;
	 
	 /**
	  * Indicates that any contents of the selection should be removed before inserting.
	  * @type Boolean
	  */
	 this.remove = true;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.makesChanges = "node";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.watchesChanges = "node";

	 // Register this command group on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}
mozile.edit.Insert.prototype = new mozile.edit.Command;
mozile.edit.Insert.prototype.constructor = mozile.edit.Insert;


/**
 * Prepares a state object for the wrap command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.Insert.prototype.prepare = function(event) {
	var state = new mozile.edit.State(this);
	
	state.element = null;
	if(typeof(this.element) == "string") {
		state.element = mozile.dom.createElement(this.element);
	}
	else if(this.element && this.element.cloneNode) {
		state.element = this.element.cloneNode(true);
	}

	state.text = null;
	if(state.element == null && this.text) {
		state.text = this.text;
	}
	

	// Some commands may need additional information from the user.
	// If so, define a method named "prompt", which takes the event and the state and returns a Boolean true when it succeeds.
	if(this.prompt) {
		if(!this.prompt(event, state)) return null;
	}

	return state;
}

/**
 * Inserts a new element at the selection.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.Insert.prototype.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	state.actions = new Array();
	var newState;

	// Remove the current range, if it is not collapsed.
	if(this.remove && !selection.isCollapsed) {
		mozile.edit.remove.request(state, fresh, mozile.edit.NEXT);
		selection = mozile.dom.selection.get();
	}
	
	// Insert some text.
	if(state.text) {
		mozile.edit.insertText.request(state, fresh, null, state.text);
		state.selection.after = selection.store();
		state.executed = true;
		return state;
	}

	// Insert an empty element, and add an empty token if needed.
	if(selection.isCollapsed) {
		var previousNode;
		if(selection.focusNode.nodeType == mozile.dom.TEXT_NODE) {
			newState = mozile.edit.splitNode.request(state, fresh,
				selection.focusNode, selection.focusOffset);
			previousNode = newState.oldContainer;
		}
		else previousNode = selection.focusNode[selection.focusOffset];

		mozile.edit.insertNode.request(state, fresh, 
			null, previousNode, state.element);

		if(!this.remove) {
			var text;
			if(mozile.edit.isBlock(state.element))
				text = mozile.edit.createEmptyToken();
			else text = document.createTextNode("");
			state.element.appendChild(text);
		}
	}

	// Insert an element and add content to it.
	else {	
		var range = selection.getRangeAt(0);
		var container = range.commonAncestorContainer;
		if(container.nodeType == mozile.dom.TEXT_NODE) container = container.parentNode;

		// Split start and end nodes until reaching the commonAncestorContainer.
		// Store these two values for IE.
		var startContainer = range.startContainer;
		var startOffset = range.startOffset;
		newState = mozile.edit.splitNodes.request(state, fresh, 
			range.endContainer, range.endOffset, container);
		var nextNode = newState.newContainer;
		newState = mozile.edit.splitNodes.request(state, fresh, 
			startContainer, startOffset, container);
		var previousNode = newState.oldContainer;

		// Insert the new element.
		mozile.edit.insertNode.request(state, fresh, 
			null, previousNode, state.element);

		// Move the nodes in between the start and end containers.
		// Loop until the nextNode is found or we run out of nodes.
		var current = state.element.nextSibling;
		while(current) {
			if(current == nextNode) break;
			var target = current;
			current = current.nextSibling;
			mozile.edit.moveNode.request(state, fresh, state.element, state.element.lastChild, target);
		}
	}
	
	// Restore the selection.
	var IP = mozile.edit.getInsertionPoint(state.element, mozile.edit.NEXT);
	if(IP) {
		// Move to the next node.
		if(this.remove) {
			IP.seekNode(mozile.edit.NEXT, false);
			if(IP) IP.select();
		}
		// Select this new node.
		else {
			IP.select();
			IP = mozile.edit.getInsertionPoint(state.element, mozile.edit.PREVIOUS);
			if(IP) IP.extend();
		}
	}

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}







/**
 * A command used to wrap a selection inside a new element.
 * @param {String} name The group's name.
 * @constructor
 */
mozile.edit.Wrap = function(name) { 
	/**
	 * The name for this command group.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command does not contain more commands.
	  * @type Boolean
	  */
	 this.group = false;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.makesChanges = "node";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.watchesChanges = "node";
	 
	 /**
	  * Determines whether wrappers may be nested.
	  * @type Boolean
	  */
	 this.nested = false;

	 // Register this command group on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}
mozile.edit.Wrap.prototype = new mozile.edit.Command;
mozile.edit.Wrap.prototype.constructor = mozile.edit.Wrap;


/**
 * Checks to see if the given node is a wrapper.
 * @param {Node} node The node to check.
 * @type Boolean
 */
mozile.edit.Wrap.prototype._isWrapper = function(node) {
	if(!node) return false;
	var targetName = mozile.dom.getLocalName(node);
	if(!targetName) return false;
	targetName = targetName.toLowerCase();

	var wrapperName = mozile.edit._getElementName(this);
	if(!wrapperName) return false;
	wrapperName = wrapperName.toLowerCase();

	if(targetName == wrapperName) {
		if(this.styleName) {
			var styleName = mozile.dom.convertStyleName(this.styleName);
			if(node.style && node.style[styleName] &&
				node.style[styleName] == this.styleValue) return true;
		}
		else return true;
	}
	
	return false;
}

/**
 * Finds the wrapper for the given node if it has one.
 * @param {Node} node The node to start searching from.
 * @param {Boolean} outerWrapper Optional. When true the method will continue to search for wrappers instead of stopping when one is found.
 * @type Element
 */
mozile.edit.Wrap.prototype._getWrapper = function(node, outerWrapper) {
	if(!node) return false;
	var wrapper = null;

	while(node) {
		if(this._isWrapper(node)) wrapper = node;
		if(wrapper && !outerWrapper) break;
		node = node.parentNode;
	}
	
	return wrapper;
}

/**
 * Indicates that the command is currently active.
 * In the case of a "strong" command it would me that the cursor is inside a "strong" element, and the command's button should indicate that fact.
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.edit.Wrap.prototype.isActive = function(event) {
	if(this.prompt) return false;
	if(this._getWrapper(event.node)) return true;
	else return false;
}

/**
 * Prepares a state object for the wrap command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.Wrap.prototype.prepare = function(event) {
	var state = new mozile.edit.State(this);
	
	state.wrapper = null;
	if(typeof(this.element) == "string") {
		state.wrapper = mozile.dom.createElement(this.element);
		if(this.styleName) {
			mozile.dom.setStyle(state.wrapper, this.styleName, this.styleValue);
		}
	}
	else if(this.element && this.element.cloneNode) {
		state.wrapper = this.element.cloneNode(true);
	}

	// Some commands may need additional information from the user.
	// If so, define a method named "prompt", which takes the event and the state and returns a Boolean true when it succeeds.
	if(this.prompt) {
		if(!this.prompt(event, state)) return null;
	}
	
	return state;
}

/**
 * Wraps the current selection in a new element.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.Wrap.prototype.execute = function(state, fresh) {
	//alert("Wrapping");
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	var range = selection.getRangeAt(0);
	state.actions = new Array();
	var wrapper = state.wrapper;
	state.wrappers = new Array();
	var nextNode, previousNode, textNode, newState;

	// Insert an empty element at the selection.
	if(range.collapsed) {
		var outerWrapper = this._getWrapper(range.commonAncestorContainer, true);
		
		// If the node is already wrapper, and wrappers shouldn't be nested, unwrap it.
		if(outerWrapper && !this.nested) {
			// If we're at the end of a wrapper, jump out of it.
			if(selection.focusNode == outerWrapper.lastChild &&
				selection.focusOffset == selection.focusNode.data.length) {
				var IP = mozile.edit.getInsertionPoint(outerWrapper, mozile.edit.PREVIOUS);
				IP.seekNode(mozile.edit.NEXT, false);
				IP.select();
				// If the wrapper is empty, remove it.
				mozile.edit._removeEmpty(state, fresh, 
					outerWrapper.lastChild, outerWrapper.parentNode);
			}
			// Split the wrapper and insert a new text node.
			else {
				newState = mozile.edit.splitNodes.request(state, fresh, range.startContainer, range.startOffset, outerWrapper.parentNode);
				nextNode = newState.newContainer;
				textNode = document.createTextNode("");
				mozile.edit.insertNode.request(state, fresh, 
					nextNode.parentNode, nextNode.previousSibling, textNode);
				selection.collapse(textNode, 0);
			}
		}
		
		// If the node has no wrapper, insert a new wrapper and text node.
		else {
			if(range.startContainer.nodeType == mozile.dom.TEXT_NODE) {
				newState = mozile.edit.splitNodes.request(state, fresh, range.startContainer, range.startOffset, range.startContainer.parentNode);
				nextNode = newState.newContainer;
			}
			else nextNode = range.startContainer.childNodes[range.startOffset];
			mozile.edit.insertNode.request(state, fresh, 
				nextNode.parentNode, nextNode.previousSibling, wrapper);
			wrapper.appendChild(document.createTextNode(""));
			selection.collapse(wrapper.firstChild, 0);
		}
	}

	// Wrap the selection with one or more new elements.
	// Create as many wrappers as needed.
	else {
		// Get the container.
		var container = range.commonAncestorContainer;
		var startWrapper, endWrapper;
		if(!this.nested) {
			startWrapper = this._getWrapper(range.startContainer, true);
			endWrapper = this._getWrapper(range.endContainer, true);
		}
		if(startWrapper && endWrapper)
			container = mozile.dom.getCommonAncestor(startWrapper, endWrapper);
		else if(startWrapper)
			container = mozile.dom.getCommonAncestor(startWrapper, container);
		else if(endWrapper)
			container = mozile.dom.getCommonAncestor(endWrapper, container);
		container = container.parentNode;

		// Split nodes.
		var node, offset, startNode, endNode;
		// Split the end node.
		node = range.endContainer;
		offset = range.endOffset;
		var endContainer = node.parentNode;
		if(endWrapper) endContainer = endWrapper.parentNode;
		newState = mozile.edit.splitNodes.request(state, fresh, 
			node, offset, endContainer);
		endNode = newState.oldContainer;
		nextNode = newState.newContainer;
		
		// Split the start node.
		node = range.startContainer;
		offset = range.startOffset;
		var startContainer = node.parentNode;
		if(startWrapper) startContainer = startWrapper.parentNode;
		newState = mozile.edit.splitNodes.request(state, fresh, 
			node, offset, startContainer);
		previousNode = newState.oldContainer;
		startNode = newState.newContainer;
		if(endNode == node) endNode = startNode;

		//alert("Start "+ mozile.xpath.getXPath(startNode) +"\n"+
		//  "End "+ mozile.xpath.getXPath(endNode) +"\n"+
		//  "Previous "+ mozile.xpath.getXPath(previousNode) +"\n"+
		//  "Next "+ mozile.xpath.getXPath(nextNode) +"\n"+
		//  "Container "+ mozile.xpath.getXPath(container) );

		// Set up the treeWalker.
		var treeWalker = document.createTreeWalker(container, mozile.dom.NodeFilter.SHOW_ALL, null, false);
		var allNodesWrapped = false;
		
		// Remove all wrappers.
		if(!this.nested) {
			allNodesWrapped = true;
			treeWalker.currentNode = startNode;
			var current = treeWalker.currentNode;
			var oldWrapper;
			while(current) {
				if(current == nextNode) break;
				oldWrapper = this._getWrapper(current);
				if(oldWrapper) {
					// Get the next "current" node.
					if(oldWrapper.nextSibling) {
						current = oldWrapper.nextSibling;
						if(current == nextNode) current = null;
						else if(current.nodeType == mozile.dom.TEXT_NODE) {
							allNodesWrapped = false;
							current = current.nextSibling;
						}
					}
					mozile.edit._unwrapNode(state, fresh, oldWrapper);
				}
				else {
					allNodesWrapped = false;
					current = treeWalker.nextNode();
				}
			}
		}

		//alert("Start "+ mozile.xpath.getXPath(startNode) +"\n"+
		//  "End "+ mozile.xpath.getXPath(endNode) +"\n"+
		//  "Previous "+ mozile.xpath.getXPath(previousNode) +"\n"+
		//  "Next "+ mozile.xpath.getXPath(nextNode) +"\n"+
		//  "Container "+ mozile.xpath.getXPath(container) +"\n"+
		//  "All Nodes Wrapped "+ allNodesWrapped);

		// If any nodes were not previously wrapped, wrap everything again.
		// startNode == nextNode is a special case; it usually occurs when several different wrappers are added in sequence.
		if(!allNodesWrapped || startNode == nextNode) {
			if(previousNode) {
				treeWalker.currentNode = previousNode;
				treeWalker.nextSibling();
			}
			else if(startNode == nextNode) {
				treeWalker.currentNode = startNode;
				nextNode = container;
			}
			current = treeWalker.currentNode;
			var target, lastParent;
			while(current) {
				//alert(mozile.xpath.getXPath(current));
				// Check to see if the node should NOT be moved.
				if(current == nextNode) break;
				// This avoids a loop in IE. Maybe a flaw in Mozile's IE TreeWalker.
				if(mozile.dom.isAncestorOf(wrapper, current, container)) break;
				if(mozile.dom.isAncestorOf(current, wrapper, container))
					current = treeWalker.nextNode();
				else if(mozile.dom.isAncestorOf(current, nextNode, container))
					current = treeWalker.nextNode();
		
				// Move the node.
				else {
					target = current;
					current = treeWalker.nextSibling();
					if(!current) current = treeWalker.nextNode();
					// If we've changed parents, create a new wrapper.
					if(target.parentNode && target.parentNode != lastParent) {
						wrapper = state.wrapper.cloneNode(true);
						state.wrappers.push(wrapper);
						mozile.edit.insertNode.request(state, fresh, null, target, wrapper);
						lastParent = target.parentNode;
						//alert(mozile.xpath.getXPath(target) +"\n"+ mozile.xpath.getXPath(wrapper));
					}
					mozile.edit.moveNode.request(state, fresh, wrapper, wrapper.lastChild, target);
				}
			}
		}

		// Restore the selection.
		selection = mozile.dom.selection.get();
		range = selection.getRangeAt(0);
		container = range.commonAncestorContainer;
		if(container.nodeType != mozile.dom.TEXT_NODE) {
			var IP = mozile.edit.getInsertionPoint(previousNode, mozile.edit.PREVIOUS);
			if(IP) {
				IP.seekNode(mozile.edit.NEXT, false);
				IP.select();
				IP = mozile.edit.getInsertionPoint(nextNode, mozile.edit.NEXT);
				if(IP) {
					IP.seekNode(mozile.edit.PREVIOUS, false);
					IP.extend();
				}
			}
		}

		//alert("Next Prev "+ mozile.xpath.getXPath(nextNode.previousSibling) +"\n"+
		//	"Next "+ mozile.xpath.getXPath(nextNode) +"\n"+
		//	"Prev "+ mozile.xpath.getXPath(previousNode) +"\n"+
		//	"Prev Next "+ mozile.xpath.getXPath(previousNode.nextSibling)
		//	);
			
		// Merge adjacent wrappers.
		if(this._isWrapper(previousNode) && 
			this._isWrapper(previousNode.nextSibling)) {
		  mozile.edit.mergeNodes.request(state, fresh, previousNode.nextSibling, previousNode);
		}
		if(this._isWrapper(nextNode) && 
			this._isWrapper(nextNode.previousSibling)) {
		  mozile.edit.mergeNodes.request(state, fresh, nextNode, nextNode.previousSibling);
		}

	}

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}



/**
 * A command used to remove a container element, but keep the child nodes.
 * @param {String} name The group's name.
 * @constructor
 */
mozile.edit.Unwrap = function(name) { 
	/**
	 * The name for this command group.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command does not contain more commands.
	  * @type Boolean
	  */
	 this.group = false;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.makesChanges = "node";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.watchesChanges = "node";
	 
	 /**
	  * Determines what node is the target of the replacement. Can be a string: "element", "block", or "localName [name]"; or a function.
	  * @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.target = "element";
	 
	 /**
	  * Determines what direction to search for the target in. 
	  * @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.direction = "ancestor";

	 // Register this command group on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}
mozile.edit.Unwrap.prototype = new mozile.edit.Command;
mozile.edit.Unwrap.prototype.constructor = mozile.edit.Unwrap;


/**
 * Indicates that the command is currently available.
 * In the case of an "unlink" command it would me that the cursor is inside a "link" element, and the command's button should indicate that fact. If not, the command's button should be "disabled" (whatever that might mean for a particular GUI).
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.edit.Unwrap.prototype.isAvailable = function(event) {
	var target = mozile.edit._getTarget(event, this.target, this.direction);
	if(target) return true;
	return false;
}

/**
 * Tests an event to see if the command should be executed.
 * @param {Event} event Optional. The event object to test.
 * @param {Node} targetNode Optional. The node to be removed.
 * @type mozile.edit.State
 */
mozile.edit.Unwrap.prototype.test = function(event, targetNode) {
	if(event) {
		if(this.accel) {
			if(mozile.edit.checkAccelerators(event, this.accels)) { /* do nothing */ }
			if(mozile.edit.checkAccelerator(event, this.accel)) { /* do nothing */ }
			else return false;
		}
		else return false;
	}

	if(!this.target) return false;
	var node = targetNode;
	if(!node) node = mozile.edit._getTarget(event, this.target, this.direction);
	if(!node) return false;

	return true;
}

/**
 * Prepares a state object for the wrap command.
 * @param {Event} event The event object to be converted into a state.
 * @param {Node} targetNode Optional. The node to be removed.
 * @type mozile.edit.State
 */
mozile.edit.Unwrap.prototype.prepare = function(event, target) {
	var state = new mozile.edit.State(this);

	if(!target) target = mozile.edit._getTarget(event, this.target, this.direction);
	state.target = state.storeNode(target);

	// Some commands may need additional information from the user.
	// If so, define a method named "prompt", which takes the event and the state and returns a Boolean true when it succeeds.
	if(this.prompt) {
		if(!this.prompt(event, state)) return null;
	}

	return state;
}

/**
 * Inserts a new element at the selection.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.Unwrap.prototype.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	state.actions = new Array();

	var target = mozile.xpath.getNode(state.target);
	mozile.edit._unwrapNode(state, fresh, target);

	state.selection.after = selection.store();
	state.executed = true;
	return state;
}






/**
 * A command used to replace one element with another.
 * @param {String} name The group's name.
 * @constructor
 */
mozile.edit.Replace = function(name) { 
	/**
	 * The name for this command group.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command does not contain more commands.
	  * @type Boolean
	  */
	 this.group = false;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.makesChanges = "node";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.watchesChanges = "node";
	 
	 /**
	  * Determines what node is the target of the replacement. Can be a string: "element", "block", or "localName [name]"; or a function.
	  * @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.target = "element";
	 
	 /**
	  * Determines what direction to search for the target in. 
	  * @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.direction = "ancestor";
	 
	 /**
	  * Determines whether attributes should be copied into the replacement.
	  * @type Boolean
	  */
	 this.copyAttributes = true;

	 // Register this command group on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}
mozile.edit.Replace.prototype = new mozile.edit.Command;
mozile.edit.Replace.prototype.constructor = mozile.edit.Replace;


/**
 * Indicates that the command is currently available.
 * In the case of an "unlink" command it would me that the cursor is inside a "link" element, and the command's button should indicate that fact. If not, the command's button should be "disabled" (whatever that might mean for a particular GUI).
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.edit.Replace.prototype.isAvailable = function(event) {
	var target = mozile.edit._getTarget(event, this.target, this.direction);
	if(target) return true;
	return false;
}

/**
 * Indicates that the command is currently active.
 * In the case of a "strong" command it would me that the cursor is inside a "strong" element, and the command's button should indicate that fact.
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.edit.Replace.prototype.isActive = function(event) {
	if(this.prompt) return false;
	if(!this.elementName) this.elementName = mozile.edit._getElementName(this);
	if(!this.elementName) return false;

	var target = mozile.edit._getTarget(event, this.target, this.direction);
	if(target) {
		var targetName = mozile.dom.getLocalName(target).toLowerCase();
		if(targetName && targetName == this.elementName) return true;
	}

	return false;
}

/**
 * Tests an event to see if the command should be executed.
 * @param {Event} event Optional. The event object to test.
 * @type mozile.edit.State
 */
mozile.edit.Replace.prototype.test = function(event) {
	if(event) {
		if(this.accel) {
			if(mozile.edit.checkAccelerators(event, this.accels)) { /* do nothing */ }
			if(mozile.edit.checkAccelerator(event, this.accel)) { /* do nothing */ }
			else return false;
		}
		else return false;
	}

	if(!this.element) return false;
	if(!this.target) return false;
	
	var node = mozile.edit._getTarget(event, this.target, this.direction);
	if(!node) return false;

	return true;
}

/**
 * Prepares a state object for the wrap command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.Replace.prototype.prepare = function(event) {
	var state = new mozile.edit.State(this);
	
	// Get the element to replace with.
	state.element = null;
	if(typeof(this.element) == "string") {
		state.element = mozile.dom.createElement(this.element);
	}
	else if(this.element && this.element.cloneNode) {
		state.element = this.element.cloneNode(true);
	}

	var target = mozile.edit._getTarget(event, this.target, this.direction);
	state.target = state.storeNode(target);

	// Copy the target's attributes into the element.
	if(this.copyAttributes) {
		for(var i=0; i < target.attributes.length; i++) {
			var attr = target.attributes[i];
			state.element.setAttribute(attr.nodeName, attr.nodeValue);
		}
		if(target.className) state.element.className = target.className;
		if(target.mozile) {
			state.element.mozile = {};
			for(var key in start.target.mozile) {
				state.element.mozile[key] = target.mozile[key];
			}
		}
	}

	// Some commands may need additional information from the user.
	// If so, define a method named "prompt", which takes the event and the state and returns a Boolean true when it succeeds.
	if(this.prompt) {
		if(!this.prompt(event, state)) return null;
	}

	return state;
}

/**
 * Inserts a new element at the selection.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.Replace.prototype.execute = function(state, fresh) {
	var selection = mozile.dom.selection.get();
	if(!fresh) selection.restore(state.selection.before);
	var range = selection.getRangeAt(0);
	state.actions = new Array();
	
	var target = mozile.xpath.getNode(state.target);
	var focusNode = mozile.xpath.getXPath(selection.focusNode, state.target);
	var focusOffset = selection.focusOffset;

	mozile.edit.insertNode.request(state, fresh, null, target, state.element);

	while(target.firstChild) {
		mozile.edit.moveNode.request(state, fresh, 
			state.element, state.element.lastChild, target.firstChild);		
	}

	mozile.edit.removeNode.request(state, fresh, target);

	var newFocus = mozile.xpath.getNode(focusNode, state.element);
	if(newFocus) selection.collapse(newFocus, focusOffset);
	state.selection.after = selection.store();
	state.executed = true;
	return state;
}




/**
 * A command used to style an element.
 * @param {String} name The group's name.
 * @constructor
 */
mozile.edit.Style = function(name) { 
	/**
	 * The name for this command group.
	 * @type String
	 */
	 this.name = name;
	 
	 /**
	  * Indicates that this command does not contain more commands.
	  * @type Boolean
	  */
	 this.group = false;
	 
	 /**
	  * Specifies what kind of change this command makes to the document.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.makesChanges = "node";
	 
	 /**
	  * Specifies what kind of change will cause this command to change its isActive or isAvailable states.
	  * See mozile.edit.Command.respond for possible values.
	  * @type String
	  */
	 this.watchesChanges = "node";
	 
	 /**
	  * Determines what node is the target of the replacement. Can be a string: "element", "block", or "localName [name]"; or a function.
	  * @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.target = "element";
	 
	 /**
	  * Determines what direction to search for the target in. 
	  * @see #mozile.edit._getTarget
	  * @type String
	  */
	 this.direction = "ancestor";
	 
	 /**
	  * Determines the style to be changed.
	  * @type String
	  */
	 this.styleName = null;
	 
	 /**
	  * Determines the new value for the style. Can be a string or a function.
	  * @type String
	  */
	 this.styleValue = null;

	 // Register this command group on the lost of all commands.
	 mozile.edit.allCommands[this.name] = this;
}
mozile.edit.Style.prototype = new mozile.edit.Command;
mozile.edit.Style.prototype.constructor = mozile.edit.Style;


/**
 * Indicates that the command is currently available.
 * In the case of an "unlink" command it would me that the cursor is inside a "link" element, and the command's button should indicate that fact. If not, the command's button should be "disabled" (whatever that might mean for a particular GUI).
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.edit.Style.prototype.isAvailable = function(event) {
	var target = mozile.edit._getTarget(event, this.target, this.direction);
	if(target) return true;
	return false;
}

/**
 * Indicates that the command is currently active.
 * In the case of a "strong" command it would me that the cursor is inside a "strong" element, and the command's button should indicate that fact.
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.edit.Style.prototype.isActive = function(event) {
	if(this.prompt) return false;
	var styleName = mozile.dom.convertStyleName(this.styleName);
	if(!styleName) return false;

	var target = mozile.edit._getTarget(event, this.target, this.direction);
	if(target && target.style && target.style[styleName] &&
		target.style[styleName] == this.styleValue) return true;

	return false;
}

/**
 * Tests an event to see if the command should be executed.
 * @param {Event} event Optional. The event object to test.
 * @type mozile.edit.State
 */
mozile.edit.Style.prototype.test = function(event) {
	if(event) {
		if(this.accel) {
			if(mozile.edit.checkAccelerators(event, this.accels)) { /* do nothing */ }
			if(mozile.edit.checkAccelerator(event, this.accel)) { /* do nothing */ }
			else return false;
		}
		else return false;
	}

	if(!this.styleName) return false;
	if(!this.styleValue) return false;
	if(!this.target) return false;
	
	var node = mozile.edit._getTarget(event, this.target, this.direction);
	if(!node) return false;
	if(!node.style) return false;

	var state = {target: node};
	state.styleName = this.styleName.replace(/\-(\w)/g, function (strMatch, p1){
		return p1.toUpperCase();
	});
	if(typeof(this.styleValue) == "function") {
		var result = this.styleValue(event, state);
		if(result === null) return false;
	}

	return true;
}

/**
 * Prepares a state object for the wrap command.
 * @param {Event} event The event object to be converted into a state.
 * @type mozile.edit.State
 */
mozile.edit.Style.prototype.prepare = function(event) {
	var state = new mozile.edit.State(this);

	var target = mozile.edit._getTarget(event, this.target, this.direction);
	state.target = state.storeNode(target);
	
	state.styleName = this.styleName.replace(/\-(\w)/g, function (strMatch, p1){
		return p1.toUpperCase();
	});
	state.styleValue = null;
	if(typeof(this.styleValue) == "function")
		state.styleValue = this.styleValue(event, state);
	else if(typeof(this.styleValue) == "string")
		state.styleValue = this.styleValue;
	state.oldValue = null;

	// Some commands may need additional information from the user.
	// If so, define a method named "prompt", which takes the event and the state and returns a Boolean true when it succeeds.
	if(this.prompt) {
		if(!this.prompt(event, state)) return null;
	}

	return state;
}

/**
 * Inserts a new element at the selection.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.edit.Style.prototype.execute = function(state, fresh) {
	var target = mozile.xpath.getNode(state.target);
	state.targetNode = target;

	state.oldValue = target.style[state.styleName];
	target.style[state.styleName] = state.styleValue;

	state.executed = true;
	return state;
}

/**
 * Removes all wrapper elements.
 * @param {mozile.edit.State} state The state information needed to unexecute this command.
 * @type Object
 */
mozile.edit.Style.prototype.unexecute = function(state, fresh) {
	state.targetNode.style[state.styleName] = state.oldValue;

	state.executed = false;
	return state;
}







/**** Support Methods ****/
/* 
 * NOTE: These methods are meant to be called by commands, but aren't themselves commands.
 */


/**
 * Merge two nodes.
 * This command operates by calling the mergeNodes command.
 * @private
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @param {Text} firstNode The first text node.
 * @param {Text} secondNode The second text node.
 * @type Boolean
 */
mozile.edit._mergeNodes = function(state, fresh, firstNode, secondNode) {
	var firstBlock = mozile.edit.getParentBlock(firstNode);
	var secondBlock = mozile.edit.getParentBlock(secondNode);
	
	//alert("Merge\n"+ mozile.xpath.getXPath(firstNode) +" "+ mozile.xpath.getXPath(secondNode) +"\n"+ mozile.xpath.getXPath(firstBlock) +"\n"+ mozile.xpath.getXPath(secondBlock));
	
	// TODO: Decide when to merge properly.
	// Currently only blocks with matching names are merged.
	if(!firstBlock || !secondBlock) return false;
	if(firstBlock == secondBlock) {
		return mozile.edit._normalize(state, fresh, firstNode, secondNode);
	}

	// Continue to merge if the elements continue to share the same name.
	var lastChild = firstBlock;
	var firstChild = secondBlock;
	var newState;
	while(lastChild.nodeType == mozile.dom.ELEMENT_NODE &&
		firstChild.nodeType == mozile.dom.ELEMENT_NODE &&
		lastChild.nodeName == firstChild.nodeName) {
		var from = firstChild;
		var to = lastChild;
		// Prepare for next iteration.
		lastChild = lastChild.lastChild;
		firstChild = firstChild.firstChild;
		mozile.edit.mergeNodes.request(state, fresh, from, to);
	}
	mozile.edit._normalize(state, fresh, lastChild, firstChild);

	if(lastChild == firstBlock) return false;
	else return true;
}

/**
 * Merge two text nodes.
 * This command operates by calling the mergeNodes command.
 * @private
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @param {Text} firstNode The first text node.
 * @param {Text} secondNode The second text node.
 * @type mozile.edit.State
 */
mozile.edit._normalize = function(state, fresh, firstNode, secondNode) {
	//alert("Normalize? "+ mozile.xpath.getXPath(firstNode) +"\n"+ mozile.xpath.getXPath(secondNode));
	if(!firstNode || !firstNode.parentNode ||
		firstNode.nodeType != mozile.dom.TEXT_NODE) {
		//alert("Non text "+ mozile.xpath.getXPath(firstNode));
		return false;
	}
	if(!secondNode || !secondNode.parentNode ||
		secondNode.nodeType != mozile.dom.TEXT_NODE) {
		//alert("Non text "+ mozile.xpath.getXPath(secondNode));
		return false;
	}
	if(firstNode.nextSibling != secondNode) return false;

	return mozile.edit.mergeNodes.request(state, fresh, firstNode, secondNode);
}

/**
 * Remove a node if it is empty.
 * This command operates by calling the removeNode command.
 * @private
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @param {Node} target The node to check for removal.
 * @type Void
 */
mozile.edit._removeEmpty = function(state, fresh, target, limitNode) {
	var selection = mozile.dom.selection.get();
	if(!target || !target.parentNode) return null;
	var parent = target.parentNode;
	var parentBlock = mozile.edit.getParentBlock(target);

	if(typeof(state.direction) == "undefined")
		state.direction = mozile.edit.PREVIOUS;
	var IP = mozile.edit.getInsertionPoint(target, -1 * state.direction);
	if(!IP) return null;
	var result = IP.seekNode(-1 * state.direction, false);
	if(!result) IP.seekNode(state.direction, false); 
	if( (target.nodeType == mozile.dom.TEXT_NODE &&
		target.data.length == 0) ||
		(target.nodeType == mozile.dom.ELEMENT_NODE &&
		target != mozile.edit.getContainer(target) &&
		target.childNodes.length == 0) ) {
		
		// Create an empty token.
		if(target == limitNode) {
			var content = mozile.edit.createEmptyToken();
			mozile.edit.insertNode.request(state, fresh, target, null, content);
			selection.collapse(content, 0);
			return null;
		}

		// Remove the node
		mozile.edit.removeNode.request(state, fresh, target);
		
		// Make sure the selection is good.
		if(state.direction == mozile.edit.PREVIOUS) {
			if(parentBlock == target ||
				parentBlock == mozile.edit.getParentBlock(IP.getNode())) {
				//alert(mozile.xpath.getXPath(parentBlock) +"\n"+ IP);
				//alert(mozile.xml.serialize(parentBlock));
				IP.select();
			}
			// If the IP is in a new block, get the last IP from this block instead.
			else {
				IP = mozile.edit.getInsertionPoint(parentBlock, state.direction);
				if(IP) IP.select();
				else mozile.debug.debug("mozile.edit._removeEmpty", "Nowhere to move the insertion point.");
			}
		}
		else IP.select();
		
		// Normalize remaining nodes.
		var firstNode, secondNode;
		if(state.direction == mozile.edit.PREVIOUS) {
			secondNode = IP.getNode();
			IP.seekNode(state.direction, false);
			firstNode = IP.getNode();
		}
		else {
			firstNode = IP.getNode();
			IP.seekNode(state.direction, false);
			secondNode = IP.getNode();
		}
		var result = mozile.edit._normalize(state, fresh, firstNode, secondNode);
		//alert("Normalized "+ result);
	}
	else {
		// Do nothing.
		return null;
	}

	if(target == limitNode) return null;
	else return mozile.edit._removeEmpty(state, fresh, parent, limitNode);
}

/**
 * Remove any empty token nodes from the target.
 * @private
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @param {Node} target The node to remove.
 * @type Void
 */
mozile.edit._removeEmptyTokens = function(state, fresh, target) {
	var node = target.firstChild;
	while(node) {	
		if(mozile.edit.isEmptyToken(node)) {
			var content = node;
			node = node.nextSibling;
			mozile.edit.removeNode.request(state, fresh, content);
		}
		else node = node.nextSibling;
	}
	return state;
}

/**
 * Checks to see if a node has no non-whitespace text, and if it has none an empty token is added.
 * @private
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @param {Node} target The element to check for emptiness.
 * @type Boolean
 */
mozile.edit._ensureNonEmpty = function(state, fresh, target) {
	if(!state || !target) return false;
	if(!target.nodeType) return false;

	if(!mozile.edit.isEmpty(target)) return true;

	switch(target.nodeType) {
		case mozile.dom.TEXT_NODE:
			mozile.edit.insertText.request(state, fresh, null, mozile.emptyToken, target);
			return true;

		case mozile.dom.ELEMENT_NODE:
			// If there's no RNG, or the RNG allows it, insert an emptyToken.
			var rng = mozile.edit.lookupRNG(target);
			if((rng && rng.mayContain("text")) || !rng) {
				// Try to use an existing text node.
				for(var i=0; i < target.childNodes.length; i++) {
					if(target.childNodes[i].nodeType == mozile.dom.TEXT_NODE) {
						return mozile.edit._ensureNonEmpty(state, fresh, target.childNodes[i]);
					}
				}
				// If that fails, insert a text node.
				var emptyToken = mozile.edit.createEmptyToken();
				mozile.edit.insertNode.request(state, fresh, 
					target, target.lastChild, emptyToken);
				return true;
			}
			// Otherwise try the child nodes.
			else if(target.firstChild) {
				var child = target.firstChild;
				var result;
				while(child) {
					result = mozile.edit._ensureNonEmpty(state, fresh, child);
					if(result) return true;
					child = child.nextSibling;
				}
			}
			return false;

		default: 
			return false;
	}
}

/**
 * Remove content from a range.
 * The method works as follows:
 * <ul>
 *   <li>Using a treeWalker, each node between the startContainer and endContianer is removed (unless it is an ancestor of one of the containers),
 *   <li>Selected text is removed from the startContainer,
 *   <li>Selected text is removed from the endContainer,
 *   <li>Remaining text nodes are normalized.
 * </ul>
 * This command operates by calling the removeText and removeNode commands.
 * @private
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @param {Integer} direction Optional. A code for the direction to delete in.
 * @type Object
 */
mozile.edit._removeRange = function(state, fresh, direction) {
	var selection = mozile.dom.selection.get();
	//if(!fresh) selection.restore(state.selection.before);
	var range = selection.getRangeAt(0);
	var container = range.commonAncestorContainer;
	if(!direction) direction = mozile.edit.PREVIOUS;
	
	// Get start and end nodes.
	var startNode = range.startContainer;
	if(startNode.nodeType == mozile.dom.ELEMENT_NODE)
		startNode = startNode.childNodes[range.startOffset];
	var endNode = range.endContainer;
	if(endNode.nodeType == mozile.dom.ELEMENT_NODE)
		endNode = endNode.childNodes[range.endOffset];

	// Remove all the nodes between the start and end node, but not the ancestors of the start and end nodes.
	var treeWalker = document.createTreeWalker(container, mozile.dom.NodeFilter.SHOW_ALL, null, false);
	treeWalker.currentNode = startNode;
	var current = treeWalker.nextNode();

	// Loop until the endNode is found (or we run out of nodes).
	while(current) {
		// Check to see if the node should NOT be removed.
		if(!current.parentNode) break;
		if(current == endNode) break;
		if(mozile.dom.isAncestorOf(current, startNode, container))
			current = treeWalker.nextNode();
		else if(mozile.dom.isAncestorOf(current, endNode, container))
			current = treeWalker.nextNode();

		// Remove the node.
		else {
			var target = current;
			current = treeWalker.nextSibling();
			if(!current) current = treeWalker.nextNode();
			mozile.edit.removeNode.request(state, fresh, target);
		}
	}

	// Take care of the startNode and endNode.
	var data;
	if(startNode.nodeType == mozile.dom.TEXT_NODE) {
		data = startNode.data.substring(0, range.startOffset);
		mozile.edit.insertText.request(state, fresh, null, data, startNode);
	}
	else mozile.edit.removeNode.request(state, fresh, startNode);

	if(endNode.nodeType == mozile.dom.TEXT_NODE) {
		data = endNode.data.substring(range.endOffset);
		mozile.edit.insertText.request(state, fresh, null, data, endNode);
	}

	// Set the cursor position.
	if(direction == mozile.edit.NEXT) {
		if(endNode && endNode.nodeType == mozile.dom.TEXT_NODE)
			selection.collapse(endNode, 0);
		else if(startNode && startNode.nodeType == mozile.dom.TEXT_NODE)
			selection.collapse(startNode, startNode.data.length);
		else mozile.debug.debug("mozile.edit._removeRange", "Nowhere to collapse.");
	}
	else {
		if(startNode && startNode.nodeType == mozile.dom.TEXT_NODE)
			selection.collapse(startNode, startNode.data.length);
		else if(endNode && endNode.nodeType == mozile.dom.TEXT_NODE)
			selection.collapse(endNode, 0);
		else mozile.debug.debug("mozile.edit._removeRange", "Nowhere to collapse.");
	}

	// Merge and normalize the start and end containers
	if(startNode && endNode && startNode.parentNode && endNode.parentNode) {
		var result = mozile.edit._mergeNodes(state, fresh, startNode, endNode);
		// Move to next block.
		if(!result && direction == mozile.edit.NEXT &&
			mozile.edit.getParentBlock(startNode) != mozile.edit.getParentBlock(endNode)) {
			selection.collapse(endNode, 0);
		}
	}

	return state;
}


/**
 * This method moves all of a target node's children out of that node, and then removes the node. Returns the last node that was in the wrapper.
 * @private
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @param {Node} target The node to remove.
 * @type Node
 */
mozile.edit._unwrapNode = function(state, fresh, target) {
	var previousNode = target.previousSibling;
	var nextNode = target.nextSibling;
	var lastChild;

	// Move all children out of the target node.
	while(target.lastChild) {
		lastChild = target.lastChild;
		mozile.edit.moveNode.request(state, fresh, null, target, target.lastChild);
	}

	// Remove the target.
	mozile.edit.removeNode.request(state, fresh, target);

	// Normalize text nodes.
	mozile.edit._normalize(state, fresh, nextNode.previousSibling, nextNode);
	mozile.edit._normalize(state, fresh, previousNode, previousNode.nextSibling);

	return lastChild;
}


/**
 * Tools for dealing with browser events.
 * @type Object
 */
mozile.event = new Object();
// JSDoc hack
mozile.event.prototype = new mozile.Module;


/**
 * Indicates that a mouse button is being held down. 
 * @type Boolean
 */
mozile.event.mousedown = false;


/**
 * Changes some of IE's event properties to use standard names.
 * @param {Event} e The event to normalize.
 * @type Object
 */
mozile.event.normalize = function(event) {
	if(!event) return event;
	if(typeof(event.target) == "undefined") event.target = event.srcElement;
	if(typeof(event.charCode) == "undefined") event.charCode = event.keyCode;
	return event;
}

/**
 * Adds an event listener to a node.
 * @param {Document} doc The document to attach the listener to.
 * @param {String} type The event type to listen for. Do NOT include the "on" prefix!
 * @param {Function} listener Optional. The function to call when the event occurs. Defailts to mozile.event.handle.
 * @param {Boolean} useCapture Optional. See the documentation for Document.addEventListener().
 * @type Void
 */
mozile.event.addListener = function(doc, type, listener, useCapture) {
	if(!listener) listener = mozile.event.handle;
	if(doc.addEventListener) doc.addEventListener(type, listener, useCapture);
	else if(doc.attachEvent) doc.attachEvent("on" + type, listener);
	else mozile.debug.inform("mozile.event.addListener", "No known event method available");
}

/**
 * Set up a series of event listeners on this document.
 * @type Void
 */
mozile.event.listen = function() {
	var events = ["mousedown", "mousemove", "mouseup", "click", "dblclick", "keydown", "keyup", "keypress"];
	for(var i=0; i < events.length; i++) {
		mozile.event.addListener(document, events[i]);
	}
}

/**
 * INCOMPLETE. Dispatch (fire) an event of a given type from the given element.
 * @param {Element} element The element fom which the event will be dispatched.
 * @param {String} type The type of event to dispatch.
 * @type Void
 */
mozile.event.dispatch = function(element, type, keyCode, charCode, ctrlKey, altKey, shiftKey, metaKey) {
	// IE event
	if(element.fireEvent) element.fireEvent("on"+type);
	// TODO: Other browsers
	else if(element.dispatchEvent) {
		var event;
		if(type.indexOf("click") > -1 || type.indexOf("mouse") > -1) {
			event = document.createEvent("MouseEvent");
			event.initMouseEvent(type, true, true, window, 1, 0, 0, 0, 0, false, false, false, false, 1, null);
		}
		else if(type.indexOf("key") > -1) {
			if(!ctrlKey) ctrlKey = false;
			if(!altKey) altKey = false;
			if(!shiftKey) shiftKey = false;
			if(!metaKey) metaKey = false;
			event = document.createEvent("KeyEvent");
			event.initKeyEvent(type, true, true, document.defaultView, ctrlKey, altKey, shiftKey, metaKey, keyCode, charCode);
		}
		else {
			event = document.createEvent("Events");
			event.initEvent(type, true, true);
		}
		element.dispatchEvent(event);
	}
}

/**
 * Cancels an event.
 * @param {Event} event The event to cancel.
 * @type Void
 */
mozile.event.cancel = function(event) {
	if(!event) return;
	if(event.stopPropagation) event.stopPropagation();
	event.cancelBubble = true;
	if(event.preventDefault) event.preventDefault();
}

/**
 * General event handler, which calls specific event handlers for the document and for the target nodes of the event.
 * @param {Event} event The event to handle.
 * @type Boolean
 * @return False if the event was cancelled, true if it was not cancelled.
 */
mozile.event.handle = function(event) {
try {
	switch(event.type) {
		case "mousemove":
			return mozile.event.handleMouseMove(event);
		case "mouseup":
		case "click":
		case "dblclick":
			mozile.event.mousedown = false;
	}

	event = mozile.event.normalize(event);
	mozile.event.findTarget(event);
	if(!event.node) return true;
	event.container = mozile.edit.getContainer(event.node);
	event.editable = Boolean(event.container);

	// Check global commands
	var state = mozile.edit.handleEvent(event);
	if(state) return mozile.event.handled(event, state);
	
	// Set editable state
	if(mozile.gui) mozile.gui.update(event);
	if(!mozile.edit.editable) return true;
	var priorStatus = mozile.edit.status;
	mozile.edit.setStatus(event.editable);
	if(!event.editable) return true;
	
	mozile.event.storeSelection(event);
	if(!priorStatus) mozile.event.fixFocus(event);

	// Check RNG Element commands for this node and editable ancestors.
	var node = event.node;
	while(event.node) {
		if(!event.rng) event.rng = mozile.edit.lookupRNG(event.node);
		if(event.rng) {
			state = event.rng.handleEvent(event);
			if(state) return mozile.event.handled(event, state);
		}
		if(event.node == event.container) break;
		else event.node = event.node.parentNode;
	}
	
	// Check default editing commands.
	event.node = node;
	state = mozile.edit.handleDefault(event);
	if(state) return mozile.event.handled(event, state);
	
	if(!mozile.event.cancelKeyEvent(event)) return false;
	if(!mozile.event.cancelHyperlink(event)) return false;

} catch(e) { mozile.debug.inform("mozile.event.handle", mozile.dumpError(e)); }

	return true;
}

/**
 * Finish handling an event, eieht by cancelling it or not.
 * @param {Event} event The event to handle.
 * @type Boolean
 */
mozile.event.handled = function(event, state) {
	mozile.edit.done(state);
	if(state.changesMade) {
		//mozile.debug.debug("mozile.event.handled", state.command.name +" made change "+ state.changesMade);
		if(mozile.gui) mozile.gui.update(event, state.changesMade);
	}
	if(state.cancel) {
		mozile.event.cancel(event);
		return false;
	}
	else return true;
}

/**
 * Handles mouse move events. Part of a hack for handling selections inside links in Mozilla. See cancelHyperlink() for the rest.
 * @param {Event} event The event to handle.
 * @type Boolean
 */
mozile.event.handleMouseMove = function(event) {
	if(!mozile.browser.isMozilla) return true;
	if(!mozile.event.mousedown) return true;
	var selection = mozile.dom.selection.get();
	if(!selection) return true;

	if(selection.focusNode != event.rangeParent ||
		selection.focusOffset != event.rangeOffset) {
		selection.extend(event.rangeParent, event.rangeOffset);
	}

	return true;
}

/**
 * Finds the target of an event. For a mouse event this will be the event.target. For a keyboard event we look at the selection's commonAncestorContainer.
 * @param {Event} event The event to handle.
 * @type Void
 */
mozile.event.findTarget = function(event) {
	event.selection = mozile.dom.selection.get();
	//if(event.type.indexOf("key") == 0) {
		if(!event.selection || event.selection.rangeCount < 1) return;
		event.range = event.selection.getRangeAt(0);
		if(!event.range) return;
		event.node = event.range.commonAncestorContainer;
		//mozile.debug.debug("mozile.event.findTarget", [event.type, event.charCode, mozile.xpath.getXPath(event.node)].join(", "));
	//}
	//else event.node = event.target;
}


/**
 * Store the current selection so it can be restored. Used to maintain a selection in the text while the Mozile GUI is being used.
 * @param {Event} event The event to handle.
 * @type Void
 */
mozile.event.storeSelection = function(event) {
	mozile.dom.selection.last = {
		anchorNode: event.selection.anchorNode,
		anchorOffset: event.selection.anchorOffset, 
		focusNode: event.selection.focusNode, 
		focusOffset: event.selection.focusOffset,
		isCollapsed: event.selection.isCollapsed
	};
}

/**
 * Makes sure the editable area is focussed. This is a hack to fix a bug when activiating Mozila's designMode. The method is to duplicate the mousedown event and send it again, esentailly clicking twice.
 * @param {Event} event The event to handle.
 * @type Void
 */
mozile.event.fixFocus = function(event) {
	if(!mozile.browser.isMozilla) return;
	if(!mozile.useDesignMode) return;
	if(event.type != "mousedown") return;
	
	var newEvent = document.createEvent("MouseEvent");
	newEvent.initMouseEvent("mousedown", true, true, event.view, 1, event.screenX, event.screenY, event.clientX, event.clientY, false, false, false, false, 1, event.relatedTarget);
	event.target.dispatchEvent(newEvent);
}

/**
 * Cancels certain keyboard events and cancels them.
 * @param {Event} event The event to handle.
 * @type Boolean
 * @return false means the event has been cancelled.
 */
mozile.event.cancelKeyEvent = function(event) {
	if(!event || !event.keyCode) return true;
	switch(event.keyCode) {
		case 8: // Backspace
		case 9: // Tab
		case 46: // Delete
			mozile.event.cancel(event);
			return false;
	}
	return true;
}

/**
 * Cancels events which would lead Mozilla to follow a hyperlink while editing.
 * @param {Event} event The event to handle.
 * @type Boolean
 * @return false means the event has been cancelled.
 */
mozile.event.cancelHyperlink = function(event) {
	if(!mozile.browser.isMozilla) return true;
	if(mozile.useDesignMode) return true;

	// Only handle certain mouse events.
	switch(event.type) {
		case "mousedown":
		case "click":
		case "dblclick":
		case "mouseup":
			break;
		default: return true;
	}

	var node = event.explicitOriginalTarget;
	var container = mozile.edit.getContainer(node);
	if(container) {
		while(node) {
			if(node.localName && node.localName.toLowerCase() == "a") {
				//mozile.require("mozile.util");
				//mozile.debug.debug("", mozile.util.dumpValues(event));
				if(event.selection && event.rangeParent && 
					event.rangeOffset != undefined) {
					if(event.type == "mousedown") {
						event.selection.collapse(event.rangeParent, event.rangeOffset);
						mozile.event.mousedown = true;
					}
					else mozile.event.mousedown = false;
					if(event.type == "dblclick") {
						mozile.event.selectWord(event.rangeParent, event.rangeOffset);
					}
				}
				mozile.event.cancel(event);
				return false;
			}
			if(node == container) break;
			node = node.parentNode;
		}
	}

	return true;
}

/**
 * Selects the word inside the given node which contains the given offset. Tries to immitate Mozilla's double-click behaviour.
 * @param {Node} node
 * @param {Integer} offset
 * @type Void
 */
mozile.event.selectWord = function(node, offset) {
	if(!node || offset == undefined) return;

	var selection = mozile.dom.selection.get();
	if(node.nodeType != mozile.dom.TEXT_NODE) {
		selection.collapse(node, offset);
	}

	else {
		//mozile.debug.debug("mozile.event.selectWord", mozile.xpath.getXPath(node) +" "+ offset);
		var match = /\s/;
		var data = node.data;

		// Stupid bug: the event.rangeParent is always the first text node
		// Ignore that case
		if(offset == data.length) return;
		
		var startOffset = offset - 2;
		while(startOffset >= 0) {
			if(data.charAt(startOffset).match(match)) {
				startOffset++;
				break;
			}
			else startOffset--;
		}
		if(startOffset < 0) startOffset = 0;

		var endOffset = offset + 1;
		while(endOffset <= data.length) {
			if(data.charAt(endOffset).match(match)) break;
			else endOffset++;
		}
		if(endOffset > data.length) endOffset = data.length;

		selection.collapse(node, startOffset);
		selection.extend(node, endOffset);
	}	
	
}


/**** Final Configuration ****/

// Activate the listeners.
mozile.event.listen();






/**
 * Tools for saving Mozile results.
 * @type Object
 */
mozile.save = new Object();
// JSDoc hack
mozile.save.prototype = new mozile.Module;

/**
 * Determines which save method will be used to save the document.
 * @type Object
 */
mozile.save.method = null;

/**
 * Determines the node to save.
 * @type Node
 */
mozile.save.target = document;

/**
 * Determines the string format to use for saving.
 * @type String
 */
mozile.save.format = null;

/**
 * The last saved state. Compare this with the mozile.edit.currentState to determine if the document needs to be saved.
 * @type mozile.edit.State
 */
mozile.save.savedState = null;

/**
 * Indicates that the user should be warned before leaving a page with unsaved changes.
 * @type Boolean
 */
mozile.save.warn = true;

/**
 * Add a BeforeUnload handler to present a warning if the document hasn't been saved.
 * @type String
 */
window.onbeforeunload = function() {
	if(!mozile.save.warn) return undefined;
	if(mozile.save.isSaved()) return undefined;
	return "There are unsaved changes in this document. Changes will be lost if you navigate away from this page.";
}

/**
 * Determines whether the document has been saved in its current state.
 * @type Boolean
 * @return true if no changes have been made since the document was last saved.
 */
mozile.save.isSaved = function() {
	// No changes can be made without the edit module.
	if(!mozile.edit) return true;
	// If the current state is null, no changes have been made.
	if(!mozile.edit.currentState) return true;
	// Otherwise, compare the currentState to the savedState.
	if(mozile.edit.currentState != mozile.save.savedState) return false;
	return true;
}

/**
 * Saves the document using the current mozile.save.method.save().
 * @type Boolean
 * @return True if the save operation was successful.
 */
mozile.save.save = function() {
	if(!mozile.save.method) return false;
	var content = mozile.save.getContent(mozile.save.target, mozile.save.format);
	var result = mozile.save.method.save(content);
	if(result && mozile.edit) mozile.save.savedState = mozile.edit.currentState;
	return result;
}

/**
 * Saves the document using the current mozile.save.method.saveAs().
 * @type Boolean
 * @return True if the save operation was successful.
 */
mozile.save.saveAs = function() {
	if(!mozile.save.method) return false;
	var content = mozile.save.getContent(mozile.save.target, mozile.save.format);
	var result = mozile.save.method.saveAs(content);
	if(result && mozile.edit) mozile.save.savedState = mozile.edit.currentState;
	return result;
}

/**
 * Gets the content of a document or a node as a string.
 * @param {Node} target Optional. Get the content of this node. If no node is given the documentElement is used.
 * @param {String} format Optional. The format of the string. Can be "uppercase" or "lowercase". If none is given no formatting is done.
 * @type String
 */
mozile.save.getContent = function(target, format) {
	var content = "";
	if(!target) target = document;
	if(target.nodeType == 9) { // DOCUMENT_NODE
		target = target.documentElement;
		content += mozile.save.getXMLDeclaration();
		content += mozile.save.getDoctypeDeclaration();
		content += mozile.save.getProcessingInstructions();
	}

	// Make a copy of the target and clean it.
	target = target.cloneNode(true);
	target = mozile.save.cleanDOM(target);
	
	content += mozile.xml.serialize(target);
	if(format) {
		if(format.toLowerCase() == "uppercase")
			content = mozile.save.toUpperCase(content);
		else if(format.toLowerCase() == "lowercase")
			content = mozile.save.toLowerCase(content);
	}
	content = mozile.save.cleanContent(content);
	
	return content;
}

/** 
 * The XML declaration for the document, if there is one.
 * @type String
 */
mozile.save.getXMLDeclaration = function() {
	var xmlDeclaration = "";
	if(document.xmlVersion) {
		xmlDeclaration = '<?xml version="'+ document.xmlVersion +'" encoding="'+ document.xmlEncoding +'"?>\n'
	}
	return xmlDeclaration;
}

/** 
 * The DOCTYPE declaration for the document, if there is one.
 * @type String
 */
mozile.save.getDoctypeDeclaration = function() {
	var doctypeDeclaration = "";
	if(document.doctype) {
		doctypeDeclaration = mozile.xml.serialize(document.doctype) +"\n";
	}
	return doctypeDeclaration;
}

/** 
 * The Processing Instructions for the document, if there are any.
 * @type String
 */
mozile.save.getProcessingInstructions = function() {
	var PIString = "";
	if(window.XPathEvaluator) {
		var	evaluator =	new	XPathEvaluator();
		var PIList = evaluator.evaluate("/processing-instruction()", document, null, XPathResult.ANY_TYPE, null);
		var PI = PIList.iterateNext();
		while (PI) {
			PIString += "<?"+ PI.target +" "+ PI.data + "?>\n";
			PI = PIList.iterateNext();
		}
	}
	return PIString;
}

/**
 * Cleans the DOM tree of a node, removing any changes that Mozile has made.
 * @param {Node} target The node to clean.
 * @type String
 */
mozile.save.cleanDOM = function(target) {
	if(document.createTreeWalker && mozile.dom.NodeFilter) {
		var treeWalker = document.createTreeWalker(target, mozile.dom.NodeFilter.SHOW_ALL, null, false);
		treeWalker.currentNode = target;
		var current = treeWalker.currentNode;
		var remove = new Array();
		
		// Mark nodes for removal.
		while(current) {
			// Clean up after mozile.dom.addStyleSheet
			if(current.getAttribute && current.getAttribute("class") == "mozileLink") remove.push(current);
			if(current.className && current.className == "mozileLink") remove.push(current);

			// Clean up after the GUI
			if(current.getAttribute && current.getAttribute("class") == "mozileGUI") remove.push(current);
			if(current.className && current.className == "mozileGUI") remove.push(current);
			
			// TODO: Clean up changes to contentEdtiable attributes.
			//if(current.mozile) alert("Mozile found "+ current.nodeName);
			
			current = treeWalker.nextNode();
		}

		// Remove nodes.
		while(remove.length) {
			if(remove[0].parentNode) remove[0].parentNode.removeChild(remove[0]);
			remove.shift();
		}
		
	}
	else mozile.debug.inform("mozile.save.cleanDOM", "Could not clean target because no TreeWalker is available.");

	return target;
}

/**
 * Cleans the XML content string from getContent(), removing any changes that Mozile has made.
 * @param {String} content The string to clean.
 * @type String
 */
mozile.save.cleanContent = function(content) {
	return content;
}

/**
 * Converts &lt; and &gt; characters.
 * @param {String} content The string to clean.
 * @type String
 */
mozile.save.cleanMarkup = function(content) {
	content = content.replace(/</g, "&lt;");
	content = content.replace(/>/g, "&gt;");
	return content;
}

/**
 * A regular expression to match the beginning of XML and HTML tags.
 * @private
 * @type RegExp
 */
mozile.save._tagPattern = /<(\/*)(\w*)/g; /* match tags */;

/**
 * Converts the element names of an XML string to upper case.
 * @param {String} content
 * @type String
 */
mozile.save.toUpperCase = function(content) {
	return content.replace(mozile.save._tagPattern, function(word) { return word.toUpperCase(); });
}

/**
 * Converts the element names of an XML string to lower case.
 * @param {String} content
 * @type String
 */
mozile.save.toLowerCase = function(content) {
	return content.replace(mozile.save._tagPattern, function(word) { return word.toLowerCase(); });
}

/**
 * Class for save methods.
 * @constructor
 * @param {String} name A name for the save method.
 */
mozile.save.Method = function(name) {
	/**
	 * The name of the save method.
	 * @type String
	 */
	this.name = name;
}

/**
 * An abstract save method.
 * @param {String} content The content to save.
 * @type Boolean
 */
mozile.save.Method.prototype.save = function(content) {
	return false;
}

/**
 * An abstract saveAs method.
 * @param {String} content The content to save.
 * @type Boolean
 */
mozile.save.Method.prototype.saveAs = function(content) {
	return this.save(content);
}


/**** Display Source Code ****/

/**
 * Display the content in a new window.
 * @type mozile.save.Method
 */
mozile.save.display = new mozile.save.Method("Display Source");

/**
 * @param {String} content The content to save.
 * @type Boolean
 */
mozile.save.display.save = function(content) {
	content = mozile.save.cleanMarkup(content);
	if(mozile.gui) {
		mozile.gui.display('<h3>Mozile Source</h3>\n<pre>'+ content +'</pre>');
	}
	else alert("Mozile Source\n\n"+ content);
	return true;
}

// Set the default save method.
mozile.save.method = mozile.save.display;



/**** Save Via POST ****/

/**
 * Save the content using HTTP POST.
 * @type mozile.save.Method
 */
mozile.save.post = new mozile.save.Method("POST");

/**
 * Indicates an asynchronous request.
 * @type Boolean
 */
mozile.save.post.async = true;

/**
 * Indicates that the response to the save request should be displayed.
 * @type Boolean
 */
mozile.save.post.showResponse = false;

/**
 * The URI to post to.
 * @type String
 */
mozile.save.post.uri = "";

/**
 * The user for the post operation.
 * @type String
 */
mozile.save.post.user = null;

/**
 * The password for the post operation.
 * @type String
 */
mozile.save.post.password = null;

/**
 * The contentType to use.
 * @type String
 */
mozile.save.post.contentType = "text/html";
if(document.contentType) mozile.save.post.contentType = document.contentType;

/**
 * The character set to use.
 * @type String
 */
mozile.save.post.characterSet = "UTF-8";
if(document.characterSet) mozile.save.post.characterSet = document.characterSet;

/**
 * Save using HTTP POST and the XMLHttpRequest object.
 * @param {String} content The content to save.
 * @type Boolean
 */
mozile.save.post.save = function(content) {
	if(!this.uri) {
		if(mozile.debug) mozile.debug.inform("mozile.save.post.save", "No URI to save to.");
		return false;
	}
	
	var CR = '\x0D';
	var LF = '\x0A';
	content = CR + LF + content + CR + LF;

	if(this.XHR) this.XHR.abort();
	this.XHR = null;
	var XHR;
	try{
		if(window.XMLHttpRequest) {
			XHR = new XMLHttpRequest();
		}
		else if(window.ActiveXObject) {
			XHR = new ActiveXObject('Microsoft.XMLHTTP');
		}
	} catch(e) { 
		if(mozile.debug) mozile.debug.inform("mozile.save.post.save", "File save failed for '"+ this.uri +"' with error message:\n"+ e);
		return false;
	}
	
	if(XHR) {	
		XHR.open("POST", this.uri, this.async, this.user, this.password);
		XHR.setRequestHeader('Content-Type', this.contentType + "; " + this.characterSet);	
		if(mozile.browser.mozile && mozile.browser.mozileVersion < 1.8) 
			XHR.setRequestHeader('Content-Length', content.length);
		XHR.setRequestHeader('Content-Location', this.uri);
		XHR.onreadystatechange = this.onreadystatechange;
		XHR.send(content);
		
		this.XHR = XHR;
		if(!this.async) {
			this.onreadystatechange();
		}
		return true;
	}
	
	if(mozile.debug) mozile.debug.inform("mozile.save.post.save", "No XMLHttpRequest available when trying to save to '"+ this.uri +"'.");
	return false;
}

/**
 * Handler for an asynchronous request.
 * @type Void
 */
mozile.save.post.onreadystatechange = function() {
	var XHR = mozile.save.post.XHR;
	if(!XHR) return;
	// If the request is not complete, ignore this change.
	if(XHR.readyState != 4) return;
	
	if(XHR.status == 0 || XHR.status == 200) {
		if(mozile.save.post.showResponse) 
			mozile.gui.display('<h3>Save Operation Response</h3>\n\n'+ XHR.responseText);
	}
	else {
		if(mozile.save.post.showResponse) 
			mozile.gui.display('<h3>Save Operation Error</h3>\n\n'+ XHR.responseText);
		else if(mozile.debug) mozile.debug.inform("mozile.save.post.save", "File save failed with status '"+ XHR.status +"' and message:\n"+ XHR.responseText);
	}
}

/**
 * Set the URI before saving.
 * @param {String} content The content to save.
 * @type Boolean
 */
mozile.save.post.saveAs = function(content) {
	var uri = prompt("Save to what URI?", this.uri);
	if(!uri) return false;
	this.uri = uri;
	return this.save(content);
}




/**
 * Tools for displaying a graphical user interface.
 * @type Object
 */
mozile.gui = new Object();
// JSDoc hack
mozile.gui.prototype = new mozile.Module;

/**
 * Points to the current GUI factory instance.
 * @type mozile.gui.Factory
 */
mozile.gui.factory = null;

/**
 * Creates a GUI using the current factory.
 * @type Void
 */
mozile.gui.create = function() {
	if(!mozile.gui.factory) throw("Error [mozile.gui.create]: No GUI factory selected.");
	else return mozile.gui.factory.create();
}

/**
 * Updates the factory using the current event.
 * @param {Event} event The current event object.
 * @param {String} change Optional. The type of change made. See mozile.edit.Command.makeChange for possible values.
 * @type Boolean
 * @return True if the GUI has been updated, false otherwise.
 */
mozile.gui.update = function(event, change) {
	if(!mozile.gui.factory) throw("Error [mozile.gui.update]: No GUI factory selected.");
	else return mozile.gui.factory.update(event, change);
}

/**
 * Displays the given content.
 * @param {String} content
 * @type Void
 */
mozile.gui.display = function(content) {
	if(!mozile.gui.factory) throw("Error [mozile.gui.display]: No GUI factory selected.");
	else return mozile.gui.factory.display(content);
}

/**
 * An abstract GUI factory class.
 * @constructor
 * @param {String} name
 */
mozile.gui.Factory = function(name) {
	/**
	 * A name for this factory.
	 * @type String
	 */
	this.name = name;
}

/**
 * Creates the GUI.
 * @type Void
 */
mozile.gui.Factory.prototype.create = function() {
	alert("Creating GUI.");
}

/**
 * Updates the GUI using the current event.
 * @param {Event} event The current event.
 * @type Boolean
 * @return True if the GUI has been updated, false otherwise.
 */
mozile.gui.Factory.prototype.update = function(event) {
	alert("Updating GUI based on "+ event);
}

/**
 * Displays content
 * @param {String} content
 * @type Void
 */
mozile.gui.Factory.prototype.display = function(content) {
	alert("Displaying content:\n"+ content);
}






/**
 * A GUI factory which uses a HTML and CSS to display a toolbar at the top of the window.
 * NOTE: There are several CSS hacks coded into this file which override the default style sheet "htmlToolbar.css". Be careful.
 */
mozile.gui.htmlToolbar = new mozile.gui.Factory("HTMLToolbarFactory");
// Set as the default factory.
mozile.gui.factory = mozile.gui.htmlToolbar;

/**
 * The toolbar "div" element.
 * @type Element
 */
mozile.gui.htmlToolbar.toolbar = null;

/**
 * A checkmark character.
 * @type String
 */
mozile.gui.htmlToolbar.checkmark = "\u2713";

/**
 * Some CSS values for Internet Explorer to use.
 * This hack is required because of CSS problems with IE.
 * @type String
 */
mozile.gui.htmlToolbar.ieColor = "#C2C2C2"; // Matches PNG matte color.
mozile.gui.htmlToolbar.ieActiveColor = "#C1D2EE";
mozile.gui.htmlToolbar.ieMenuWidth = "170px";
mozile.gui.htmlToolbar.ieBorder = "";
mozile.gui.htmlToolbar.ieActiveBorder = "1px solid #316AC5";
mozile.gui.htmlToolbar.iePadding = "1px";
mozile.gui.htmlToolbar.ieActivePadding = "0px";



/**** Global Commands ****/

/**
 * Display information about Mozile.
 * @type mozile.edit.Command
 */
mozile.gui.htmlToolbar.about = new mozile.edit.Command("About");
mozile.gui.htmlToolbar.about.label = "About Mozile";
mozile.gui.htmlToolbar.about.image = "silk/information";

/**
 * Displays a dialog with information about Mozile.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.gui.htmlToolbar.about.execute = function(state, fresh) {
	alert([
		"About Mozile "+ mozile.version, 
		mozile.homepage, 
		mozile.about, 
		mozile.copyright +" "+ mozile.license, 
		"Contributors: "+ mozile.credits, 
		"Acknowledgements: "+ mozile.acknowledgements
	].join("\n"));
	state.reversible = false;
	state.executed = true;
	return state;
}

/**
 * Display information about Mozile.
 * @type mozile.edit.Command
 */
mozile.gui.htmlToolbar.help = new mozile.edit.Command("Help");
mozile.gui.htmlToolbar.help.image = "silk/help";

/**
 * Displays help information.
 * @param {mozile.edit.State} state The state information needed to execute this command.
 * @param {Boolean} fresh Optional. A value of "true" indicates that the window's selection is already in the correct place and does not need to be moved.
 * @type Object
 */
mozile.gui.htmlToolbar.help.execute = function(state, fresh) {
	window.open(mozile.help, "MozileHelp", "");
	state.reversible = false;
	state.executed = true;
	return state;
}


/**
 * Groups togther commands into the main Mozile menu.
 * @type mozile.edit.Command
 */
mozile.gui.htmlToolbar.mainMenu = new mozile.edit.CommandGroup("Main Menu");
mozile.gui.htmlToolbar.mainMenu.image = "Mozile-16";
mozile.gui.htmlToolbar.mainMenu.addCommand(mozile.gui.htmlToolbar.about);
mozile.gui.htmlToolbar.mainMenu.addCommand(mozile.gui.htmlToolbar.help);

mozile.gui.htmlToolbar._commands = new Array(mozile.gui.htmlToolbar.mainMenu);


/**** Implement Interfaces ****/

/**
 * Create HTML or XHTML elements.
 * When there is no mozile.defaultNS, the element is created in the default namespace. If there is a mozile.defaultNS, the element is created in the XHTML namespace.
 * @param {String} name The name of the element to create.
 * @type Element
 */
mozile.gui.htmlToolbar.createElement = function(name) {
	if(mozile.defaultNS) {
		mozile.require("mozile.xml");
		return mozile.dom.createElementNS(mozile.xml.ns.xhtml, name);
	}
	else return mozile.dom.createElement(name);
}

/**
 * Create an image. THis can be complicated because IE has crappy PNG support.
 * @param {String} name The name of the image file. This will be a partial path.
 * @type Element
 */
mozile.gui.htmlToolbar.createImage = function(name) {
	if(!name || typeof(name) != "string") return null;

	var filetype = ".png";
	if(name.indexOf(".") > -1) filetype = "";
	var img;
	var src = [mozile.root, "images", name + filetype].join(mozile.filesep);
	
	// Hack to work around IE's PNG failings.
	if(false && mozile.browser.isIE && (filetype == ".png" || 
		name.toLowerCase().substring(name.length-3, name.length) == "png") ) {
		img = mozile.gui.htmlToolbar.createElement("span");
		img.style.cssText = "width: 16px; height: 16px; display:inline-block; "
			+ "filter:progid:DXImageTransform.Microsoft.AlphaImageLoader"
			+ "(src=\'" + src + "\', sizingMethod='image');";
    //mozile.debug.debug("mozile.gui.htmlToolbar.createImage", img.style.cssText);
	}
	else {
		img = mozile.gui.htmlToolbar.createElement("img");
		img.setAttribute("src", src);
	}

	return img;
}

/**
 * Set up the toolbar.
 * @type Void
 */
mozile.gui.htmlToolbar.create = function() {
	if(this.toolbar) return;

	// Add style sheet
	var href = [mozile.root, "src", "gui", "htmlToolbar.css"].join(mozile.filesep);
	mozile.dom.addStyleSheet(href, "text/css");

	// Create toolbar.
	this.toolbar = mozile.gui.htmlToolbar.createElement("div");
	this.toolbar.setAttribute("id", "mozileToolbar");
	if(mozile.browser.isIE) this.toolbar.className = "mozileGUI";
	else this.toolbar.setAttribute("class", "mozileGUI");
	var body = mozile.dom.getBody();
	body.appendChild(this.toolbar);
	
	// TODO: Fix this IE CSS hack
	if(mozile.browser.isIE) {
		mozile.dom.setStyle(this.toolbar, "background-color", this.ieColor);
	}

	// Update toolbar.
	this.updateButtons();
	this.reposition();

	// TODO: Messed up in Mozilla
	if(!mozile.browser.isMozilla || !mozile.useDesignMode) {
		mozile.gui.htmlToolbar.interval = window.setInterval("mozile.gui.htmlToolbar.reposition()", mozile.updateInterval);
	}
}


/**
 * Reposition the toolbar.
 * This includes hacks to make Mozile use CSS "position: fixed".
 * @type Void
 */
mozile.gui.htmlToolbar.reposition = function() {
	if(!mozile.gui.htmlToolbar.toolbar) mozile.gui.htmlToolbar.create();

	if(mozile.browser.isMozilla) {
		mozile.dom.setStyle(mozile.gui.htmlToolbar.toolbar, "position", "fixed");
		mozile.dom.setStyle(mozile.gui.htmlToolbar.toolbar, "top", "-1px");
	}
	else {
		var top = document.documentElement.scrollTop;
		if(top) mozile.dom.setStyle(mozile.gui.htmlToolbar.toolbar, "top", top +"px");
	}
	
	if(document.documentElement.clientWidth) {
		var left = (document.documentElement.clientWidth - mozile.gui.htmlToolbar.toolbar.offsetWidth) / 2
		if(left) mozile.dom.setStyle(mozile.gui.htmlToolbar.toolbar, "left", left +"px");
	}
}

/**
 * Update the toolbar, given the information from the event.
 * Only events from new nodes which might change the state of the toolbar are handled.
 * @param {Event} event The current event object.
 * @param {String} change Optional. The type of change made. See mozile.edit.Command.respond for possible values.
 * @type Boolean
 */
mozile.gui.htmlToolbar.update = function(event, change) {
	if(!event) return false;
	if(!change) {
		// Ignore key events, except for keyup.
		if(event.type.indexOf("key") > -1 && event.type != "keyup") return false;
	}
	this.reposition();

	if(!event || !event.node) {
		mozile.gui.htmlToolbar.closeMenus();
		return false;
	}
	if(!change) change = "none";
	
	// Refresh the set of buttons.
	if(change == "node" || event.node != this.lastNode) {
		change = "node";
		if(mozile.dom.isAncestorOf(this.toolbar, event.node)) return false;
		else mozile.gui.htmlToolbar.closeMenus();
		this.lastNode = event.node;
		
		if(!event.rng) event.rng = mozile.edit.lookupRNG(event.node);
		if(!event.rng) return false;
		this.updateButtons(event.rng, event);
	}

	this.updateCommands(event, change);
	this.reposition();
	
	return true;
}

/**
 * Update all the toolbar's buttons.
 * @param {mozile.rng.Element} rng Optional. When provided, commands belonging to the RNG Element will be updated.
 * @param {Event} event Optional. The current event object.
 * @type Void
 */
mozile.gui.htmlToolbar.updateButtons = function(rng, event) {
	mozile.dom.removeChildNodes(this.toolbar);
	this.toolbar.commands = new Array();

	// GUI commands
	for(var i=0; i < mozile.gui.htmlToolbar._commands.length; i++) {
		this.updateButton(mozile.gui.htmlToolbar._commands[i]);
	}	
	this.toolbar.appendChild(document.createTextNode("|"));

	// Global commands
	for(var i=0; i < mozile.edit._commands.length; i++) {
		this.updateButton(mozile.edit._commands[i]);
	}

	// Context commands
	if(rng && rng._commands && rng._commands.length) {
		this.toolbar.appendChild(document.createTextNode("|"));
		for(var i=0; i < rng._commands.length; i++) {
			this.updateButton(rng._commands[i], event);
		}
	}
}

/**
 * Adds a button to the toolbar for the given command. Creates the button if necessary.
 * @param {mozile.edit.Command} command
 * @param {Event} event Optional. The current event object.
 * @type Void
 */
mozile.gui.htmlToolbar.updateButton = function(command, event) {
	if(command.button) {
		this.toolbar.appendChild(command.button.element);
	}
	else if(command.group || command.image) {
		var button = new mozile.gui.htmlToolbar.Button(command);
		this.toolbar.appendChild(button.element);
		this.toolbar.commands.push(command);
	}
	if(command.button) {
		this.toolbar.commands.push(command);
	}
}

/**
 * Update all the buttons on the toolbar.
 * @param {Event} event Optional. The current event object.
 * @param {String} change Optional. The type of change made. See mozile.edit.Command.respond for possible values.
 * @type Void
 */
mozile.gui.htmlToolbar.updateCommands = function(event, change) {
	if(!change || typeof(change) != "string") return;
	if(change == "none") return;
	for(var i=0; i < this.toolbar.commands.length; i++) {
		var command = this.toolbar.commands[i];
		if(command.respond(change)) {
			//mozile.debug.debug("mozile.gui.htmlToolbar.updateCommand", "Updating "+ command.name +" because of change "+ change);
			mozile.gui.htmlToolbar.updateCommand(command, event);
		}
	}
}

/**
 * Update the status of a command's button, if it has one.
 * @param {mozile.edit.Command} command
 * @param {Event} event Optional. The current event object.
 * @type Void
 */
mozile.gui.htmlToolbar.updateCommand = function(command, event) {
	if(command.button) {
		command.button.isAvailable(event);
		command.button.isActive(event);
	}
}

/**
 * Close all the menus belonging to the toolbar.
 * @type Void
 */
mozile.gui.htmlToolbar.closeMenus = function() {
	if(!this.toolbar.commands) return;
	for(var i=0; i < this.toolbar.commands.length; i++) {
		var button = this.toolbar.commands[i].button;
		if(button.menu && button.menu.opened) button.menu.close();
	}
}

/**
 * Displays the content string in a new window.
 * @param {String} content
 * @type Boolean
 */
mozile.gui.htmlToolbar.display = function(content) {
	var win = window.open("", "MozileDisplay", "");
	win.document.write(content);
}





/**
 * Creates a button for a command.
 * @constructor
 * @param {mozile.edit.Command} command
 */
mozile.gui.htmlToolbar.Button = function(command) {
	if(!command) return;
	/**
	 * The command that owns this button.
	 * @type mozile.edit.Command
	 */
	this.command = command;

	/**
	 * The type of this object.
	 * @type String
	 */
	this.type = "Button";

	/**
	 * The button or menu that this command belongs to, if any.
	 * @type mozile.edit.Command
	 */
	this.parent = null;

	/**
	 * The button's element.
	 * @type Element
	 */
	this.element = mozile.gui.htmlToolbar.createElement("span");
	this.element.setAttribute("class", "mozileButton");
	this.element.onclick = function(event) { 
		var c = mozile.edit.getCommand(command.name); 
		if(c) c.button.press(event);
	}

	// Create the button's icon.
	this.image = mozile.gui.htmlToolbar.createImage(command.image);
	this.element.appendChild(this.image);
	
	// Create a pull down menu.
	if(command.group) {
		this.menu = new mozile.gui.htmlToolbar.Menu(command);
		this.menu.parent = this;
		this.element.appendChild(this.menu.element);
		var img = mozile.gui.htmlToolbar.createImage("arrow-down.gif");
		this.element.appendChild(img);
	}

	command.button = this;
}

/**
 * Activate the button.
 * @param {Event} event The event object which caused the button press.
 * @type Void
 */
mozile.gui.htmlToolbar.Button.prototype.press = function(event) {
	mozile.event.normalize(event);
	// Make sure it's the content that's selected and not the toolbar.
	var selection = mozile.dom.selection.get();
	//if(mozile.browser.isIE) selection.restore();
	//else if(this.type =="MenuItem") selection.restore();
	selection.restore();

	var opened = false;
	if(this.menu && this.menu.opened) opened = true;
	if(!this.parent) mozile.gui.htmlToolbar.closeMenus();
	if(this.menu) {
		if(opened) this.menu.close();
		else this.menu.open(event);
	}

	else {
		mozile.gui.htmlToolbar.closeMenus();
		var command = mozile.edit.getCommand(this.command.name);
		if(command) {
			if(!command.test()) {
				mozile.debug.debug("mozile.gui.htmlToolbar.Button.press", "Command '"+ command.name +"' failed its test and will not be executed.");
				return;
			}
			var state = command.prepare(event);
			if(!state) return; // usually means that the prompt has failed.
			state = command.execute(state, false);
			if(state) mozile.edit.done(state);
			else alert("The execution of the command '"+ command.name +"' did not return a state.");
			
			// This is a hack to restore the selection and focus.
			if(mozile.browser.isMozilla) {
				var target = selection.focusNode;
				var newEvent = document.createEvent("MouseEvents");
				newEvent.initMouseEvent("click", true, true, window, 1, 0, 0, mozile.dom.getX(target), mozile.dom.getY(target), false, false, false, false, 1, null);
				target.dispatchEvent(newEvent);
				if(state && state.selection) {
					if(state.selection.after) selection.restore(state.selection.after);
					else selection.restore(state.selection.before);
				}
				mozile.gui.htmlToolbar.closeMenus();
			}
		}
		else alert("No command matching name '"+ this.command.name +"'.");
	}

	mozile.event.cancel(event);
}


/**
 * Get the position of the button.
 * @type Void
 */
mozile.gui.htmlToolbar.Button.prototype.getPosition = function() {
	var position = {
		x: mozile.dom.getX(this.element),
		y: mozile.dom.getY(this.element),
		width: this.element.offsetWidth,
		height: this.element.offsetHeight
	};
	//alert(mozile.util.dumpValues(position));
	return position;
}

/**
 * Determine if the command is available.
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.gui.htmlToolbar.Button.prototype.isAvailable = function(event) {
	var available = this.command.isAvailable(event);
	if(available == this.available) return available;
	this.element.setAttribute("available", available);
	if(mozile.browser.isIE && this.image) {
		if(available) this.image.style.cssText = "";
		else this.image.style.cssText = "filter:alpha(opacity=50)";
	}
	this.available = available;
	return available;
}

/**
 * Determine if the command is active. 
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.gui.htmlToolbar.Button.prototype.isActive = function(event) {
	var active = this.command.isActive(event);
	if(active == this.active) return active;
	this.element.setAttribute("active", active);
	if(mozile.browser.isIE && this.image) {
		if(active) {
			mozile.dom.setStyle(this.image, "border", mozile.gui.htmlToolbar.ieActiveBorder);
			mozile.dom.setStyle(this.image, "padding", mozile.gui.htmlToolbar.ieActivePadding);
		}
		else {
			mozile.dom.setStyle(this.image, "border", mozile.gui.htmlToolbar.ieBorder);
			mozile.dom.setStyle(this.image, "padding", mozile.gui.htmlToolbar.iePadding);
		}
	}
	this.active = active;
	return active;
}




/**
 * Creates a menu item for a command.
 * @constructor
 * @param {mozile.edit.Command} command
 */
mozile.gui.htmlToolbar.MenuItem = function(command) {
	if(!command) return;

	/**
	 * The command that owns this button.
	 * @type mozile.edit.Command
	 */
	this.command = command;

	/**
	 * The type of this object.
	 * @type String
	 */
	this.type = "MenuItem";

	/**
	 * The button or menu that this command belongs to, if any.
	 * @type mozile.edit.Command
	 */
	this.parent = null;

	/**
	 * The button's element.
	 * @type Element
	 */
	this.element = mozile.gui.htmlToolbar.createElement("tr");
	this.element.setAttribute("class", "mozileMenuItem");
	this.element.onclick = function(event) { 
		var c = mozile.edit.getCommand(command.name); 
		if(c) c.menuItem.press(event);
	}
	
	this.cells = {};
	this.cells.active = mozile.gui.htmlToolbar.createElement("td");
	this.cells.icon = mozile.gui.htmlToolbar.createElement("td");
	this.cells.label = mozile.gui.htmlToolbar.createElement("td");
	this.cells.accel = mozile.gui.htmlToolbar.createElement("td");
	if(mozile.browser.isIE) {
		this.cells.active.className = "mozileActive";
		this.cells.icon.className = "mozileIcon";
		this.cells.label.className = "mozileLabel";
		this.cells.accel.className = "mozileAccel";
	}
	else {
		this.cells.active.setAttribute("class", "mozileActive");
		this.cells.icon.setAttribute("class", "mozileIcon");
		this.cells.label.setAttribute("class", "mozileLabel");
		this.cells.accel.setAttribute("class", "mozileAccel");
	}
	this.element.appendChild(this.cells.active);
	this.element.appendChild(this.cells.icon);
	this.element.appendChild(this.cells.label);
	this.element.appendChild(this.cells.accel);

	// Create the button's icon.
	if(command.image) {
		var img = mozile.gui.htmlToolbar.createImage(command.image);
		this.cells.icon.appendChild(img);
	}

	// Create the button's label.
	var name = command.name;
	if(command.label) name = command.label;
	this.cells.label.appendChild(document.createTextNode(name));
	
	// Create a submenu.
	if(command.group) {
		this.menu = new mozile.gui.htmlToolbar.Menu(command);
		this.menu.parent = this;
		this.element.appendChild(this.menu.element);
		img = mozile.gui.htmlToolbar.createImage("menu-arrow.gif");
		this.cells.accel.appendChild(img);
	}
	// Create an accelerator notification.
	else if(command.accel) {
		var accel = mozile.edit.parseAccelerator(command.accel);
		var span = mozile.gui.htmlToolbar.createElement("span");
		span.appendChild(document.createTextNode(accel.abbr));
		this.cells.accel.appendChild(span);
	}
	
	command.menuItem = this;
}
mozile.gui.htmlToolbar.MenuItem.prototype = new mozile.gui.htmlToolbar.Button;
mozile.gui.htmlToolbar.MenuItem.prototype.constructor = mozile.gui.htmlToolbar.MenuItem;


/**
 * Determine if the command is active. 
 * @param {Event} event Optional. The current event.
 * @type Boolean
 */
mozile.gui.htmlToolbar.MenuItem.prototype.isActive = function(event) {
	var active = this.command.isActive(event);
	if(active == this.active) return active;
	mozile.dom.removeChildNodes(this.cells.active);
	if(active) {
		//this.cells.active.appendChild(document.createTextNode(mozile.gui.htmlToolbar.checkmark));
		this.cells.active.appendChild(
			mozile.gui.htmlToolbar.createImage("silk/tick"));
	}
	this.active = active;
	return active;
}



/**
 * Creates a menu for a command group.
 * @constructor
 * @param {mozile.edit.Command} command
 */
mozile.gui.htmlToolbar.Menu = function(command) {
	if(!command) return;

	/**
	 * The command that owns this button.
	 * @type mozile.edit.Command
	 */
	this.command = command;

	/**
	 * The type of this object.
	 * @type String
	 */
	this.type = "Menu";

	/**
	 * The button or menu that this command belongs to, if any.
	 * @type mozile.edit.Command
	 */
	this.parent = null;

	/**
	 * The button's element.
	 * @type Element
	 */
	this.element = mozile.gui.htmlToolbar.createElement("table");
	this.element.setAttribute("class", "mozileMenu");
	this.element.setAttribute("cellspacing", "0px");
	mozile.dom.setStyle(this.element, "display", "none");
	// The tbody element is require for IE.
	var tbody = mozile.gui.htmlToolbar.createElement("tbody");
	this.element.appendChild(tbody);

	// TODO: Fix this IE CSS hack
	if(mozile.browser.isIE) {
		mozile.dom.setStyle(this.element, "background-color", mozile.gui.htmlToolbar.ieColor);
	}
	
	// Attach menu items.
	this.menuItems = new Array();
	for(var i=0; i < command._commands.length; i++) {
		var menuItem = new mozile.gui.htmlToolbar.MenuItem(command._commands[i]);
		if(menuItem) {
			this.menuItems.push(menuItem);
			menuItem.parent = this;
			tbody.appendChild(menuItem.element);
		}
	}

	/**
	 * Indicates that the menu is currently open.
	 * @type Boolean
	 */
	this.opened = false;
	
	command.menu = this;
}
mozile.gui.htmlToolbar.Menu.prototype = new mozile.gui.htmlToolbar.Button;
mozile.gui.htmlToolbar.Menu.prototype.constructor = mozile.gui.htmlToolbar.Menu;


/**
 * Open the menu. First the menu is moved into position, and then the CSS "display" property is set to "block".
 * @param {Event} event Optional. The current event.
 * @type Void
 */
mozile.gui.htmlToolbar.Menu.prototype.open = function(event) {
try{
	//mozile.debug.debug("mozile.gui.htmlToolbar.Menu.open", mozile.xpath.getXPath(this.element));
	this.reposition();
	mozile.dom.setStyle(this.element, "display", "block");
	this.opened = true;
	for(var i=0; i < this.menuItems.length; i++) {
		this.menuItems[i].isAvailable(event);
		this.menuItems[i].isActive(event);
	}
} catch(e) { alert(mozile.dumpError(e)); }
}

/**
 * Close the menu. The CSS "display" property is set to "none", so the menu is hidden. All open child menus are also closed.
 * @type Void
 */
mozile.gui.htmlToolbar.Menu.prototype.close = function() {
	mozile.dom.setStyle(this.element, "display", "none");
	this.opened = false;
	for(var i=0; i < this.menuItems.length; i++) {
		if(this.menuItems[i].menu && this.menuItems[i].menu.opened) {
			this.menuItems[i].menu.close();
		}
	}
}

/**
 * Move the menu into the proper position.
 * @type Void
 */
mozile.gui.htmlToolbar.Menu.prototype.reposition = function() {
	if(!this.parent) return;
	var left = 0;
	var top = 0;
	var position = this.parent.getPosition();

	// TODO: Fix these CSS hacks.
	if(mozile.browser.isIE) {
		mozile.dom.setStyle(this.element, "position", "absolute");
		mozile.dom.setStyle(this.element, "width", mozile.gui.htmlToolbar.ieMenuWidth);
		mozile.dom.setStyle(this.element, "border", mozile.gui.htmlToolbar.ieBorder);
		left = this.parent.element.offsetLeft;
		top = this.parent.element.offsetTop;
	}
	else {
		var width = this.element.clientWidth;
		mozile.dom.setStyle(this.element, "position", "fixed");
		mozile.debug.debug("", position.x +" + "+ width +" = "+ (position.x + width) +" ? "+ document.documentElement.clientWidth);
		if(position.x + width > document.documentElement.clientWidth) {
			left = document.documentElement.clientWidth - width;
		}
		else left = position.x;
		top = position.y;
	}

	if(this.parent.type == "Button") {
		top += position.height;
		// Browser dependant tweaking.
		//if(mozile.browser.isMozilla) top += 1;
	}
	else {
		left += position.width;
		// Browser dependant tweaking.
		//if(mozile.browser.isMozilla) left += 3;
	}

	mozile.dom.setStyle(this.element, "left", left +"px");
	mozile.dom.setStyle(this.element, "top", top +"px");
}

