/*
    SPDX-FileCopyrightText: 2016 Carlos Nihelton <carlosnsoliveira@gmail.com>

    SPDX-License-Identifier: GPL-2.0-or-later
*/

#ifndef CLANGTIDY_REPLACEMENTPARSER_H
#define CLANGTIDY_REPLACEMENTPARSER_H

// KDevPlatform
#include <language/editor/documentrange.h>
// Qt
#include <QFile>
#include <QRegularExpression>
#include <QVector>
// Boost
#include <boost/utility/string_ref.hpp>

using KDevelop::DocumentRange;
using KDevelop::IndexedString;

namespace ClangTidy
{

/**
 * \struct
 * \brief contains basic elements for one replacement in source code.
 *
 */
struct Replacement {
    size_t offset, length; ///< read from YAML.
    QString replacementText; ///< read from YAML.
    DocumentRange range; ///< created from line and column.
};

using Replacements = QVector<Replacement>;

/**
 * Implements the parser for the YAML file generated by clang-tidy with the recommended corrections.
 */
class ReplacementParser
{
public:
    ReplacementParser() = default;
    explicit ReplacementParser(const QString& yaml_file, const QString& source_file);

public:
    void parse();
    size_t count() const { return cReplacements; }
    Replacements allReplacements() { return all_replacements; }

protected:
    /**
    * \function
    * \brief generates the next replacement from the regex capture list.
    * \param smatch the captured match.
    * \return Replacement
    */
    Replacement nextNode(const QRegularExpressionMatch& smatch);

    /**
    * \function
    * \brief compose a range in KTextEditor from the offset and length components of the Replacement being processed.
    * \return KDevelop::DocumentRange
    * \warning the range can be invalid in case offset and length overcome the substring length.
    */
    KDevelop::DocumentRange composeNextNodeRange(size_t offset, size_t length);

private:
    size_t currentLine; ///< current line on source code while parsing.
    size_t currentColumn; ///< current column on source code while parsing.
    size_t currentOffset; ///< current offset in bytes since the beginning of the source code while parsing.
    size_t cReplacements; ///< current count of replacements parsed.

    QString m_yamlname;
    QString m_sourceFile;
    IndexedString i_source;
    QString m_yamlContent;
    std::string m_sourceCode;
    boost::string_ref m_sourceView;
    static const QRegularExpression regex, check;
    Replacements all_replacements;
};

}

#endif // CLANGTIDY_REPLACEMENT_H
