/***************************************************************************
 *   Copyright (C) 2008 by S. MANKOWSKI / G. DE BURE support@mankowski.fr  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>  *
 ***************************************************************************/
/** @file
 * A skrooge plugin to calculate.
 *
 * @author Stephane MANKOWSKI
 */
#include "skgcalculatorpluginwidget.h"

#include <KStandardDirs>
#include <KLocale>

#include <QDomDocument>
#include <QDir>
#include <QKeyEvent>
#include <qmath.h>

#include "skgmainpanel.h"
#include "skgtraces.h"
#include "skgdocumentbank.h"
#include "skgobjectmodel.h"
#include "skginterestobject.h"
#include "skgaccountobject.h"
#include "skgtransactionmng.h"
#include "skgunitobject.h"

SKGCalculatorPluginWidget::SKGCalculatorPluginWidget(SKGDocumentBank* iDocument)
    : SKGTabPage(iDocument)
{
    SKGTRACEIN(1, "SKGCalculatorPluginWidget::SKGCalculatorPluginWidget");
    if (!iDocument) return;

    m_timer.setSingleShot(true);
    connect(&m_timer, SIGNAL(timeout()), this, SLOT(onAmortizationComputation()), Qt::QueuedConnection);

    ui.setupUi(this);
    ui.kUnitEdit->setDocument(iDocument);
    ui.kUnitEdit->setWhereClauseCondition("t_type IN ('1','2','C')");

    ui.kUpdate->setIcon(KIcon("dialog-ok-apply"));
    ui.kAdd->setIcon(KIcon("list-add"));

    ui.kLoadSummaryTitle->setPixmap(KIcon("configure").pixmap(22, 22), KTitleWidget::ImageLeft);
    ui.kPaymentSummaryTitle->setPixmap(KIcon("bookmarks").pixmap(22, 22), KTitleWidget::ImageLeft);
    ui.kAmortizationTableTitle->setPixmap(KIcon("view-pim-calendar").pixmap(22, 22), KTitleWidget::ImageLeft);

    ui.kInterestResultsTitle->setPixmap(KIcon("bookmarks").pixmap(22, 22), KTitleWidget::ImageLeft);
    ui.kInterestValuesTitle->setPixmap(KIcon("configure").pixmap(22, 22), KTitleWidget::ImageLeft);
    ui.kDefineValueTitle->setPixmap(KIcon("configure").pixmap(22, 22), KTitleWidget::ImageLeft);

    ui.kCreditValueDate->addItem(i18nc("A period of 15 days",  "Fifteen"));
    ui.kDebitValueDate->addItem(i18nc("A period of 15 days",  "Fifteen"));
    for (int i = 0 ; i <= 5 ; ++i) {
        ui.kCreditValueDate->addItem(i18nc("Used for configurating when interests are paid on an account : %s days after being calculated",  "Day +%1", i));
        ui.kDebitValueDate->addItem(i18nc("Used for configurating when interests are paid on an account : %s days after being calculated",  "Day -%1", i));
    }

    ui.kMode->addItem(i18nc("24 fifteen is the name of a financial method to compute interests on an account",  "24 fifteen"));
    ui.kMode->addItem(i18nc("A period of 360 days",  "360 days"));
    ui.kMode->addItem(i18nc("A period of 365 days",  "365 days"));

    ui.kAmortizationTable->verticalHeader()->setDefaultSectionSize(ui.kAmortizationTable->verticalHeader()->minimumSectionSize());

    SKGWidgetSelector::SKGListQWidget list;
    list.push_back(ui.kInterestFrm);
    list.push_back(ui.kBtnFrm);
    ui.kWidgetSelector->addButton(KIcon("taxes-finances"), i18n("Interest"), i18n("Interests calculator"), list);
    ui.kWidgetSelector->addButton(KIcon("accessories-calculator"), i18n("Amortization Table"), i18n("Loan amortization table calculator"), ui.kAmortizationFrm);
    connect(ui.kWidgetSelector, SIGNAL(selectedModeChanged(int)), this, SLOT(onBtnModeClicked(int)));

    // Bind account creation view
    m_objectModel = new SKGObjectModel(static_cast<SKGDocumentBank*>(getDocument()), "v_interest", "", this, "", false);
    ui.kInterestView->setModel(m_objectModel);

    SKGObjectModel* resultModel = new SKGObjectModel(static_cast<SKGDocumentBank*>(getDocument()), "interest_result", "", this, "", false);
    ui.kInterestResultTable->setModel(resultModel);

    connect(ui.kInterestView, SIGNAL(selectionChangedDelayed()), this, SLOT(onSelectionChanged()));
    connect(m_objectModel, SIGNAL(beforeReset()), ui.kInterestView, SLOT(saveSelection()));
    connect(m_objectModel, SIGNAL(afterReset()), ui.kInterestView, SLOT(resetSelection()));

    ui.kWidgetSelector->setSelectedMode(0);

    // Refresh
    connect(ui.kDisplayAccountCombo, SIGNAL(currentIndexChanged(int)), this, SLOT(onFilterChanged()), Qt::QueuedConnection);
    connect(ui.KYearEdit, SIGNAL(currentIndexChanged(int)), this, SLOT(onFilterChanged()) , Qt::QueuedConnection);

    connect(getDocument(), SIGNAL(tableModified(QString,int)), this, SLOT(dataModified(QString,int)), Qt::QueuedConnection);
    dataModified("", 0);

    // Set Event filters to catch CTRL+ENTER or SHIFT+ENTER
    this->installEventFilter(this);

    // Synchronize zooms of both tables
    connect(ui.kInterestResultTable, SIGNAL(zoomChanged(int)), ui.kInterestView, SLOT(setZoomPosition(int)));
    connect(ui.kInterestView, SIGNAL(zoomChanged(int)), ui.kInterestResultTable, SLOT(setZoomPosition(int)));
}

SKGCalculatorPluginWidget::~SKGCalculatorPluginWidget()
{
    SKGTRACEIN(1, "SKGCalculatorPluginWidget::~SKGCalculatorPluginWidget");
    m_objectModel = NULL;
}

bool SKGCalculatorPluginWidget::eventFilter(QObject* iObject, QEvent* iEvent)
{
    if (iEvent && iEvent->type() == QEvent::KeyPress) {
        QKeyEvent* keyEvent = static_cast<QKeyEvent*>(iEvent);
        if ((keyEvent->key() == Qt::Key_Return || keyEvent->key() == Qt::Key_Enter) && iObject == this) {
            if (QApplication::keyboardModifiers() &Qt::ControlModifier && ui.kAdd->isEnabled()) {
                ui.kAdd->click();
            } else if (QApplication::keyboardModifiers() &Qt::ShiftModifier && ui.kUpdate->isEnabled()) {
                ui.kUpdate->click();
            }
        }
    }

    return false;
}

QString SKGCalculatorPluginWidget::getState()
{
    SKGTRACEIN(10, "SKGCalculatorPluginWidget::getState");
    QDomDocument doc("SKGML");
    QDomElement root = doc.createElement("parameters");
    doc.appendChild(root);

    root.setAttribute("currentPage", SKGServices::intToString(ui.kWidgetSelector->getSelectedMode()));
    root.setAttribute("account", ui.kDisplayAccountCombo->currentText());
    root.setAttribute("year", ui.KYearEdit->text());

    root.setAttribute("amortizationLoan", ui.kLoanEdit->text());
    root.setAttribute("amortizationUnit", ui.kUnitEdit->text());
    root.setAttribute("amortizationRate", SKGServices::doubleToString(ui.kAnnualRateEdit->value()));
    root.setAttribute("amortizationLenght", SKGServices::intToString(ui.kLenghtEdit->value()));
    root.setAttribute("amortizationInsuranceRate", SKGServices::doubleToString(ui.kInsuranceRateEdit->value()));


    // Memorize table settings
    root.setAttribute("view", ui.kInterestView->getState());
    root.setAttribute("viewResult", ui.kInterestResultTable->getState());

    return doc.toString();
}

void SKGCalculatorPluginWidget::setState(const QString& iState)
{
    SKGTRACEIN(10, "SKGCalculatorPluginWidget::setState");
    QDomDocument doc("SKGML");
    doc.setContent(iState);
    QDomElement root = doc.documentElement();

    QString account = root.attribute("account");
    QString currentPage = root.attribute("currentPage");
    QString year = root.attribute("year");
    QString amortizationLoan = root.attribute("amortizationLoan");
    QString amortizationUnit = root.attribute("amortizationUnit");
    QString amortizationRate = root.attribute("amortizationRate");
    QString amortizationInsuranceRate = root.attribute("amortizationInsuranceRate");
    QString amortizationLenght = root.attribute("amortizationLenght");

    if (!currentPage.isEmpty()) ui.kWidgetSelector->setSelectedMode(SKGServices::stringToInt(currentPage));
    if (!account.isEmpty()) {
        bool previous = ui.kDisplayAccountCombo->blockSignals(true);
        ui.kDisplayAccountCombo->setText(account);
        ui.kDisplayAccountCombo->blockSignals(previous);
    }
    if (!amortizationLoan.isEmpty()) ui.kLoanEdit->setText(amortizationLoan);
    if (!amortizationUnit.isEmpty()) ui.kUnitEdit->setText(amortizationUnit);
    if (!amortizationRate.isEmpty()) ui.kAnnualRateEdit->setValue(SKGServices::stringToDouble(amortizationRate));
    if (!amortizationInsuranceRate.isEmpty()) ui.kInsuranceRateEdit->setValue(SKGServices::stringToDouble(amortizationInsuranceRate));
    if (!amortizationLenght.isEmpty()) ui.kLenghtEdit->setValue(SKGServices::stringToInt(amortizationLenght));

    if (!year.isEmpty()) ui.KYearEdit->setText(year);

    // Update model
    if (m_objectModel) {
        bool previous = m_objectModel->blockRefresh(true);
        onFilterChanged();
        m_objectModel->blockRefresh(previous);
    }

    // !!! Must be done here after onFilterChanged
    ui.kInterestView->setState(root.attribute("view"));
    ui.kInterestResultTable->setState(root.attribute("viewResult"));
}

QString SKGCalculatorPluginWidget::getDefaultStateAttribute()
{
    return "SKGCALCULATOR_DEFAULT_PARAMETERS";
}

void SKGCalculatorPluginWidget::onBtnModeClicked(int mode)
{
    Q_UNUSED(mode);

    // Save zoom position of previous page
    int zoom = zoomPosition();

    // Save zoom position on new page
    setZoomPosition(zoom);

    Q_EMIT selectionChanged();
}

void SKGCalculatorPluginWidget::onSelectionChanged()
{
    SKGTRACEIN(10, "SKGCalculatorPluginWidget::onSelectionChanged");
    // Mapping
    QItemSelectionModel* selModel = ui.kInterestView->selectionModel();
    if (selModel) {
        QModelIndexList indexes = selModel->selectedRows();
        if (indexes.count() && m_objectModel) {
            QModelIndex idx = indexes[indexes.count() - 1];

            SKGInterestObject interest(m_objectModel->getObject(idx));

            ui.kDateEdit->setDate(interest.getDate());
            ui.kRateEdit->setValue(interest.getRate());
            ui.kCreditValueDate->setCurrentIndex(static_cast<int>(interest.getIncomeValueDateMode()));
            ui.kDebitValueDate->setCurrentIndex(static_cast<int>(interest.getExpenditueValueDateMode()));
            ui.kMode->setCurrentIndex(static_cast<int>(interest.getInterestComputationMode()));
        }

        Q_EMIT selectionChanged();
    }
}

void SKGCalculatorPluginWidget::dataModified(const QString& iTableName, int iIdTransaction)

{
    SKGTRACEIN(10, "SKGCalculatorPluginWidget::dataModified");
    Q_UNUSED(iIdTransaction);
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    // Refresh widgets
    QSqlDatabase* db = getDocument()->getDatabase();
    setEnabled(db != NULL);
    if (db != NULL && (iTableName == "interest" || iTableName.isEmpty())) {
        // Correction bug 2299394 vvv
        if (ui.kInterestView->isAutoResized()) ui.kInterestView->resizeColumnsToContentsDelayed();
        // Correction bug 2299394 ^^^

        // Correction 215658: vvv because the table is modified, the selection is modified
        onSelectionChanged();
        // Correction 215658: ^^^
    }

    // Refresh interest parameter view
    QString current = ui.kDisplayAccountCombo->currentText();
    if (db != NULL && (iTableName == "v_account_display" || iTableName.isEmpty())) {
        // Clear
        ui.kDisplayAccountCombo->clear();

        SKGStringListList listAccount;
        getDocument()->executeSelectSqliteOrder(QString("SELECT t_ICON, t_name from v_account_display where t_close='N' order by t_name"), listAccount);

        int nbAccounts = listAccount.count();
        if (nbAccounts >= 0) {
            QDir dirLogo(KStandardDirs::locate("data", QString::fromLatin1("skrooge/images/logo/")));
            for (int i = 1; i < nbAccounts; ++i) {  // Ignore header
                QIcon icon(dirLogo.absoluteFilePath(listAccount.at(i).at(0)));
                QString text = listAccount.at(i).at(1);
                ui.kDisplayAccountCombo->addItem(icon, text);
            }
        }

        int posText = ui.kDisplayAccountCombo->findText(current);
        if (posText == -1) posText = 0;
        ui.kDisplayAccountCombo->setCurrentIndex(posText);

        ui.kAdd->setEnabled(ui.kDisplayAccountCombo->count() > 0);
    }

    // Fill years
    if (db != NULL && (iTableName == "v_operation_display" || iTableName.isEmpty())) {
        disconnect(ui.KYearEdit, SIGNAL(currentIndexChanged(int)), this, SLOT(onFilterChanged()));

        int lastYear = QDate::currentDate().year();
        int firstYear = lastYear;

        QStringList list;
        getDocument()->getDistinctValues("v_operation_display", "d_DATEYEAR", "", list);
        if (list.count()) firstYear = SKGServices::stringToInt(list.at(0));

        QString year = ui.KYearEdit->text();
        ui.KYearEdit->clear();
        for (int i = lastYear; i >= firstYear; --i) {
            QString v = SKGServices::intToString(i);
            if (year.isEmpty()) year = v;
            ui.KYearEdit->addItem(v);
        }

        ui.KYearEdit->setText(year);
        connect(ui.KYearEdit, SIGNAL(currentIndexChanged(int)), this, SLOT(onFilterChanged()));
    }
    // Refresh computation
    if (db != NULL && (iTableName == "interest" || iTableName == "account" || iTableName.isEmpty())) {
        computeInterest();
    }
    QApplication::restoreOverrideCursor();
}

void SKGCalculatorPluginWidget::computeInterest()
{
    // Compute
    SKGAccountObject account(getDocument());
    SKGError err = account.setName(ui.kDisplayAccountCombo->currentText());
    IFOKDO(err, account.load())
    QList<SKGAccountObject::SKGInterestItem> oInterestList;
    double oInterests = 0;
    IFOKDO(err, account.getInterestItems(oInterestList, oInterests, SKGServices::stringToInt(ui.KYearEdit->text())))
    IFOK(err) {
        // Refresh table
        ui.kInterestResultTable->setState(ui.kInterestResultTable->getState());

        // Set text
        SKGDocumentBank* doc = static_cast<SKGDocumentBank*>(getDocument());
        if (doc) {
            SKGServices::SKGUnitInfo unit1 = doc->getPrimaryUnit();
            SKGServices::SKGUnitInfo unit2 = doc->getSecondaryUnit();

            QString s1 = doc->formatMoney(oInterests, unit1);
            ui.kInterestLbl->setText(i18nc("The Annual interest is the amount of money gained in one year on a remunerated account",  "Annual interest=%1", s1));
            if (!unit2.Symbol.isEmpty() && unit2.Value) {
                s1 = doc->formatMoney(oInterests, unit2);
                ui.kInterestLbl->setToolTip(i18nc("The Annual interest is the amount of money gained in one year on a remunerated account",  "Annual interest=%1", s1));
            }
        }
    }
}

void SKGCalculatorPluginWidget::onFilterChanged()
{
    SKGTRACEIN(1, "SKGCalculatorPluginWidget::onFilterChanged");
    if (!isEnabled()) return;
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    // Compute where clause
    QString account = ui.kDisplayAccountCombo->currentText();
    QString filter2 = "t_ACCOUNT='" % SKGServices::stringToSqlString(account) % "' ORDER BY d_date";

    // Update model
    if (m_objectModel) {
        if (m_objectModel->setFilter(filter2)) ui.kInterestView->setState(ui.kInterestView->getState());

        // Compute result
        computeInterest();
    }
    QApplication::restoreOverrideCursor();
}

SKGObjectBase::SKGListSKGObjectBase SKGCalculatorPluginWidget::getSelectedObjects()
{
    SKGObjectBase::SKGListSKGObjectBase output;
    if (ui.kWidgetSelector->getSelectedMode() == 0) output = ui.kInterestView->getSelectedObjects();
    return output;
}

int SKGCalculatorPluginWidget::getNbSelectedObjects()
{
    int output = 0;
    if (ui.kWidgetSelector->getSelectedMode() == 0) output = ui.kInterestView->getNbSelectedObjects();
    return output;
}

// void SKGCalculatorPluginWidget::onCalculatorReturnPressed(const QString& iText)
// {
// //         ui.kCalculatorList->addItem(iText);
// }

// /*void SKGCalculatorPluginWidget::onCalculatorListClicked(const QString& iText)
// {
// //         ui.kCalculatorEdit->setText(iText);
// //         ui.kCalculatorEdit->setFocus();
// }*/

void SKGCalculatorPluginWidget::onSelectedInterestChanged()
{
    int nbSel = getNbSelectedObjects();
    ui.kDateEdit->setEnabled(nbSel <= 1);
    ui.kRateEdit->setEnabled(nbSel <= 1);
    ui.kAdd->setEnabled(nbSel <= 1);
}

void SKGCalculatorPluginWidget::onAdd()
{
    SKGError err;
    SKGTRACEINRC(10, "SKGCalculatorPluginWidget::onAdd", err);
    {
        SKGInterestObject interestObj;
        {
            QString accountname = ui.kDisplayAccountCombo->currentText();
            SKGBEGINTRANSACTION(*getDocument(), i18nc("Lets the user create parameters for computing interests on an account", "Interest parameter creation for account '%1'", accountname), err);

            // Get Parent account
            SKGAccountObject accountObj(getDocument());
            IFOKDO(err, accountObj.setName(accountname))
            IFOKDO(err, accountObj.load())

            // Create interest object
            IFOKDO(err, accountObj.addInterest(interestObj))
            IFOKDO(err, interestObj.setDate(ui.kDateEdit->date()))
            IFOKDO(err, interestObj.setRate(ui.kRateEdit->value()))
            IFOKDO(err, interestObj.setIncomeValueDateMode(static_cast<SKGInterestObject::ValueDateMode>(ui.kCreditValueDate->currentIndex())))
            IFOKDO(err, interestObj.setExpenditueValueDateMode(static_cast<SKGInterestObject::ValueDateMode>(ui.kDebitValueDate->currentIndex())))
            IFOKDO(err, interestObj.setInterestComputationMode(static_cast<SKGInterestObject::InterestMode>(ui.kMode->currentIndex())))
            IFOKDO(err, interestObj.save())
        }

        // status bar
        IFOK(err) {
            err = SKGError(0, i18nc("User defined parameters for computing interests were successfully created", "Interest parameter created"));
            ui.kInterestView->selectObject(interestObj.getUniqueID());
        } else {
            err.addError(ERR_FAIL, i18nc("Error message: User defined parameters for computing interests could not be created",  "Interest parameter creation failed"));
        }
    }

    // Display error
    SKGMainPanel::displayErrorMessage(err);
}

void SKGCalculatorPluginWidget::onUpdate()
{
    SKGError err;
    SKGTRACEINRC(10, "SKGCalculatorPluginWidget::onUpdate", err);

    // Get Parent account
    QString accountname = ui.kDisplayAccountCombo->currentText();
    SKGAccountObject accountObj(getDocument());
    IFOKDO(err, accountObj.setName(accountname))
    IFOKDO(err, accountObj.load())

    SKGObjectBase::SKGListSKGObjectBase updatedInterest = getSelectedObjects();
    int nb = updatedInterest.count();
    {
        SKGInterestObject interestObj;
        SKGBEGINTRANSACTION(*getDocument(), i18nc("Lets the user update parameters for computing interests on an account", "Interest parameter update for account '%1'", accountname), err);

        for (int i = 0; !err && i < nb; ++i) {
            interestObj = updatedInterest.at(i);
            if (nb == 1) {
                IFOKDO(err, interestObj.setDate(ui.kDateEdit->date()))
                IFOKDO(err, interestObj.setRate(ui.kRateEdit->value()))
            }
            IFOKDO(err, interestObj.setIncomeValueDateMode(static_cast<SKGInterestObject::ValueDateMode>(ui.kCreditValueDate->currentIndex())))
            IFOKDO(err, interestObj.setExpenditueValueDateMode(static_cast<SKGInterestObject::ValueDateMode>(ui.kDebitValueDate->currentIndex())))
            IFOKDO(err, interestObj.setInterestComputationMode(static_cast<SKGInterestObject::InterestMode>(ui.kMode->currentIndex())))
            IFOKDO(err, interestObj.save())
        }
    }

    // status bar
    IFOKDO(err, SKGError(0, i18nc("User defined parameters for computing interests were successfully updated", "Interest parameter updated")))
    else {
        err.addError(ERR_FAIL, i18nc("Error message: User defined parameters for computing interests could not be updated",  "Interest parameter update failed"));
    }

    // Display error
    SKGMainPanel::displayErrorMessage(err);
}

void SKGCalculatorPluginWidget::onAmortizationComputationDelayed()
{
    m_timer.start(300);
}

void SKGCalculatorPluginWidget::onAmortizationComputation()
{
    SKGTRACEIN(10, "SKGCalculatorPluginWidget::onAmortizationComputation");
    QApplication::setOverrideCursor(QCursor(Qt::WaitCursor));

    KLocale* loc = KGlobal::locale();
    if (loc) {
        // Get unit
        SKGServices::SKGUnitInfo unitInfo;
        unitInfo.Symbol = ui.kUnitEdit->text();

        SKGUnitObject unit(getDocument());
        unit.setSymbol(unitInfo.Symbol);
        unit.load();
        unitInfo.NbDecimal = unit.getNumberDecimal();

        int p = qPow(10, unitInfo.NbDecimal);

        // Computation
        int periodInMonths = 12;
        int numberPayments = ui.kLenghtEdit->value() * periodInMonths;
        double periodicRate = ui.kAnnualRateEdit->value() / 100 / periodInMonths;
        double insurance = qRound(ui.kLoanEdit->value() * ui.kInsuranceRateEdit->value() / 100 / periodInMonths * p) / p;
        double periodicAmount = qRound(ui.kLoanEdit->value() * periodicRate / (1 - pow(1 + periodicRate, -numberPayments)) * p) / p;

        // Fill table
        double dueAmount = ui.kLoanEdit->value();
        double sumInterest = 0;
        ui.kAmortizationTable->setRowCount(numberPayments);
        for (int i = 0; i < numberPayments; ++i) {
            // Compute
            double roundedInterest = qRound(dueAmount * periodicRate * p) / p;
            double principal = qRound((i == numberPayments - 1 ? dueAmount : periodicAmount - roundedInterest) * p) / p;
            double amount = roundedInterest + principal;
            dueAmount -= principal;
            sumInterest += roundedInterest;

            // Add items
            ui.kAmortizationTable->setItem(i, 0, new QTableWidgetItem(loc->formatMoney(amount, unitInfo.Symbol, unitInfo.NbDecimal)));
            ui.kAmortizationTable->setItem(i, 1, new QTableWidgetItem(loc->formatMoney(principal, unitInfo.Symbol, unitInfo.NbDecimal)));
            ui.kAmortizationTable->setItem(i, 2, new QTableWidgetItem(loc->formatMoney(roundedInterest, unitInfo.Symbol, unitInfo.NbDecimal)));
            ui.kAmortizationTable->setItem(i, 3, new QTableWidgetItem(loc->formatMoney(insurance, unitInfo.Symbol, unitInfo.NbDecimal)));
            ui.kAmortizationTable->setItem(i, 4, new QTableWidgetItem(loc->formatMoney(dueAmount, unitInfo.Symbol, unitInfo.NbDecimal)));
        }

        ui.kAmortizationResult->setText(i18n("Number of payments:	%1\n"  // NOLINT(whitespace/tab)
                                             "Monthly payment: 	%2\n"  // NOLINT(whitespace/tab)
                                             "Monthly insurance: 	%3\n"  // NOLINT(whitespace/tab)
                                             "Total principal paid: 	%4\n"  // NOLINT(whitespace/tab)
                                             "Total interest paid: 	%5\n"  // NOLINT(whitespace/tab)
                                             "Total insurance paid: 	%6\n"  // NOLINT(whitespace/tab)
                                             "Total paid: 		%7" ,  // NOLINT(whitespace/tab)
                                             SKGServices::intToString(numberPayments),
                                             loc->formatMoney(periodicAmount, unitInfo.Symbol, unitInfo.NbDecimal),
                                             loc->formatMoney(insurance, unitInfo.Symbol, unitInfo.NbDecimal),
                                             loc->formatMoney(ui.kLoanEdit->value(), unitInfo.Symbol, unitInfo.NbDecimal),
                                             loc->formatMoney(sumInterest, unitInfo.Symbol, unitInfo.NbDecimal),
                                             loc->formatMoney(numberPayments * insurance, unitInfo.Symbol, unitInfo.NbDecimal),
                                             loc->formatMoney(ui.kLoanEdit->value() + sumInterest + numberPayments * insurance, unitInfo.Symbol, unitInfo.NbDecimal)));
    }

    QApplication::restoreOverrideCursor();
}

QWidget* SKGCalculatorPluginWidget::zoomableWidget()
{
    if (ui.kWidgetSelector->getSelectedMode() == 0) return ui.kInterestView;
    else {
        return ui.kAmortizationTable;
    }
}

#include "skgcalculatorpluginwidget.moc"


