/***************************************************************************
                          threeditem.h  -  description
                             -------------------
    begin                : Tue Dec 25 2001
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef THREEDITEM_H
#define THREEDITEM_H

#include <kurl.h>
#include "frameitem.h"

class QLibrary;

/**
  * 3D item class. Base class for 3D array items and 3d function items.
  * @author Werner Stille
  */
class ThreeDItem : public FrameItem
{
public:
  /** Constructor */
  ThreeDItem();
  /** Copy constructor */
  ThreeDItem(const ThreeDItem& a);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  ThreeDItem(KplNamespace::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    */
  ThreeDItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param act true for visible item.
    * @param lgx true for logarithmic x axis.
    * @param lgy true for logarithmic y axis.
    * @param lgz true for logarithmic z axis.
    * @param frame true to show frame.
    * @param ndx number of digits for x axis.
    * @param ndy number of digits for y axis.
    * @param ndz number of digits for z axis.
    * @param mtx number of intervals between major tics of x axis.
    * @param mty number of intervals between major tics of y axis.
    * @param mtz number of intervals between major tics of z axis.
    * @param gm 0 for no tics, 1 for axes with tics,
    * 2 for axes with tics and labels.
    * @param gm3 0 for lines parallel to x, 1 for lines parallel to z,
    * 2 for x,z grid.
    * @param ex logarithm of normalization factor for x axis.
    * @param ey logarithm of normalization factor for y axis.
    * @param ez logarithm of normalization factor for z axis.
    * @param col_f color for frame and labels.
    * @param col_g color for grids and tics.
    * @param col_d color for data.
    * @param x_0 left page margin in centimeters.
    * @param wx width in centimeters.
    * @param y_0 bottom page margin in centimeters.
    * @param hy height in centimeters.
    * @param xmin minimum x value.
    * @param xmax maximum x value.
    * @param ymin minimum y value.
    * @param ymax maximum y value.
    * @param zmin minimum z value.
    * @param zmax maximum z value.
    * @param d_x x stepsize.
    * @param d_z z stepsize.
    * @param xt distance between major tics of x axis.
    * @param yt distance between major tics of y axis.
    * @param zt distance between major tics of z axis.
    * @param relSize relative size of text, markers and lines.
    * @param phi_ angle of rotation about y axis in deg.
    * @param theta_ tilt angle of y axis in deg.
    * @param b_x x box width in cm.
    * @param b_y y box width in cm.
    * @param b_z z box width in cm.
    * @param xr x position of center of box bottom plane.
    * @param yr y position of center of box bottom plane.
    * @param f_x normalization factor for x values.
    * @param f_y normalization factor for y values.
    * @param f_z normalization factor for z values.
    * @param sX x axis text.
    * @param sY y axis text.
    * @param sZ z axis text.
    * @param sH heading text.
    * @param aut pointer to structure containing autoplot settings.
    */
  ThreeDItem(bool act, bool lgx, bool lgy, bool lgz, bool frame, int ndx,
             int ndy, int ndz, int mtx, int mty, int mtz, int gm, int gm3,
             int ex, int ey, int ez, const QString& col_f,
             const QString& col_g, const QString& col_d, double x_0, double wx,
             double y_0, double hy, double xmin, double xmax, double ymin,
             double ymax, double zmin, double zmax, double d_x, double d_z,
             double xt, double yt, double zt, double relSize, double phi_,
             double theta_, double b_x, double b_y, double b_z, double xr,
             double yr, double f_x, double f_y, double f_z, const QString& sX,
             const QString& sY, const QString& sZ, const QString& sH,
             KplNamespace::AutoStruct* aut);
  /** Destructor */
  ~ThreeDItem();
  /** Assignment operator */
  ThreeDItem& operator=(const ThreeDItem& a);
  virtual void draw(KplGraph* g);
#ifndef KPL_CLASSES_ONLY
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  virtual void exportTable(QTextStream& ts, KplDoc* m) const;
#endif

protected:
  /**
    * Calculates transformation matrix.
    * @param phi angle of rotation about y axis in deg.
    * @param theta tilt angle of y axis in deg.
    */
  void rot3d(double phi, double theta);
  /** Initializes 3D structure. */
  void init3d();
  /**
    * Calculates 2D x value.
    * @param x 3D x value.
    * @param z 3D z value.
    */
  double xFrom3d(double x, double z);
  /**
    * Calculates 2D y value.
    * @param x 3D x value.
    * @param y 3D y value.
    * @param z 3D z value.
    */
  double yFrom3d(double x, double y, double z);
  /**
    * Draws line in 3D space.
    * @param g pointer to KplGraph object.
    * @param x1 3D x value of point 1.
    * @param y1 3D y value of point 1.
    * @param z1 3D z value of point 1.
    * @param x2 3D x value of point 2.
    * @param y2 3D y value of point 2.
    * @param z2 3D z value of point 2.
    */
  void line3d(KplGraph* g, double x1, double y1, double z1,
              double x2, double y2, double z2);
  /**
    * Plots curve in 3D space.
    * @param g pointer to KplGraph object.
    * @param xArr true of array in xData.
    * @param yArr true of array in yData.
    * @param zArr true of array in zData.
    * @param xData array or start value of x data.
    * @param yData array or start value of y data.
    * @param zData array or start value of z data.
    * @param xScale factor for x array step width x data.
    * @param yScale factor for y array step width y data.
    * @param zScale factor for z array step width z data.
    * @param firstLine true for first line of 3D surface.
    * @param nPnts number of points of the curve.
    * @param xOff 2D x offset for 3D y axis mimimum.
    * @param yOff 2D y offset for 3D y axis mimimum.
    * @param vtx array to store vertices.
    */
  void plot3d(KplGraph* g, bool xArr, bool yArr, bool zArr,
              const double* xData, const double* yData,
              const double* zData, double xScale, double yScale,
              double zScale, bool firstLine, int nPnts,
              double xOff, double yOff, double* vtx);
  /**
    * Plots 3D frame.
    * @param g pointer to KplGraph object.
    * @param ihcor index of vertex furthest in the background.
    */
  void frame3d(KplGraph* g, int ihcor);
  /**
    * Plots 3D axis.
    * @param g pointer to KplGraph object.
    * @param rmi minimum axis value.
    * @param rma maximum axis value.
    * @param tic distance between major tics.
    * @param mtic number of intervals between major tics.
    * @param f normalization factor.
    * @param scal scaling factor.
    * @param xa 3D x position of axis start point.
    * @param za 3D z position of axis start point.
    * @param da2 tic length.
    * @param ndig number of digits.
    * @param ia 0 for x axis 1 for y axis 2 for z axis.
    * @param log true for logarithmic axis.
    */
  void axis3d(KplGraph* g, double rmi, double rma, double tic, int mtic,
              double f, double scal, double xa, double za, double da2,
              int ndig, int ia, bool log);
  /**
    * Draws frame and axes of 3D plot.
    * @param g pointer to KplGraph object.
    */
  void rast3d(KplGraph* g);
  /**
    * Plots 3D data.
    * @param g pointer to KplGraph object.
    */
  void plar3d(KplGraph* g);
  bool showFrame, logz;
  mutable bool logxo, logzo;
  int ndigz, mticz, iez, mode3D;
  mutable int nx, nz;
  unsigned colData;
  double zmi, zma, dx, dz, ztic, phi, theta, boxx, boxy, boxz, scalx, scaly,
         scalz, xref, yref, fx, fy, fz;
  mutable double xStep, zStep, xmio, xmao, dxo, zmio, zmao, dzo;
  mutable double** yv;
  double vertex[16];
  QString sz;
  struct P3d{
    P3d(int n3d = 2000);
    ~P3d();
    int nRes;
    double scal, length, height, phi, theta, a11, a13, a21, a22, a23,
           xOff, yOff;
    double *yMin, *yMax;
  } p3d;

  friend class ThreeDDlg;
};

class Fun3DItem : public ThreeDItem
{
public:
  /** Constructor */
  Fun3DItem();
  /** Copy constructor */
  Fun3DItem(const Fun3DItem& f);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  Fun3DItem(KplNamespace::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    * @param url URL of plot file.
    */
  Fun3DItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut,
            const KURL& url);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param act true for visible item.
    * @param lgx true for logarithmic x axis.
    * @param lgy true for logarithmic y axis.
    * @param lgz true for logarithmic z axis.
    * @param frame true to show frame.
    * @param ndx number of digits for x axis.
    * @param ndy number of digits for y axis.
    * @param ndz number of digits for z axis.
    * @param mtx number of intervals between major tics of x axis.
    * @param mty number of intervals between major tics of y axis.
    * @param mtz number of intervals between major tics of z axis.
    * @param gm 0 for no tics, 1 for axes with tics,
    * 2 for axes with tics and labels.
    * @param gm3 0 for lines parallel to x, 1 for lines parallel to z,
    * 2 for x,z grid.
    * @param ex logarithm of normalization factor for x axis.
    * @param ey logarithm of normalization factor for y axis.
    * @param ez logarithm of normalization factor for z axis.
    * @param col_f color for frame and labels.
    * @param col_g color for grids and tics.
    * @param col_d color for data.
    * @param x_0 left page margin in centimeters.
    * @param wx width in centimeters.
    * @param y_0 bottom page margin in centimeters.
    * @param hy height in centimeters.
    * @param xmin minimum x value.
    * @param xmax maximum x value.
    * @param ymin minimum y value.
    * @param ymax maximum y value.
    * @param zmin minimum z value.
    * @param zmax maximum z value.
    * @param d_x x stepsize.
    * @param d_z z stepsize.
    * @param xt distance between major tics of x axis.
    * @param yt distance between major tics of y axis.
    * @param zt distance between major tics of z axis.
    * @param relSize relative size of text, markers and lines.
    * @param phi_ angle of rotation about y axis in deg.
    * @param theta_ tilt angle of y axis in deg.
    * @param b_x x box width in cm.
    * @param b_y y box width in cm.
    * @param b_z z box width in cm.
    * @param xr x position of center of box bottom plane.
    * @param yr y position of center of box bottom plane.
    * @param f_x normalization factor for x values.
    * @param f_y normalization factor for y values.
    * @param f_z normalization factor for z values.
    * @param sX x axis text.
    * @param sY y axis text.
    * @param sZ z axis text.
    * @param sH heading text.
    * @param name function name.
    * @param url URL of the function library.
    * @param aut pointer to structure containing autoplot settings.
    */
  Fun3DItem(bool act, bool lgx, bool lgy, bool lgz, bool frame, int ndx,
            int ndy, int ndz, int mtx, int mty, int mtz, int gm, int gm3,
            int ex, int ey, int ez, const QString& col_f,
            const QString& col_g, const QString& col_d, double x_0,
            double wx, double y_0, double hy, double xmin,
            double xmax, double ymin, double ymax, double zmin, double zmax,
            double d_x, double d_z, double xt, double yt, double zt,
            double relSize, double phi_, double theta_, double b_x, double b_y,
            double b_z, double xr, double yr, double f_x, double f_y,
            double f_z, const QString& sX, const QString& sY,
            const QString& sZ, const QString& sH, const QString& name,
            const KURL& url, KplNamespace::AutoStruct* aut);
  /** Destructor */
  ~Fun3DItem();
  /** Assignment operator */
  Fun3DItem& operator=(const Fun3DItem& f);
  virtual ItemTypes iType() const;
  virtual void draw(KplGraph* g);
#ifndef KPL_CLASSES_ONLY
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  virtual void setText(QListViewItem* it, bool* arrays, bool* funcs) const;
  virtual int editItem(QWidget* parent, KplDoc* m, int i);
  virtual void exportTable(QTextStream& ts, KplDoc* m) const;
#endif
  virtual KplItem* copy() const;
  virtual void setPar(int i, double value, bool yFun);
  /**
   * Gets module handle and address of a shared library function for 3D plots.
   * @param path path of the library.
   * @param name function name.
   * @param lib pointer to library.
   * @param fkt pointer to function address.
   * @return true for success.
   */
  static bool getFunc3DAddr(const QString& path, const QString& name,
                            QLibrary** lib,
                            double (**fkt)(double, double, const double*));

protected:
  /**
   * Initializes several elements.
   */
  void init();
  /**
   * Calculates function value table.
   */
  int calcTable() const;
  double (*fkty)(double, double, const double*);
  mutable double (*fktyo)(double, double, const double*);
  double py[20];
  mutable double pyo[20];
  QString namey;
  KURL pathy;
  QLibrary* liby;

  friend class Fun3DDlg;
  friend class Kpl;
};

class Array3DItem : public ThreeDItem
{
public:
  /** Constructor */
  Array3DItem();
  /** Copy constructor */
  Array3DItem(const Array3DItem& a);
  /**
    * Constructor. Initializes corresponding to autoplot settings.
    * @param aut pointer to structure containing autoplot settings.
    */
  Array3DItem(KplNamespace::AutoStruct* aut);
  /**
    * Constructor. Initializes corresponding to plot file.
    * @param plo pointer to KSimpleConfig object containing item data.
    * @param aut pointer to structure containing autoplot settings.
    */
  Array3DItem(KSimpleConfig* plo, KplNamespace::AutoStruct* aut,
              const KURL& uPlo);
  /**
    * Constructor. Initializes corresponding to arguments.
    * @param act true for visible item.
    * @param lgx true for logarithmic x axis.
    * @param lgy true for logarithmic y axis.
    * @param lgz true for logarithmic z axis.
    * @param frame true to show frame.
    * @param ndx number of digits for x axis.
    * @param ndy number of digits for y axis.
    * @param ndz number of digits for z axis.
    * @param mtx number of intervals between major tics of x axis.
    * @param mty number of intervals between major tics of y axis.
    * @param mtz number of intervals between major tics of z axis.
    * @param gm 0 for no tics, 1 for axes with tics,
    * 2 for axes with tics and labels.
    * @param gm3 0 for lines parallel to x, 1 for lines parallel to z,
    * 2 for x,z grid.
    * @param ex logarithm of normalization factor for x axis.
    * @param ey logarithm of normalization factor for y axis.
    * @param ez logarithm of normalization factor for z axis.
    * @param col_f color for frame and labels.
    * @param col_g color for grids and tics.
    * @param col_d color for data.
    * @param x_0 left page margin in centimeters.
    * @param wx width in centimeters.
    * @param y_0 bottom page margin in centimeters.
    * @param hy height in centimeters.
    * @param xmin minimum x value.
    * @param xmax maximum x value.
    * @param ymin minimum y value.
    * @param ymax maximum y value.
    * @param zmin minimum z value.
    * @param zmax maximum z value.
    * @param d_x x stepsize.
    * @param d_z z stepsize.
    * @param xt distance between major tics of x axis.
    * @param yt distance between major tics of y axis.
    * @param zt distance between major tics of z axis.
    * @param relSize relative size of text, markers and lines.
    * @param phi_ angle of rotation about y axis in deg.
    * @param theta_ tilt angle of y axis in deg.
    * @param b_x x box width in cm.
    * @param b_y y box width in cm.
    * @param b_z z box width in cm.
    * @param xr x position of center of box bottom plane.
    * @param yr y position of center of box bottom plane.
    * @param f_x normalization factor for x values.
    * @param f_y normalization factor for y values.
    * @param f_z normalization factor for z values.
    * @param sX x axis text.
    * @param sY y axis text.
    * @param sZ z axis text.
    * @param sH heading text.
    * @param ix x column index.
    * @param iy y column index.
    * @param iz z column index.
    * @param ie y error column index.
    * @param istart start row index.
    * @param np number of data points.
    * @param errb true to plot error bars.
    * @param _smf smoothing factor.
    * @param u URL of data file.
    * @param aut pointer to structure containing autoplot settings.
    * @param internal true for internal stored array data.
    * @param idec 0 for decimal point, 1 for comma.
    */
  Array3DItem(bool act, bool lgx, bool lgy, bool lgz, bool frame, int ndx,
              int ndy, int ndz, int mtx, int mty, int mtz, int gm, int gm3,
              int ex, int ey, int ez, const QString& col_f,
              const QString& col_g, const QString& col_d, double x_0,
              double wx, double y_0, double hy, double xmin,
              double xmax, double ymin, double ymax, double zmin, double zmax,
              double d_x, double d_z, double xt, double yt, double zt,
              double relSize, double phi_, double theta_, double b_x,
              double b_y, double b_z, double xr, double yr, double f_x,
              double f_y, double f_z, const QString& sX, const QString& sY,
              const QString& sZ, const QString& sH, int i_x, int i_y, int i_z,
              int i_e, int i_start, int np, int errb, double _smf,
              const KURL& u, KplNamespace::AutoStruct* aut, bool _int = false,
              int idec = 0);
  /** Destructor */
  ~Array3DItem();
  /** Assignment operator */
  Array3DItem& operator=(const Array3DItem& a);
  virtual ItemTypes iType() const;
  virtual void draw(KplGraph* g);
#ifndef KPL_CLASSES_ONLY
  virtual void writePlo(KSimpleConfig* plo, const KURL& url, bool abs,
                        KplDoc* m) const;
  virtual void setText(QListViewItem* it, bool* arrays, bool* funcs) const;
  virtual int editItem(QWidget* parent, KplDoc* m, int i);
  virtual void exportTable(QTextStream& ts, KplDoc* m) const;
#endif
  virtual KplItem* copy() const;

protected:
  /**
   * Calculates value table.
   */
  void calcTable() const;
  bool internal;
  int ix, iy, iz, ie, istart, n, errbars, nrows, ncols, idec;
  mutable int ixo, iyo, izo, ieo, istarto, no, errbarso;
  double smf;
  mutable double smfo;
  double** xv;
  KURL url;

  friend class Array3DDlg;
  friend class Kpl;
};

#endif
