/***************************************************************************
                          kgraph.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <math.h>
#include <qbitmap.h>
#include <qfontdatabase.h>
#include <qimage.h>
#include <qpaintdevicemetrics.h>
#include <kio/netaccess.h>
#include <kmessagebox.h>
#include <kurl.h>
#include "kgraph.h"
#include "utils.h"

//#define NO_CLIPPING

const char* KGraph::face[] = {"Helvetica", "Symbol", "Dingbats"};

const int KGraph::pattern[] = {Qt::NoBrush, Qt::SolidPattern, Qt::HorPattern,
                               Qt::VerPattern, Qt::CrossPattern,
                               Qt::BDiagPattern, Qt::FDiagPattern,
                               Qt::DiagCrossPattern};

KGraph::KGraph() : minx(0), maxx(0), miny(0), maxy(0), kxcur(0), kycur(0),
 ndigx(0), ndigy(0), minx0(0), miny0(0), textdir(0.0), scalx(30.0),
 scaly(-30.0), xcur(0.0), ycur(0.0), xscal(30.0), yscal(-30.0),
 relang(0.0), relsin(0.0), relcos(1.0), dcur(0.0), factor(1.0),
 pen1(Qt::black, 0, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin),
 pen2(Qt::black, 0, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin),
 pen3(Qt::black, 0, Qt::SolidLine, Qt::RoundCap, Qt::RoundJoin)
{
  memset(dash, 0, sizeof(dash));
}

KGraph::~KGraph()
{
}

void KGraph::r2i(double x, double y, int* ix, int* iy)
{
  xcur = x;
  ycur = y;
  *ix = minx + qRound(xscal * (x - xmin));
  *iy = miny + qRound(yscal * (y - ymin));
}

void KGraph::drawLineTo(double x, double y)
{
  QPoint p1(kxcur, kycur);
  r2i(x, y, &kxcur, &kycur);
  drawLine(p1, QPoint(kxcur, kycur));
}

bool KGraph::i2cm(const QPoint& p, double* xcm, double* ycm) const
{
  *xcm = (p.x() - minx0) / (scalx * factor);
  *ycm = (p.y() - miny0) / (scaly * factor);
  return ((p.x() < minx) || (p.x() > maxx) || (p.y() > miny) ||
          (p.y() < maxy));
}

void KGraph::i2r(const QPoint& p, double _fxn, double _fyn,
                 double* xr, double* yr) const
{
  *xr = ((p.x() - minx) / xscal + xmin) / _fxn;
  if (logx)
    *xr = pow(10, *xr);
  *yr = ((p.y() - miny) / yscal + ymin) / _fyn;
  if (logy)
    *yr = pow(10, *yr);
}

void KGraph::cm2r(double xcm, double ycm, double _fxn, double _fyn,
                  double* xr, double* yr) const
{
  i2r(fromcm(xcm, ycm), _fxn, _fyn, xr, yr);
}

void KGraph::draw(double x, double y)
{
  if ((x != xcur) || (y != ycur)) {
    if (dash[1] == dash[0])
      drawLineTo(x, y);
    else {
      dcur = dcur - dash[3] * (int) (dcur / dash[3]);
      double x1 = (x - xcur) * xscal / scalx;
      double y1 = (y - ycur) * yscal / scaly;
      double d0 = sqrt(x1 * x1 + y1 * y1);
      double d = d0;
      x1 = xcur;
      y1 = ycur;
      int i = 0;
      for (int j = 0; j < 3; j++)
        if (dcur >= dash[j])
          i = j + 1;
      while ((dcur + d) > dash[i]) {
        if (double f = (dash[i] - dcur) / d) {
          d = d * (1.0 - f);
          xcur = x - d * (x - x1) / d0;
          ycur = y - d * (y - y1) / d0;
          dcur = dash[i];
          if (i % 2)
            move(xcur, ycur);
          else
            drawLineTo(xcur, ycur);
        }
        if (i == 3)
          dcur = 0.0;
        i = (i + 1) % 4;
      }
      xcur = x;
      ycur = y;
      if (i % 2)
        move(xcur, ycur);
      else
        drawLineTo(xcur, ycur);
      dcur += d;
    }
  }
}

void KGraph::line(double x1, double y1, double x2, double y2)
{
  move(x1, y1);
  draw(x2, y2);
}

void KGraph::plInit(QPaintDevice* pd, int offLeft, int offTop)
{
  QPaintDeviceMetrics pdm(pd);
  begin(pd);
  eraseRect(window());
  if (pd->devType() == QInternal::Printer)
    devType = Printer;
  fDpi = 75.0 / pdm.logicalDpiY();
  scalx = pdm.logicalDpiX() / 2.54;
  scaly = -pdm.logicalDpiY() / 2.54;
  minx0 = minx = -offLeft;
  maxx = pdm.width() + minx;
  miny0 = miny = pdm.height() + offTop;
  maxy = offTop;
  logx = logy = false;
  xscal = scalx;
  yscal = scaly;
  int d = maxx - minx;
  xl = d / xscal;
  yl = (maxy - miny) / yscal;
  factor = 1.0;
  xmin = ymin = 0.0;
  xcur = xmin;
  ycur = ymin;
  xmax = xl;
  ymax = yl;
  relang = 0.0;
  relsin = 0.0;
  relcos = 1.0;
  relSiz = 1.0;
  textdir = 0.0;
  ndigx = ndigy = -1;
  linetype(1);
  pen1.setWidth(d / 240);
  pen2.setWidth(d / 480);
  pen3.setWidth(d / 339);
  setPen(pen3);
  setFont(QFont(face[0], QMAX(qRound(0.0355 * fDpi * maxx), 1)));
  move(xcur, ycur);
#ifndef NO_CLIPPING
  setClipRect(-offLeft, offTop, pdm.width(), pdm.height());
  setClipping(false);
#endif
}

void KGraph::format(double x, double y)
{
  xmin = ymin = 0.0;
  xmax = x;
  ymax = y;
  minx = minx0;
  miny = miny0;
  xscal = scalx * factor;
  yscal = scaly * factor;
  maxx = minx + qRound(xscal * xmax);
  maxy = miny + qRound(yscal * ymax);
  xcur = xmin;
  ycur = ymin;
#ifndef NO_CLIPPING
  setClipRect(minx , maxy, maxx - minx, miny - maxy);
  setClipping(false);
#endif
}

void KGraph::setColFrame(unsigned icol1)
{
  pen1.setColor(QColor(icol1));
}

void KGraph::setColGrid(unsigned icol2)
{
  pen2.setColor(QColor(icol2));
}

void KGraph::setColData(unsigned icol3)
{
  pen3.setColor(QColor(icol3));
  setPen(pen3);
}

void KGraph::setCol(unsigned icol1, unsigned icol2, unsigned icol3)
{
  setColFrame(icol1);
  setColGrid(icol2);
  setColData(icol3);
}

double KGraph::setRelSize(double r)
{
  double old = relSiz;
  relSiz = r;
  double d = relSiz * (maxx - minx);
  pen1.setWidth(qRound(0.00416667 * d));
  pen2.setWidth(qRound(0.00208333 * d));
  pen3.setWidth(qRound(0.00295 * d));
  setPen(pen3);
  QFont fnt = font();
  fnt.setPointSize(QMAX(qRound(0.0355 * fDpi * d), 1));
  setFont(fnt);
  return old;
}

void KGraph::Window(double xmi, double w, double ymi, double h)
{
  xmin = xmi;
  xmax = xmi + w;
  ymin = ymi;
  ymax = ymi + h;
  minx = minx0 + qRound(xscal * xmin);
  maxx = minx + qRound(xscal * w);
  miny = miny0 + qRound(yscal * ymin);
  maxy = miny + qRound(yscal * h);
  xcur = xmin;
  ycur = ymin;
  move(xcur, ycur);
  dcur = 0.0;
  double d = relSiz * (maxx - minx);
  int _dx = qRound(0.00416667 * d);
  pen1.setWidth(_dx);
  pen2.setWidth(qRound(0.00208333 * d));
  pen3.setWidth(qRound(0.00295 * d));
  setPen(pen3);
  QFont fnt = font();
  fnt.setPointSize(QMAX(qRound(0.0355 * fDpi * d), 1));
  setFont(fnt);
#ifndef NO_CLIPPING
  setClipRect(minx + (_dx >> 1), maxy + (_dx >> 1), maxx - minx - _dx,
              miny - maxy - _dx);
  setClipping(false);
#endif
}

void KGraph::scale(double xmi, double xma, double ymi, double yma,
                   bool lgx, bool lgy)
{
  logx = lgx;
  logy = lgy;
  xmin = logx ? log10(xmi) : xmi;
  xmax = logx ? log10(xma) : xma;
  ymin = logy ? log10(ymi) : ymi;
  ymax = logy ? log10(yma) : yma;
  xscal = (maxx - minx) / (xmax - xmin);
  yscal = (maxy - miny) / (ymax - ymin);
  xcur = xmin;
  ycur = ymin;
  move(xcur, ycur);
}

void KGraph::setDig(int nx, int ny)
{
  ndigx = nx;
  ndigy = ny;
}

void KGraph::frame()
{
  setPen(pen1);
  drawRect(minx, maxy, maxx - minx, miny - maxy);
  setPen(pen3);
}

void KGraph::raster(double xtic, double ytic, int intx, int inty, int mode,
                    double xOff, double yOff, bool xBottom, bool xTop,
                    bool yLeft, bool yRight)
{
  if (mode == Nothing)
    return;
  double dx = 0.0, dy, x, y;
  if (mode > FrameOnly) {
    if (logx) {
      if (xtic < 2.5)
        xtic = 2.0;
      else
        if (xtic < 5.5)
          xtic = 3.0;
        else
          if (xtic < 99.0)
            xtic = 10.0;
    }
    if (logy) {
      if (ytic < 2.5)
        ytic = 2.0;
      else
        if (ytic < 5.5)
          ytic = 3.0;
        else
          if (ytic < 99.0)
            ytic = 10.0;
    }
    setPen(pen2);
    dx = 0.99999 * (xmax - xmin);
    dy = 0.99999 * (ymax - ymin);
    if (mode <= AxesWithLabels) {
      dx *= 0.02;
      dy = -dx * xscal / yscal;
    }
    if (xOff && (!logx) && mode <= AxesWithLabels && intx > 1) {
      double d = xtic / intx;
      for (double xm = xmin + xOff - d; xm > xmin; xm -= d) {
        if (xBottom)
          line(xm, ymin, xm, ymin + 0.5 * dy);
        if (xTop)
          line(xm, ymax, xm, ymax - 0.5 * dy);
      }
    }
    for (x = xmin + xOff; x < xmax;)
      if (logx) {
        double xa = x;
        double mant = incLog(&x, xtic);
        if (x < xmax) {
          if (xBottom)
            line(x, ymin, x, ymin + dy);
          if (xTop)
            line(x, ymax, x, ymax - dy);
        }
        if (intx > 1) {
          double xd = pow(10.0, xa - mant);
          for (;;) {
            xa = log10(pow(10.0, xa) + xd);
            if ((xa >= x) || (xa >= xmax))
              break;
            if (xBottom)
              line(xa, ymin, xa,
                   ymin + (mode <= AxesWithLabels ? 0.5 * dy : dy));
            if (xTop)
              line(xa, ymax, xa,
                   ymax - (mode <= AxesWithLabels ? 0.5 * dy : dy));
          }
        }
      } else {
        if (mode <= AxesWithLabels && intx > 1) {
          double xm = x;
          for (int i = 1; i < intx; i++)
            if ((xm += xtic / intx) < xmax) {
              if (xBottom)
                line(xm, ymin, xm, ymin + 0.5 * dy);
              if (xTop)
                line(xm, ymax, xm, ymax - 0.5 * dy);
            }
        }
        if (x > xmin) {
          if (xBottom)
            line(x, ymin, x, ymin + dy);
          if (xTop)
            line(x, ymax, x, ymax - dy);
        }
        x += xtic;
      }
    if (yOff && (!logy) && mode <= AxesWithLabels && inty > 1) {
      double d = ytic / inty;
      for (double ym = ymin + yOff - d; ym > ymin; ym -= d) {
        if (yLeft)
          line(xmin, ym, xmin + 0.5 * dx, ym);
        if (yRight)
          line(xmax, ym, xmax - 0.5 * dx, ym);
      }
    }
    for (y = ymin + yOff; y < ymax;)
      if (logy) {
        double ya = y;
        double mant = incLog(&y, ytic);
        if (y < ymax) {
          if (yLeft)
            line(xmin, y, xmin + dx, y);
          if (yRight)
            line(xmax, y, xmax - dx, y);
        }
        if (inty > 1) {
          double yd = pow(10.0, ya - mant);
          for (;;) {
            ya = log10(pow(10.0, ya) + yd);
            if ((ya >= y) || (ya >= ymax))
              break;
            if (yLeft)
              line(xmin, ya,
                   xmin + (mode <= AxesWithLabels ? 0.5 * dx : dx), ya);
            if (yRight)
              line(xmax, ya,
                   xmax - (mode <= AxesWithLabels ? 0.5 * dx : dx), ya);
          }
        }
      } else {
        if (mode <= AxesWithLabels && inty > 1) {
          double ym = y;
          for (int i = 1; i < inty; i++)
            if ((ym += ytic / inty) < ymax) {
              if (yLeft)
                line(xmin, ym, xmin + 0.5 * dx, ym);
              if (yRight)
                line(xmax, ym, xmax - 0.5 * dx, ym);
            }
        }
        if (y > ymin) {
          if (yLeft)
            line(xmin, y, xmin + dx, y);
          if (yRight)
            line(xmax, y, xmax - dx, y);
        }
        y += ytic;
      }
  }
  if ((mode == AxesWithLabels) || (mode == GridWithLabels)) {
    QString label;
    double yB = 0.0;
    double yT = 0.0;
    double d = 0.03225 * relSiz * (xmax - xmin) * xscal / yscal;
    if (xBottom)
      yB = ymin + d;
    if (xTop)
      yT = ymax - d;
    int ix, iy;
    bool e = (logx && (xtic > 9.9) && ((xmax > 2.999999) ||
                                       (xmin < -2.999999)));
    int nd = ndigx;
    double xe = xmax + 1.0e-3 * (xmax - xmin);
    double tol = 1.0e-12 * xtic;
    for (x = xmin + xOff; x <= xe;) {
      if (x > xmax)
         x = xmax;
      if (e) {
        if (xBottom) {
          r2i(x, yB, &ix, &iy);
          text(label.sprintf("10#nu%d", qRound(x)), ix, iy, 2);
        }
        if (xTop) {
          r2i(x, yT, &ix, &iy);
          text(label.sprintf("10#nu%d", qRound(x)), ix, iy, 2);
        }
      } else {
        if (logx)
          if (x < 0)
            nd = 1 - int(x + 0.01);
          else
            nd = -1;
        int l = prenum(logx ? pow(10.0, x) :
                              ((fabs(x) < tol) ? 0.0 : x), nd, label);
        setPen(pen1);
        if (xBottom) {
          r2i(x, yB, &ix, &iy);
          drawText(ix, iy, 0, 0, Qt::AlignCenter | Qt::DontClip, label, l);
        }
        if (xTop) {
          r2i(x, yT, &ix, &iy);
          drawText(ix, iy, 0, 0, Qt::AlignCenter | Qt::DontClip, label, l);
        }
      }
      if (logx)
        incLog(&x, xtic);
      else
        x += xtic;
    }
    double xL = 0.0;
    double xR = 0.0;
    d = 0.025 * relSiz * (xmax - xmin);
    if (yLeft)
      xL = xmin - d;
    if (yRight)
      xR = xmax + d;
    dy = -0.5 * d * xscal / yscal;
    e = (logy && (ytic > 9.9) && ((ymax > 2.999999) || (ymin < -2.999999)));
    nd = ndigy;
    double ye = ymax + 1.0e-3 * (ymax - ymin);
    tol = 1.0e-12 * ytic;
    for (y = ymin + yOff; y <= ye;) {
      if (y > ymax)
        y = ymax;
      if (e) {
        if (yLeft) {
          r2i(xL, y, &ix, &iy);
          text(label.sprintf("10#nu%d", qRound(y)), ix, iy, 3);
        }
        if (yRight) {
          r2i(xR, y, &ix, &iy);
          text(label.sprintf("10#nu%d", qRound(y)), ix, iy, 1);
        }
      } else {
        if (logy)
          if (y < 0)
            nd = 1 - int(y + 0.01);
          else
            nd = -1;
        int l = prenum(logy ? pow(10.0, y) :
                              ((fabs(y) < tol) ? 0.0 : y), nd, label);
        setPen(pen1);
        if (yLeft) {
          r2i(xL, y, &ix, &iy);
          drawText(ix, iy - 1, 0, 0,
                   Qt::AlignRight | Qt::AlignVCenter | Qt::DontClip,
                   label, l);
        }
        if (yRight) {
          r2i(xR, y, &ix, &iy);
          drawText(ix, iy - 1, 0, 0,
                   Qt::AlignLeft | Qt::AlignVCenter | Qt::DontClip,
                   label, l);
        }
      }
      if (logy)
        incLog(&y, ytic);
      else
        y += ytic;
    }
  }
  setPen(pen1);
  frame();
  setPen(pen3);
}

void KGraph::setDir(double dir)
{
  textdir = dir;
  if (textdir) {
    relang = 0.01745329252 * textdir;
    relsin = sin(relang);
    relcos = cos(relang);
  } else {
    relang = relsin = 0.0;
    relcos = 1.0;
  }
}

void KGraph::text(const QString& str, int ix, int iy, int iorg, QRect* bRect)
{
  if (int l = str.length()) {
    QWMatrix wm = worldMatrix();
    translate(double(ix), double(iy));
    if (textdir)
      rotate(-textdir);
    setPen(pen1);
    int tf = 0;
    if (!str.contains('#')) {
      switch (iorg) {
        case 1:
          tf = Qt::AlignLeft | Qt::AlignVCenter | Qt::DontClip;
          break;
        case 2:
          tf = Qt::AlignCenter | Qt::DontClip;
          break;
        case 3:
          tf = Qt::AlignRight | Qt::AlignVCenter | Qt::DontClip;
      }
      drawText(0, 0, 0, 0, tf, str, -1, bRect);
    } else {
      const QString fontcode = "NHBOSZI";
      const QString levcode = "NUD";
      const int iface[7] = {0, 0, 0, 0, 1, 2, 0};
      const int weight[7] = {QFont::Normal, QFont::Normal, QFont::Bold,
                             QFont::Normal, QFont::Normal, QFont::Normal,
                             QFont::Bold};
      const bool italic[7] = {false, false, false, true, false, false, true};
      const double ys[2] = {0.6, -0.4};
      int ix2 = 0;
      tf = Qt::AlignLeft | Qt::AlignVCenter | Qt::DontClip;
      double dr = 0.025 * relSiz * (xmax - xmin) * xscal;
      int j0 = (iorg == 1) ? 1 : 0;
      int idx = 0;
      int idy;
      for (int j = j0; j < 2; j++) {
        if (j) {
          if (iorg == 2)
            idx >>= 1;
          if (iorg != 1)
            ix2 = -idx;
        }
        QString s = str;
        int ilev = 0;
        while (s.length()) {
          QString c;
          if ((l = s.find('#')) < 0)
            l = s.length();
          else
            c = s.mid(l);
          if (l) {
            QRect r;
            if (j == 0) {
              r = boundingRect(0, 0, maxx, maxy,
                               Qt::AlignLeft | Qt::AlignVCenter, s, l);
              idx += r.width();
            } else {
              idy = (ilev) ? -qRound(dr * ys[ilev - 1]) : 0;
              drawText(ix2, idy, 0, 0, tf, s, l, &r);
              ix2 += r.width();
              if (bRect)
                *bRect = bRect->unite(r);
            }
          }
          if (!c.isNull()) {
            if (c.length() < 3)
              break;
            s = c.mid(3);
            int ifnt = fontcode.find(c.at(1).upper());
            ilev = levcode.find(c.at(2).upper());
            if ((ifnt >= 0) && (ilev >= 0)) {
              int k = iface[ifnt];
              if (k == 2)
                QFontDatabase().styles(face[k]);
              QFont fnt(face[k],
                        QMAX(qRound((ilev ? 0.0251 : 0.0355) * relSiz * fDpi *
                                    (maxx - minx)), 1),
                        weight[ifnt], italic[ifnt]);
              if (k > 0) {
                fnt.setStyleStrategy(QFont::PreferBitmap);
                if (QFontInfo(fnt).family().find(QString(face[k]), 0, false))
                  fnt.setStyleStrategy(QFont::PreferDefault);
              }
              setFont(fnt);
            }
          } else
            s.remove(0, l);
        }
        setFont(QFont(face[0],
                QMAX(qRound(0.0355 * relSiz * fDpi * (maxx - minx)), 1)));
      }
    }
    setPen(pen3);
    if (bRect)
      *bRect = xForm(*bRect);
    setWorldMatrix(wm);
  }
}

void KGraph::textcm(const QString& str, double x, double y, int al,
                    QRect* bRect)
{
  double xa, ya;
  double dh = 0.013 * relSiz * (maxx - minx) / (scalx * factor);
  cm2r(x - dh * relsin, y + dh * relcos, 1.0, 1.0, &xa, &ya);
  int ix, iy;
  r2i(logx ? log10(xa) : xa, logy ? log10(ya) : ya, &ix, &iy);
  text(str, ix, iy, al, bRect);
}

void KGraph::arrow(double x, double y, double dir, double len,
                   QRect* bRect)
{
  QWMatrix wm = worldMatrix();
  QPoint p = fromcm(x, y);
  translate(double(p.x()), double(p.y()));
  QPainter::scale(1.0, -scaly / scalx);
  if (dir)
    rotate(-dir);
  double dx = 0.03333 * relSiz * (maxx - minx);
  int idx = -qRound(dx);
  int idy = QMAX(1, qRound(0.2 * dx));
  QPointArray a;
  a.setPoints(3, 0, 0, idx, idy, idx, -idy);
  setBrush(pen1.color());
  setPen(Qt::NoPen);
  drawPolygon(a);
  setBrush(Qt::NoBrush);
  setPen(pen1);
  int idx2 = -qRound(factor * scalx * len);
  if (idx2 <= idx)
    drawLine(idx, 0, idx2, 0);
  if (bRect)
    *bRect = xForm(QRect(idx2, -idy, -idx2, idy + idy));
  setPen(pen3);
  setWorldMatrix(wm);
}

void KGraph::arc(double x, double y, double w, double h, double a,
                 double alen, double dir, QRect* bRect)
{
  QWMatrix wm = worldMatrix();
  QPoint p = fromcm(x, y);
  translate(double(p.x()), double(p.y()));
  if (dir)
    rotate(-dir);
  int iw = qRound(factor * scalx * w);
  int ih = -qRound(factor * scaly * h);
  QRect r(-iw / 2, -ih / 2, iw, ih);
  drawArc(r, qRound(16 * a), qRound(16 * alen));
  if (bRect)
    *bRect = xForm(r);
  setWorldMatrix(wm);
}

void KGraph::rectangle(double x, double y, double w, double h,
                       int fillStyle, double dir, QRect* bRect)
{
  QWMatrix wm = worldMatrix();
  QPoint p = fromcm(x, y);
  translate(double(p.x()), double(p.y()));
  if (dir)
    rotate(-dir);
  if (fillStyle) {
    setBrush(QBrush(pen3.color(), (Qt::BrushStyle) pattern[fillStyle]));
    setPen(Qt::NoPen);
  }
  int idy = qRound(factor * scaly * h);
  QRect r(0, idy, qRound(factor * scalx * w), -idy);
  drawRect(r);
  if (fillStyle) {
    setBrush(Qt::NoBrush);
    setPen(pen3);
  }
  if (bRect)
    *bRect = xForm(r);
  setWorldMatrix(wm);
}

void KGraph::ellipse(double x, double y, double w, double h,
                     int fillStyle, double dir, QRect* bRect)
{
  QWMatrix wm = worldMatrix();
  QPoint p = fromcm(x, y);
  translate(double(p.x()), double(p.y()));
  if (dir)
    rotate(-dir);
  if (fillStyle) {
    setBrush(QBrush(pen3.color(), (Qt::BrushStyle) pattern[fillStyle]));
    setPen(Qt::NoPen);
  }
  int iw = qRound(factor * scalx * w);
  int ih = -qRound(factor * scaly * h);
  QRect r(-iw / 2, -ih / 2, iw, ih);
  drawEllipse(r);
  if (fillStyle) {
    setBrush(Qt::NoBrush);
    setPen(pen3);
  }
  if (bRect)
    *bRect = xForm(r);
  setWorldMatrix(wm);
}

void KGraph::image(const KURL& url, double x, double y, double sx, double sy,
                   QRect* bRect)
{
  QString fn;
  double x1, x2, y1, y2;
  if (Utils::boundingBox(url, fn, &x1, &y1, &x2, &y2)) {
    double w = 0.0352778 * sx * (x2 - x1);
    double h = 0.0352778 * sy * (y2 - y1);
    int iw = qRound(factor * scalx * w);
    int ih = qRound(-factor * scaly * h);
    QPoint p = fromcm(x, y + h);
    QImageIO iio;
    iio.setFileName(fn);
    QString s = QString::number(iw) + ":" + QString::number(ih);
    iio.setParameters(s.latin1());
    iio.read();
    drawImage(p, iio.image());
    if (bRect)
      *bRect = QRect(p, QSize(iw, ih));
    if (!url.isLocalFile())
      KIO::NetAccess::removeTempFile(fn);
  } else
    KMessageBox::error(0, url.url());
}

void KGraph::scaleBar(const QString& str, double x, double y, int orientation,
                      double len, QRect* bRect)
{
  QWMatrix wm = worldMatrix();
  QPoint p = fromcm(x, y);
  translate(double(p.x()), double(p.y()));
  QPainter::scale(1.0, -scaly / scalx);
  if (orientation)
    rotate(-90.0);
  int idy = qRound(0.005 * (maxx - minx));
  int idx = qRound(factor * scalx * len);
  setPen(pen1);
  drawLine(0, 0, idx, 0);
  drawLine(0, -idy, 0, idy);
  drawLine(idx, -idy, idx, idy);
  QRect r2;
  if (bRect)
    r2 = xForm(QRect(0, -idy, idx, idy + idy));
  setPen(pen3);
  setWorldMatrix(wm);
  if (orientation)
    textcm(str, x  + 2 * idy / (factor * scalx),
           y + 0.5 * len + 0.014 * relSiz * (maxx - minx) / (factor * scaly),
           1, bRect);
  else
    textcm(str, x + 0.5 * len, y - (2.0 + relSiz) * idy / (factor * scaly),
           2, bRect);
  if (bRect)
    *bRect = bRect->unite(r2);
}

void KGraph::letH(const QString& str, bool xTop)
{
  int ix, iy;
  r2i(0.5 * (xmin + xmax),
      ymax - (xTop ? 0.1875 : 0.04275) * relSiz * (xmax - xmin) * xscal / yscal,
      &ix, &iy);
  text(str, ix, iy, 2);
}

void KGraph::letX(const QString& str, int iex, bool xTop)
{
  letxy(str, iex, 1, xTop);
}

void KGraph::letY(const QString& str, int iex, bool yRight)
{
  letxy(str, iex, 2, yRight);
}

void KGraph::setSymbol(int is)
{
  kSymb = is;
}

void KGraph::circle(double x0, double y0, double r)
{
  QPoint p = fromcm(x0, y0);
  int ir = qRound(factor * scalx * r);
  drawEllipse(p.x() - ir, p.y() - ir, ir + ir, ir + ir);
}

void KGraph::arcrel(double x, double y, double r, double angst, double angend)
{
  int x0, y0;
  r2i(x, y, &x0, &y0);
  int ir = qRound(xscal * fabs(r));
  if (r > 0.0) {
    double phi = 1.74532925e-2 * angst;
    x0 -= qRound(ir * cos(phi));
    y0 += qRound(ir * sin(phi));
  }
  drawArc(x0 - ir, y0 - ir, ir + ir, ir + ir,
          qRound(16.0 * angst), qRound(16.0 * angend));
}

double KGraph::incLog(double* x, double f)
{
  double mant = *x - int(*x + 1.0e-6 * fabs(*x));
  if (fabs(mant) < 1.0e-9)
    mant = 0.0;
  if (mant < 0.0)
    mant += 1.0;
  if (mant < 0.1)
    *x += log10(f);
  else
    if (mant < 0.4)
      *x += log10(2.5);
    else
      if (mant < 0.6)
        *x += log10(10.0 / 3.0);
      else
        *x += log10(2.0);
  return mant;
}

void KGraph::plArray(const double *x, const double *y, double fx, double fy,
                     int n, int fillStyle, bool clip, double xOff, double yOff)
{
  if (kSymb == -17)
    return;
#ifndef NO_CLIPPING
  setClipping(clip);
#endif
  double xa, ya;
  if (fillStyle) {
    setBrush(QBrush(pen3.color(), (Qt::BrushStyle) pattern[fillStyle]));
    setPen(Qt::NoPen);
    QPointArray ap(n);
    int kx, ky;
    int l = 0;
    for (int i = 0; i < n; i++) {
      xa = fx * (xOff + x[i]);
      ya = fy * (yOff + y[i]);
      if (Utils::valid(xa, logx) && Utils::valid(ya, logy)) {
        r2i(logx ? log10(xa) : xa, logy ? log10(ya) : ya, &kx, &ky);
        ap[l++] = QPoint(kx, ky);
      }
    }
    if (l > 2)
      drawPolygon(ap, true, 0, l);
    setBrush(Qt::NoBrush);
    setPen(pen3);
  } else {
    if (kSymb > 0) {
      linetype(kSymb);
      bool b = false;
      for (int i = 0; i < n; i++) {
        xa = fx * (xOff + x[i]);
        ya = fy * (yOff + y[i]);
        if (Utils::valid(xa, logx) && Utils::valid(ya, logy))
          if (b)
            draw(logx ? log10(xa) : xa, logy ? log10(ya) : ya);
          else {
            move(logx ? log10(xa) : xa, logy ? log10(ya) : ya);
            b = true;
          }
      }
      linetype(1);
    } else {
      QPointArray ap(10);
      int k = (-kSymb) % 17;
      if ((k > 10) || (k == 1)) {
        setBrush(pen3.color());
        setPen(Qt::NoPen);
      }
      const int ax[48] = {0, -2, 0, 2, 0, 2, -2, -2, 2, 2, 0, -2, 2, 0,
                          2, -2, 2, -2, 2, 2, 1, -1, -2, -2, -1, 1, 2, 2,
                          2, -2, 0, -2, 2, 0, 0, 0, -2, 2, 2, 0, -2, 0, 0,
                          0, -2, 2, 0, 0},
                ay[48] = {2, 0, -2, 0, 2, 2, 2, -2, -2, 2, 2, -1, -1, 2,
                          2, 2, -2, -2, 2, 1, 2, 2, 1, -1, -2, -2, -1, 1,
                          2, -2, 0, 2, -2, 2, -2, 0, 0, 0, 2, 0, 2, 0, -2,
                          2, 0, 0, 2, -2},
                as[10] = {0, 5, 10, 14, 19, 28, 33, 38, 43, 48};
      int j, r;
      int l = 0;
      double f = 0.005 * relSiz * (maxx - minx);
      for (int i = 0; i < n; i++) {
        xa = fx * (xOff + x[i]);
        ya = fy * (yOff + y[i]);
        if (Utils::valid(xa, logx) && Utils::valid(ya, logy)) {
          move(logx ? log10(xa) : xa, logy ? log10(ya) : ya);
          switch (k) {
            case 0:
            case 11:
              r = qRound(f + f);
              drawEllipse(kxcur - r, kycur - r, r + r, r + r);
              break;
            case 1:
              r = qRound(0.4 * f);
              drawEllipse(kxcur - r, kycur - r, r + r, r + r);
              break;
            default:
              r = (k < 12) ? k : k - 10;
              QPoint p1(kxcur, kycur);
              for (j = as[r - 2]; j < as[r - 1]; j++) {
                QPoint p2(qRound(kxcur + f * ax[j]),
                          qRound(kycur - f * ay[j]));
                l = j - as[r - 2];
                if (k < 12) {
                  if (l)
                    drawLine(p1, p2);
                  p1 = p2;
                } else
                  ap[l] = p2;
              }
              if (k >= 12)
                drawPolygon(ap, false, 0, l);
          }
        }
      }
      if ((k > 10) || (k == 1)) {
        setBrush(Qt::NoBrush);
        setPen(pen3);
      }
    }
  }
#ifndef NO_CLIPPING
  setClipping(false);
#endif
}

void KGraph::plError(const double* x, const double* y, const double* e,
                     double fx, double fy, int n, bool clip, double xOff,
                     double yOff)
{
#ifndef NO_CLIPPING
  setClipping(clip);
#endif
  int idy = qRound(0.01 * relSiz * (maxx - minx));
  int idx = idy >> 1;
  for (int i = 0; i < n; i++)
    if (Utils::valid(x[i], logx) && Utils::valid(y[i], logy)) {
      int ie;
      if (logy)
        ie = int(-0.4342944819 * e[i] * yscal / y[i]);
      else
        ie = int(-fy * e[i] * yscal);
      double xa = fx * (xOff + x[i]);
      if (logx)
        xa = log10(xa);
      double ya = fy * (yOff + y[i]);
      if (logy)
        ya = log10(ya);
      if ((xa >= xmin) && (xa <= xmax) &&
          ((ya - ie * yscal) >= ymin) && ((ya + ie * yscal) <= ymax) &&
          (ie >= idy)) {
        move(xa, ya);
        drawLine(kxcur, kycur + idy, kxcur, kycur + ie);
        drawLine(kxcur - idx, kycur + ie, kxcur + idx, kycur + ie);
        drawLine(kxcur, kycur - idy, kxcur, kycur - ie);
        drawLine(kxcur - idx, kycur - ie, kxcur + idx, kycur - ie);
      }
    }
#ifndef NO_CLIPPING
  setClipping(false);
#endif
}

void KGraph::resetDash()
{
  dcur = 0.0;
}

QPixmap KGraph::pixmap(int symbol, unsigned color)
{
  QPixmap pm((symbol > 0) ? QSize(40, 2) : QSize(16, 16));
  plInit(&pm, 0, 0);
  setColData(color);
  setSymbol(symbol);
  int n;
  double xa[2], ya[2];
  if (symbol > 0) {
    setRelSize(13.0);
    xa[0] = 2.0 / scalx;
    xa[1] = 38.0 / scalx;
    ya[0] = ya[1] = -1.0 / scaly;
    n = 2;
  } else {
    setRelSize(34.0);
    xa[0] = 8.0 / scalx;
    ya[0] = -8.0 / scaly;
    n = 1;
  }
  plArray(xa, ya, 1.0, 1.0, n, false, false);
  end();
  pm.setMask(pm.createHeuristicMask());
  return pm;
}

void KGraph::linetype(int is)
{
  const int pat[9][2] = {{8, 8}, {7, 0}, {2, 0}, {1, 0}, {9, 0},
                         {8, 1}, {7, 2}, {4, 4}, {1, 1}};

  int j = QMAX(0, (is - 1) % 9);
  double f = relSiz * (maxx - minx) / (scalx * 320.0);
  double dsp = (double) ((16 - pat[j][0] - pat[j][1]) >> 1);
  dash[0] = f * pat[j][0];
  dash[1] = dash[0] + f * dsp;
  dash[2] = dash[1] + f * pat[j][1];
  dash[3] = dash[2] + f * dsp;
}

void KGraph::letxy(const QString& str, int iex, int iax, bool opposite)
{
  if (str.length() || iex) {
    double x, y;
    QString label;
    double dx = 0.025 * relSiz * (xmax - xmin);
    double dirsav = textdir;
    if (iax == 2) {
      int l = 3;
      if (logy)
        if ((ymax > 2.999999) || (ymin < -2.999999))
          l += (ymin < 0) ? 4 : 3;
        else
          l += 1 + int(QMAX(ymax, 1.99 - ymin));
      else
        l += QMAX(prenum(ymin, ndigy, label), prenum(ymax, ndigy, label));
      if (opposite)
        x = xmax + (0.75 * l + 0.71) * dx;
      else
        x = xmin - (0.75 * l + 0.71) * dx;
      y = 0.5 * (ymin + ymax);
      setDir(opposite ? -90.0 : 90.0);
    } else {
      x = 0.5 * (xmin + xmax);
      if (opposite)
        y = ymax - 3.79 * dx * xscal / yscal;
      else
        y = ymin + 3.79 * dx * xscal / yscal;
    }
    int ix, iy;
    r2i(x, y, &ix, &iy);
    if (iex)
      text("10#nu" + QString::number(iex) + "#nn " + str, ix, iy, 2);
    else
      text(str, ix, iy, 2);
    if (iax == 2)
      setDir(dirsav);
  }
}
