/***************************************************************************
                          fitdlg.h  -  description
                             -------------------
    begin                : Tue May 25 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef FITDLG_H
#define FITDLG_H

#include <qmemarray.h>
#include <qptrlist.h>
#include <kdialogbase.h>
#include "kplnamespace.h"

class QButtonGroup;
class QCheckBox;
class QLabel;
class QLineEdit;
class QListBox;
class KConfig;
class KURL;
class ArrayItem;
class FunItem;
class KplDoc;
class KplDoubleEdit;
class KplSpinBox;
class LMFit;
class SourceWidget;

/**
  * Correlation matrix dialog class. Provides a dialog for displaying
  * the parameter correlation matrix.
  *@author Werner Stille
  */
class CorrDlg : public KDialog
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param np number of fitted parameters.
   * @param ip array with parameter indices.
   * @param corr np * np matrix with correlation coefficients.
   */
  CorrDlg(QWidget* parent, int np, const int* ip, const double* corr);
  /** Destructor. */
  ~CorrDlg();
};

/**
  * Error model dialog class. Provides a dialog for setting
  * the calculation of the error bars for nonlinear fits.
  * @author Werner Stille
  */
class ErrModDlg : public KDialogBase
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param model pointer to KplDoc model object.
   * @param err0 pointer to structure containing the data error settings.
   */
  ErrModDlg(QWidget* parent, KplDoc* model,
            KplNamespace::DataErrorStruct* err0);
  /** Destructor. */
  ~ErrModDlg();

protected:
  KplDoc* m;
  KplNamespace::DataErrorStruct *err, *errt;
  QLineEdit *fileName, *func;
  QButtonGroup *bg;

protected slots:
  /** Displays the dialog for selection of a library file to load. */
  void slotFile();
  /** Displays the dialog for selection of a function. */
  void slotFunc();
  /** Displays the dialog for selection of a parameter file to read. */
  void slotGetParameter();
  /** Displays the dialog for parameter editing. */
  void slotEditParameter();
  /** Displays the dialog for selection of a parameter file to save. */
  void slotSaveParameter();
  /** Accepts changes and leaves the dialog. */
  virtual void slotOk();
};

/**
  * Fit dialog class. Provides a dialog for performing a parameter fit
  * of an ordinary function to a data set of an array.
  *@author Werner Stille
  */
class FitDlg : public KDialogBase
{
  Q_OBJECT
public:
  /**
   * Constructor.
   * @param parent pointer to parent widget.
   * @param model pointer to KplDoc model object.
   * @param ad pointer to array item list.
   * @param fd pointer to function item list.
   * @param mode combination of ShowDlg and AutoFit.
   */
  FitDlg(QWidget* parent, KplDoc* model, QPtrList<ArrayItem>* ad,
         QPtrList<FunItem>* fd, int mode);
  /** Destructor. */
  ~FitDlg();
  /**
   * Checks and copies settings and parameters.
   * @param accept true for leaving the dialog, false for applying.
   */
  void getValues(bool accept);
  /** Performs Levenberg-Marquardt fit. */
  void fit();
  /** Dialog mode flags. */
  enum DlgMode{ShowDlg = 1, AutoFit = 2, SavePar = 4, Follow = 8};

protected:
  bool running;
  int ip[KPL_NPMAX];
  int np, dlgMode;
  double corr[KPL_NPMAX * KPL_NPMAX];
  KplDoc* m;
  QMemArray<double> fvec, sig;
  QPtrList<ArrayItem>* ad;
  QPtrList<FunItem>* fd;
  QPtrList<QWidget> widgetList;
  KplNamespace::AutoStruct* aut;
  LMFit* lm;
  QListBox* results;
  KplDoubleEdit* eTol;
  KplDoubleEdit* par[KPL_NPMAX];
  QLabel* lArr;
  QLabel* epar[KPL_NPMAX];
  QCheckBox* enFit[KPL_NPMAX];
  QCheckBox *errCol, *nonLin;
  KplSpinBox *sArr, *sItmax;
  QPushButton* showCorr;
  QPushButton* errMod;
  KplNamespace::DataErrorStruct* err;
  KConfig* config;
  SourceWidget* sw;

protected slots:
  /** Displays the dialog for selection of a parameter file to read. */
  void slotGetParameter();
  /** Displays the dialog for selection of a parameter file to save. */
  void slotSaveParameter();
  /**
   * Disables/enables error model button.
   * @param state error column state, disables button when true.
   */
  void errColToggled(bool state);
  /**
   * Disables/enables widgets for nonlinear fits.
   * @param on enables widgets when true.
   */
  void enableNonLin(bool on);
  /**
   * updates display for selected array in data error settings.
   * @param i index of array in list of arrays to be fitted.
   */
  void updArray(int i);
  /** Performs Levenberg-Marquardt fit. */
  virtual void slotUser1();
  /** Saves and shows residuals. */
  virtual void slotUser2();
  /** Applies settings without leaving the dialog (if displayed). */
  virtual void slotApply();
  /** Accepts changes and leaves the dialog. */
  virtual void slotOk();
  /** Stops fitting or leaves the dialog without applying the changes. */
  virtual void slotCancel();
  /** Displays the dialog for setting the calculation of error bars. */
  void slotErrMod();
  /**
   * Displays message line in listbox.
   * @param msg message text.
   */
  void slotMessage(const QString& msg);
  /** Displays parameter correlation matrix. */
  void slotCorr();
};

#endif
