/***************************************************************************
                          arraydlg.cpp  -  description
                             -------------------

    This file is a part of kpl - a program for graphical presentation of
    data sets and functions.

    begin                : Sun Apr 25 1999
    copyright            : (C) 2005 by Werner Stille
    email                : stille@uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qcheckbox.h>
#include <qfileinfo.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qtextstream.h>
#include <kcolorbutton.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kurl.h>
#include "arraydlg.h"
#include "arrayitem.h"
#include "brushdlg.h"
#include "editdlg.h"
#include "kpl.h"
#include "kpldoc.h"
#include "kpldoubleedit.h"
#include "kpldoublespinbox.h"
#include "kplspinbox.h"
#include "symboldlg.h"
#include "utils.h"

ArrayDlg::ArrayDlg(QWidget* _parent, KplDoc* model, ArrayItem* _ad) :
 KDialogBase(Plain, i18n("Array"), Help | Ok | Apply | Cancel, Ok, _parent, 0,
             true),
 m(model), ad(_ad), adt(0)
{
  QFrame* frame = plainPage();
  QVBoxLayout* vbox = new QVBoxLayout(frame, 0, spacingHint());
  QGroupBox* g = new QGroupBox(0, Qt::Vertical, i18n("Data"), frame);
  vbox->addWidget(g);
  QGridLayout* grid = new QGridLayout(g->layout(), 7, 6, spacingHint());
  QPushButton* b = new QPushButton(i18n("&File"), g);
  grid->addWidget(b, 0, 0);
  connect(b, SIGNAL(clicked()), SLOT(slotFile()));
  QHBoxLayout* hbox = new QHBoxLayout(spacingHint());
  hbox->addWidget(fileName = new QLabel(g));
  if (!ad->internal)
    fileName->setText(ad->url.isLocalFile() ?  ad->url.path() : ad->url.url());
  fileName->setFrameStyle(QFrame::Panel | QFrame::Sunken);
  fileName->setSizePolicy(QSizePolicy::MinimumExpanding,
                          QSizePolicy::Minimum);
  hbox->addWidget(bReload = new QPushButton(g));
  bReload->setIconSet(BarIconSet("reload"));
  bReload->setDisabled(fileName->text().isEmpty());
  connect(bReload, SIGNAL(clicked()), SLOT(slotReload()));
  grid->addMultiCellLayout(hbox, 0, 0, 1, 5);
  grid->addWidget(internal = new QCheckBox(i18n("Internal data"), g), 1, 1);
  internal->setChecked(ad->internal);
  grid->addMultiCellWidget(edit = new QPushButton(i18n("&Edit"), g),
                           1, 1, 2, 4);
  edit->setEnabled(ad->internal);
  connect(edit, SIGNAL(clicked()), SLOT(slotEdit()));
  grid->addWidget(new QLabel(i18n("x column"), g), 2, 0);
  grid->addWidget(ix = new KplSpinBox(0, 0, 1, g), 2, 1);
  grid->addItem(new QSpacerItem(20, 10, QSizePolicy::MinimumExpanding), 2, 3);
  grid->addWidget(new QLabel(i18n("y column"), g), 2, 4);
  grid->addWidget(iy = new KplSpinBox(0, 0, 1, g), 2, 5);
  grid->addWidget(new QLabel(i18n("Error column"), g), 3, 0);
  grid->addWidget(ie = new KplSpinBox(0, 0, 1, g), 3, 1);
  ie->setEnabled(ad->errbars);
  grid->addMultiCellWidget(err = new QCheckBox(i18n("Error bar"), g),
                           3, 3, 2, 5);
  err->setChecked(ad->errbars);
  grid->addWidget(new QLabel(i18n("Start index"), g), 4, 0);
  grid->addWidget(iStart = new KplSpinBox(0, 0, 1, g), 4, 1);
  grid->addWidget(new QLabel(i18n("Number of points"), g), 4, 4);
  grid->addWidget(n = new KplSpinBox(0, 0, 1, g), 4, 5);
  grid->addWidget(new QLabel(i18n("x shift"), g), 5, 0);
  Kpl* kpl = ((Kpl*) m->parent());
  grid->addWidget(ex0 = new KplDoubleSpinBox(ad->x0, kpl->xStep(), g), 5, 1);
  grid->addWidget(new QLabel(i18n("y shift"), g), 5, 4);
  grid->addWidget(ey0 = new KplDoubleSpinBox(ad->y0, kpl->yStep(), g), 5, 5);
  grid->addWidget(new QLabel(i18n("x normalization"), g), 6, 0);
  grid->addWidget(efx = new KplDoubleEdit(ad->fx, g), 6, 1);
  grid->addWidget(new QLabel(i18n("y normalization"), g), 6, 4);
  grid->addWidget(efy = new KplDoubleEdit(ad->fy, g), 6, 5);
  vbox->addWidget(g = new QGroupBox(0, Qt::Vertical, i18n("Representation"),
                                    frame));
  grid = new QGridLayout(g->layout(), 2, 6, spacingHint());
  grid->addWidget(new QLabel(i18n("Symbol"), g), 0, 0);
  grid->addWidget(symb = new KplSpinBox(-17, 9, 1, g), 0, 1);
  symb->setValue(ad->symb);
  SymbolButton* bSymb;
  grid->addWidget(bSymb = new SymbolButton(g, symb, ad->fillStyle), 0, 2);
  grid->addWidget(new QLabel(i18n("Color"), g), 0, 4);
  grid->addWidget(colData = new KColorButton(ad->color, g), 0, 5);
  colData->setMinimumWidth(50);
  grid->addWidget(new QLabel(i18n("Size"), g), 1, 0);
  grid->addWidget(eRelSize = new KplSpinBox(10, 800, 1, g), 1, 1);
  eRelSize->setValue(qRound(100 * ad->relsiz));
  eRelSize->setSuffix(" %");
  eRelSize->setDisabled(ad->fillStyle);
  grid->addWidget(new QLabel(i18n("Fill pattern"), g), 1, 4);
  grid->addWidget(fill = new BrushButton(g, ad->fillStyle), 1, 5);
  setSpinBoxes(ad);
  Utils::setSize(this, "ArrayDialog");
  setHelp("SEC-ARRAY");
  connect(internal, SIGNAL(toggled(bool)), SLOT(slotInternal(bool)));
  connect(err, SIGNAL(toggled(bool)), ie, SLOT(setEnabled(bool)));
  connect(iStart, SIGNAL(valueChanged(int)), SLOT(slotStartChanged(int)));
  connect(n, SIGNAL(valueChanged(int)), SLOT(slotNChanged(int)));
  connect(fill, SIGNAL(valueChanged(bool)), symb, SLOT(setDisabled(bool)));
  connect(fill, SIGNAL(valueChanged(bool)), bSymb, SLOT(setDisabled(bool)));
  connect(fill, SIGNAL(valueChanged(bool)), eRelSize, SLOT(setDisabled(bool)));
  connect(ex0, SIGNAL(valueChanged(double)), SLOT(slotApply()));
  connect(ey0, SIGNAL(valueChanged(double)), SLOT(slotApply()));
}

ArrayDlg::~ArrayDlg()
{
  Utils::saveSize(this, "ArrayDialog");
  delete adt;
}

void ArrayDlg::setSpinBoxes(ArrayItem* _ad)
{
  int iColMax = QMAX(0, _ad->ncols - 1);
  ix->setRange(0, iColMax);
  ix->setValue(_ad->ix);
  iy->setRange(0, iColMax);
  iy->setValue(_ad->iy);
  ie->setRange(0, iColMax);
  ie->setValue(_ad->ie);
  iStart->setRange(0, QMAX(0, _ad->nrows - 1));
  iStart->setValue(_ad->istart);
  n->setRange(0, _ad->nrows);
  n->setValue(_ad->n);
}

bool ArrayDlg::load(KURL &url, int idec)
{
  bool success = !url.isEmpty();
  if (success) {
    delete adt;
    adt = new ArrayItem;
    adt->url = url;
    adt->idec = idec;
    success = readData();
  }
  return success;
}

void ArrayDlg::getValues(bool ok)
{
  if (adt) {
    *ad = *adt;
    delete adt;
    adt = 0;
  }
  ad->x0 = ex0->value();
  ad->y0 = ey0->value();
  ad->fx = efx->value();
  ad->fy = efy->value();
  ad->color = colData->color().rgb();
  ad->ix = ix->interpretedValue();
  ad->iy = iy->interpretedValue();
  ad->ie = ie->interpretedValue();
  ad->symb = symb->interpretedValue();
  ad->relsiz = 0.01 * eRelSize->interpretedValue();
  ad->istart = iStart->interpretedValue();
  ad->url = fileName->text();
  ad->errbars = err->isChecked();
  ad->internal = internal->isChecked();
  ad->fillStyle = fill->value();
  ad->istart = QMAX(ad->istart, 0);
  ad->n = QMIN(n->interpretedValue(), ad->nrows - ad->istart);
  m->setModified();
  m->backupItems();
  if (ok)
    accept();
  else
    setSpinBoxes(ad);
}

void ArrayDlg::slotStartChanged(int ia)
{
  int nmax = ad->nrows - ia;
  if (n->interpretedValue() > nmax)
    n->setValue(nmax);
}

void ArrayDlg::slotNChanged(int na)
{
  int imin = ad->nrows - na;
  if (iStart->interpretedValue() > imin)
    iStart->setValue(imin);
}

void ArrayDlg::slotFile()
{
  int idec = adt ? adt->idec : ad->idec;
  KURL url = m->getReadURL(idec, (adt ? adt : ad)->url.fileName());
  if (load(url, idec)) {
    fileName->setText(url.isLocalFile() ? url.path() : url.url());
    m->setCurrentDir(url);
    bReload->setDisabled(fileName->text().isEmpty());
    disconnect(internal, SIGNAL(toggled(bool)), this,
               SLOT(slotInternal(bool)));
    internal->setChecked(false);
    edit->setEnabled(false);
    connect(internal, SIGNAL(toggled(bool)), SLOT(slotInternal(bool)));
  }
}

void ArrayDlg::slotReload()
{
  KURL url(fileName->text());
  load(url, adt ? adt->idec : ad->idec);
}

void ArrayDlg::slotInternal(bool state)
{
  if (state) {
    ArrayItem* a2 = 0;
    if (adt)
      a2 = new ArrayItem(*adt);
    delete adt;
    adt = new ArrayItem;
    ArrayItem* src = a2 ? a2 : ad;
    adt->url = KURL();
    adt->nrows = src->nrows;
    adt->ncols = 3;
    adt->istart = iStart->interpretedValue();
    adt->n = n->interpretedValue();
    int dim = adt->ncols * adt->nrows;
    if (dim) {
      double* x1 = new double[dim];
      memset(x1, 0, dim * sizeof(double));
      adt->x = new double*[adt->ncols];
      for (int j = 0; j < adt->ncols; j++)
        adt->x[j] = &x1[j * adt->nrows];
      if (src->ncols > 3) {
        int siz = adt->nrows * sizeof(double);
        memcpy(adt->x[0], src->x[ix->interpretedValue()], siz);
        memcpy(adt->x[1], src->x[iy->interpretedValue()], siz);
        memcpy(adt->x[2], src->x[ie->interpretedValue()], siz);
        adt->iy = 1;
        adt->ie = 2;
      } else {
        memcpy(adt->x[0], src->x[0], src->ncols * src->nrows * sizeof(double));
        adt->ix = ix->interpretedValue();
        adt->iy = iy->interpretedValue();
        adt->ie = ie->interpretedValue();
      }
    }
    setSpinBoxes(adt);
    fileName->clear();
    adt->idec = 0;
    delete a2;
  } else {
    ArrayItem* src = adt ? adt : ad;
    KURL url;
    if (m->getWriteURL(this, url, "*.dat\n*")) {
      QFile f(url.isLocalFile() ? url.path() : m->tmpFile());
      bool success = f.open(IO_WriteOnly);
      if (success) {
        QTextStream t(&f);
        for (int i = 0; i < src->nrows; i++)
          for (int j = 0; j < src->ncols; j++)
            t << m->number(src->x[j][i])
              << (j < (src->ncols - 1) ? " " : "\n");
        f.close();
        fileName->setText(url.isLocalFile() ? url.path() : url.url());
        m->setCurrentDir(url);
        if (!url.isLocalFile())
          m->copyTmp(f.name(), url);
      } else {
        KMessageBox::error(this, i18n("while trying to open file"));
        internal->setChecked(true);
        return;
      }
    } else {
      internal->setChecked(true);
      return;
    }
  }
  edit->setEnabled(state);
  bReload->setDisabled(fileName->text().isEmpty());
}

void ArrayDlg::slotEdit()
{
  QTextStream t(&buf, IO_WriteOnly);
  ArrayItem* src = adt ? adt : ad;
  for (int i = 0; i < src->nrows; i++)
    for (int j = 0; j < src->ncols; j++)
      t << m->number(src->x[j][i])
        << (j < (src->ncols - 1) ? m->separator() : QString("\n"));
  EditDlg dlg(this, &buf, m->options()->prec + 4);
  connect(&dlg, SIGNAL(applyClicked()), SLOT(slotEditApply()));
  if (dlg.exec()) {
    delete adt;
    adt = new ArrayItem;
    readData(true);
  }
}

bool ArrayDlg::readData(bool fromBuffer)
{
  if (fromBuffer)
    adt->readFile(&buf);
  else
    adt->readFile();
  if (adt->nrows) {
    adt->adjustCols(ad->ix, ad->iy, ad->ie);
    adt->adjustRows(ad->istart, fromBuffer ? adt->nrows : ad->n);
    setSpinBoxes(adt);
    return adt->nrows;
  } else {
    delete adt;
    adt = 0;
    KMessageBox::error(this, i18n("while reading the data"));
    return 0;
  }
}

void ArrayDlg::slotEditApply()
{
  delete adt;
  adt = new ArrayItem;
  if (readData(true))
    slotApply();
}

void ArrayDlg::slotOk()
{
  getValues(true);
}

void ArrayDlg::slotApply()
{
  getValues(false);
}
