/*
  Copyright (c) 2007 Loïc Corbasson <loic.corbasson@gmail.com>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Library General Public
  License as published by the Free Software Foundation; either
  version 2 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Library General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; see the file COPYING.LIB.  If not, write to
  the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
  Boston, MA 02110-1301, USA.
*/
#ifndef EVENTVIEWS_CALENDARDECORATION_H
#define EVENTVIEWS_CALENDARDECORATION_H

#include "eventviews_export.h"

#include <CalendarSupport/Plugin>

#include <QUrl>

#include <QDate>
#include <QPixmap>

namespace EventViews
{

namespace CalendarDecoration
{

/**
  @class Element

  @brief Class for calendar decoration elements

  It provides entities like texts and pictures for a given date.
  Implementations can implement all functions or only a subset.
 */
class EVENTVIEWS_EXPORT Element : public QObject
{
    Q_OBJECT

public:
    typedef QList<Element *> List;

    explicit Element(const QString &id);
    virtual ~Element();

    /**
      Return a name for easy identification.
      This will be used for example for internal configuration (position, etc.),
      so don't i18n it and make it unique for your decoration.
    */
    virtual QString id() const;

    /**
      Description of element.
    */
    virtual QString elementInfo() const;

    /**
      Return a short text for a given date,
      usually only a few words.
    */
    virtual QString shortText();

    /**
      Return a long text for a given date.
      This text can be of any length,
      but usually it will have one or a few lines.

      Can for example be used as a tool tip.
    */
    virtual QString longText();

    /**
      Return an extensive text for a given date.
      This text can be of any length,
      but usually it will have one or a few paragraphs.
    */
    virtual QString extensiveText();

    /**
      Return a pixmap for a given date and a given size.
    */
    virtual QPixmap newPixmap(const QSize &);

    /**
      Return a URL pointing to more information about the content of the
      element.
     */
    virtual QUrl url();

Q_SIGNALS:
    void gotNewPixmap(const QPixmap &) const;
    void gotNewShortText(const QString &) const;
    void gotNewLongText(const QString &) const;
    void gotNewExtensiveText(const QString &) const;
    void gotNewUrl(const QUrl &) const;

protected:
    QString mId;
};

/**
  This class provides a stored element, which contains all data for the given
  date/month/year.
*/
class EVENTVIEWS_EXPORT StoredElement : public Element
{
    Q_OBJECT
public:
    explicit StoredElement(const QString &id);
    StoredElement(const QString &id, const QString &shortText);
    StoredElement(const QString &id, const QString &shortText,
                  const QString &longText);
    StoredElement(const QString &id, const QString &shortText,
                  const QString &longText, const QString &extensiveText);
    StoredElement(const QString &id, const QPixmap &pixmap);

    virtual void setShortText(const QString &t);
    QString shortText() Q_DECL_OVERRIDE;

    virtual void setLongText(const QString &t);
    QString longText() Q_DECL_OVERRIDE;

    virtual void setExtensiveText(const QString &t);
    QString extensiveText() Q_DECL_OVERRIDE;

    virtual void setPixmap(const QPixmap &p);
    virtual QPixmap pixmap();

    virtual void setUrl(const QUrl &u);
    QUrl url() Q_DECL_OVERRIDE;

protected:
    QString mShortText;
    QString mLongText;
    QString mExtensiveText;
    QPixmap mPixmap;
    QUrl mUrl;
};

/**
  @class Decoration

  @brief This class provides the interface for a date dependent decoration.

  The decoration is made of various decoration elements,
  which show a defined text/picture/widget for a given date.
 */
class EVENTVIEWS_EXPORT Decoration : public CalendarSupport::Plugin
{
public:
    static int interfaceVersion()
    {
        return 2;
    }
    static QString serviceType()
    {
        return QStringLiteral("Calendar/Decoration");
    }

    typedef QList<Decoration *> List;

    Decoration();
    virtual ~Decoration();

    /**
      Return all Elements for the given day.
    */
    virtual Element::List dayElements(const QDate &date);

    /**
      Return all elements for the week the given date belongs to.
    */
    virtual Element::List weekElements(const QDate &d);

    /**
      Return all elements for the month the given date belongs to.
    */
    virtual Element::List monthElements(const QDate &d);

    /**
      Return all elements for the year the given date belongs to.
    */
    virtual Element::List yearElements(const QDate &d);

protected:
    /**
      Register the given elements for the given date. They will be deleted when
      this object is destroyed.
    */
    Element::List registerDayElements(const Element::List &e, const QDate &d);

    /**
      Register the given elements for the week the given date belongs to. They
      will be deleted when this object is destroyed.
    */
    Element::List registerWeekElements(const Element::List &e, const QDate &d);

    /**
      Register the given elements for the month the given date belongs to. They
      will be deleted when this object is destroyed.
    */
    Element::List registerMonthElements(const Element::List &e, const QDate &d);

    /**
      Register the given elements for the year the given date belongs to. They
      will be deleted when this object is destroyed.
    */
    Element::List registerYearElements(const Element::List &e, const QDate &d);

    /**
      Create day elements for given date.
    */
    virtual Element::List createDayElements(const QDate &);

    /**
      Create elements for the week the given date belongs to.
    */
    virtual Element::List createWeekElements(const QDate &);

    /**
      Create elements for the month the given date belongs to.
    */
    virtual Element::List createMonthElements(const QDate &);

    /**
      Create elements for the year the given date belongs to.
    */
    virtual Element::List createYearElements(const QDate &);

    /**
      Map all dates of the same week to a single date.
    */
    QDate weekDate(const QDate &date);

    /**
      Map all dates of the same month to a single date.
    */
    QDate monthDate(const QDate &date);

    /**
      Map all dates of the same year to a single date.
    */
    QDate yearDate(const QDate &date);

private:
    QMap<QDate, Element::List> mDayElements;
    QMap<QDate, Element::List> mWeekElements;
    QMap<QDate, Element::List> mMonthElements;
    QMap<QDate, Element::List> mYearElements;
};

class EVENTVIEWS_EXPORT DecorationFactory : public CalendarSupport::PluginFactory
{
    Q_OBJECT
public:
    Decoration *createPluginFactory() Q_DECL_OVERRIDE = 0;
};

}

}

#endif
