#![allow(clippy::derived_hash_with_manual_eq)]

mod about_dialog;
mod account_widget;
mod add_dialog;
mod app;
mod article_list;
mod article_popover;
mod article_view;
mod color;
mod config;
mod content_page;
mod discover;
mod edit_category_dialog;
mod edit_feed_dialog;
mod edit_tag_dialog;
mod enclosure;
mod error;
mod error_dialog;
mod global_actions;
mod gobject_models;
mod i18n;
mod image_dialog;
mod image_widget;
mod infrastructure;
mod login_screen;
mod main_window;
mod reset_page;
mod saved_state;
mod self_stack;
mod settings;
mod share;
mod shortcuts_dialog;
mod sidebar;
mod tag_popover;
mod themes;
mod undo_action;
mod util;
mod welcome_screen;

use crate::app::App;
use crate::config::APP_ID;
use gettextrs::*;
use gtk4::prelude::*;
use tracing::level_filters::LevelFilter;
use tracing_subscriber::prelude::*;

fn main() {
    // nicer backtrace
    color_backtrace::install();

    let mut layers = Vec::new();
    let filter = tracing_subscriber::EnvFilter::builder()
        .with_default_directive(LevelFilter::INFO.into())
        .from_env_lossy();

    let targets = tracing_subscriber::filter::Targets::new()
        .with_target("r2d2", LevelFilter::OFF)
        .with_target("html5ever", LevelFilter::OFF)
        .with_default(LevelFilter::INFO);

    let stdout_layer = tracing_subscriber::fmt::layer().compact().with_ansi(true).boxed();
    layers.push(stdout_layer);

    if std::env::var("FILE_LOG").is_ok() {
        if !crate::app::STATE_DIR.exists() {
            std::fs::create_dir_all(crate::app::STATE_DIR.as_path()).unwrap();
        }

        let log_path = crate::app::STATE_DIR.join("newsflash.log");
        println!("log to file: {log_path:?}");

        let log_file = std::fs::OpenOptions::new()
            .read(true)
            .append(true)
            .create(true)
            .open(log_path)
            .expect("Failed to init log file");
        let file_layer = tracing_subscriber::fmt::layer()
            .with_ansi(false)
            .with_writer(log_file)
            .boxed();

        layers.push(file_layer);
    }

    tracing_subscriber::Registry::default()
        .with(filter)
        .with(targets)
        .with(layers)
        .init();

    // Gtk setup
    gtk4::init().expect("Error initializing gtk.");
    gtk4::Window::set_default_icon_name(APP_ID);
    glib::set_application_name("Newsflash");
    glib::set_prgname(Some("news-flash"));

    clapper::init().expect("Failed to initialize clapper");

    // Setup translations
    setlocale(LocaleCategory::LcAll, "");
    let localedir = bindtextdomain(config::GETTEXT_PACKAGE, config::LOCALEDIR).expect("Unable to bind the text domain");
    let textdomain = textdomain(config::GETTEXT_PACKAGE).expect("Unable to switch to the text domain");

    tracing::trace!(?localedir);
    if let Ok(textdomain) = std::str::from_utf8(&textdomain) {
        tracing::debug!(%textdomain);
    }

    // Run app itself
    App::new().run();
}
