/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.pinot.segment.local.segment.creator;

import org.apache.pinot.common.Utils;
import org.apache.pinot.segment.local.segment.creator.impl.stats.SegmentPreIndexStatsCollectorImpl;
import org.apache.pinot.segment.spi.creator.SegmentCreationDataSource;
import org.apache.pinot.segment.spi.creator.SegmentPreIndexStatsCollector;
import org.apache.pinot.segment.spi.creator.StatsCollectorConfig;
import org.apache.pinot.spi.data.readers.GenericRow;
import org.apache.pinot.spi.data.readers.RecordReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;


/**
 * {@link SegmentCreationDataSource} that uses a
 * {@link RecordReader} as the underlying data source.
 */
// TODO: make it Closeable so that resource in record reader can be released
public class RecordReaderSegmentCreationDataSource implements SegmentCreationDataSource {
  private static final Logger LOGGER = LoggerFactory.getLogger(RecordReaderSegmentCreationDataSource.class);

  private final RecordReader _recordReader;
  private TransformPipeline _transformPipeline;

  public RecordReaderSegmentCreationDataSource(RecordReader recordReader) {
    _recordReader = recordReader;
  }

  public void setTransformPipeline(TransformPipeline transformPipeline) {
    _transformPipeline = transformPipeline;
  }

  @Override
  public SegmentPreIndexStatsCollector gatherStats(StatsCollectorConfig statsCollectorConfig) {
    try {
      TransformPipeline transformPipeline = _transformPipeline != null ? _transformPipeline
          : new TransformPipeline(statsCollectorConfig.getTableConfig(), statsCollectorConfig.getSchema());

      SegmentPreIndexStatsCollector collector = new SegmentPreIndexStatsCollectorImpl(statsCollectorConfig);
      collector.init();

      // Gather the stats
      boolean continueOnError =
          statsCollectorConfig.getTableConfig().getIngestionConfig() != null && statsCollectorConfig.getTableConfig()
              .getIngestionConfig().isContinueOnError();
      GenericRow reuse = new GenericRow();
      while (_recordReader.hasNext()) {
        reuse.clear();
        try {
          reuse = _recordReader.next(reuse);
          TransformPipeline.Result result = transformPipeline.processRow(reuse);
          for (GenericRow row : result.getTransformedRows()) {
            collector.collectRow(row);
          }
        } catch (Exception e) {
          if (!continueOnError) {
            throw new RuntimeException("Caught exception while reading data", e);
          }
          LOGGER.debug("Caught exception while reading data", e);
        }
      }
      transformPipeline.reportStats();
      collector.build();
      return collector;
    } catch (Exception e) {
      LOGGER.error("Caught exception while gathering stats", e);
      Utils.rethrowException(e);
      return null;
    }
  }

  @Override
  public RecordReader getRecordReader() {
    try {
      _recordReader.rewind();
    } catch (Exception e) {
      LOGGER.error("Caught exception while rewinding record reader", e);
      Utils.rethrowException(e);
    }

    return _recordReader;
  }
}
