/*
 * Decompiled with CFR 0.152.
 */
package com.twitter.common.util;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.twitter.common.base.MorePreconditions;
import com.twitter.common.quantity.Amount;
import com.twitter.common.quantity.Time;
import com.twitter.common.quantity.Unit;
import com.twitter.common.stats.Stats;
import com.twitter.common.stats.StatsProvider;
import com.twitter.common.util.BackoffStrategy;
import com.twitter.common.util.Clock;
import com.twitter.common.util.Random;
import com.twitter.common.util.StateMachine;
import com.twitter.common.util.TruncatedBinaryBackoff;
import java.util.Deque;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Logger;
import javax.annotation.Nullable;

public class BackoffDecider {
    private static final Logger LOG = Logger.getLogger(BackoffDecider.class.getName());
    private final Iterable<BackoffDecider> deciderGroup;
    private final TimedStateMachine stateMachine;
    private final String name;
    private final double toleratedFailureRate;
    @VisibleForTesting
    final RequestWindow requests;
    private final BackoffStrategy strategy;
    private final Amount<Long, Time> recoveryPeriod;
    private long previousBackoffPeriodNs = 0L;
    private final Random random;
    private final Clock clock;
    private final AtomicLong backoffs;
    private final RecoveryType recoveryType;

    private BackoffDecider(String name, int seedSize, double toleratedFailureRate, @Nullable Iterable<BackoffDecider> deciderGroup, BackoffStrategy strategy, @Nullable Amount<Long, Time> recoveryPeriod, long requestWindowNs, int numBuckets, RecoveryType recoveryType, StatsProvider statsProvider, Random random, Clock clock) {
        MorePreconditions.checkNotBlank((String)name);
        Preconditions.checkArgument((seedSize > 0 ? 1 : 0) != 0);
        Preconditions.checkArgument((toleratedFailureRate >= 0.0 && toleratedFailureRate < 1.0 ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)strategy);
        Preconditions.checkArgument((recoveryPeriod == null || (Long)recoveryPeriod.getValue() > 0L ? 1 : 0) != 0);
        Preconditions.checkArgument((requestWindowNs > 0L ? 1 : 0) != 0);
        Preconditions.checkArgument((numBuckets > 0 ? 1 : 0) != 0);
        Preconditions.checkNotNull((Object)((Object)recoveryType));
        Preconditions.checkNotNull((Object)statsProvider);
        Preconditions.checkNotNull((Object)random);
        Preconditions.checkNotNull((Object)clock);
        this.name = name;
        this.toleratedFailureRate = toleratedFailureRate;
        this.deciderGroup = deciderGroup;
        this.strategy = strategy;
        this.recoveryPeriod = recoveryPeriod;
        this.recoveryType = recoveryType;
        this.random = random;
        this.clock = clock;
        this.backoffs = statsProvider.makeCounter(name + "_backoffs");
        this.requests = new RequestWindow(requestWindowNs, numBuckets, seedSize);
        this.stateMachine = new TimedStateMachine(name);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long awaitBackoff() throws InterruptedException {
        long backoffTimeMs;
        if (this.shouldBackOff() && (backoffTimeMs = this.stateMachine.getStateRemainingMs()) > 0L) {
            Object waitCondition;
            Object object = waitCondition = new Object();
            synchronized (object) {
                waitCondition.wait(backoffTimeMs);
            }
            return backoffTimeMs;
        }
        return 0L;
    }

    public synchronized boolean shouldBackOff() {
        boolean preventRequest;
        block0 : switch (this.stateMachine.getState()) {
            case NORMAL: {
                preventRequest = false;
                break;
            }
            case BACKOFF: {
                if (this.deciderGroup != null && this.allOthersBackingOff()) {
                    LOG.info("Backends in group with " + this.name + " down, forcing back up.");
                    this.stateMachine.transitionUnbounded(State.FORCED_NORMAL);
                    return false;
                }
                if (this.stateMachine.isStateExpired()) {
                    long recoveryPeriodNs = this.recoveryPeriod == null ? this.stateMachine.getStateDurationNs() : ((Long)this.recoveryPeriod.as((Unit)Time.NANOSECONDS)).longValue();
                    this.stateMachine.transition(State.RECOVERY, recoveryPeriodNs);
                    LOG.info(String.format("%s recovering for %s ms", this.name, Amount.of((long)recoveryPeriodNs, (Unit)Time.NANOSECONDS).as((Unit)Time.MILLISECONDS)));
                } else {
                    preventRequest = true;
                    break;
                }
            }
            case RECOVERY: {
                if (this.deciderGroup != null && this.allOthersBackingOff()) {
                    return false;
                }
                if (this.stateMachine.isStateExpired()) {
                    LOG.info(String.format("%s recovery successful", this.name));
                    this.stateMachine.transitionUnbounded(State.NORMAL);
                    this.previousBackoffPeriodNs = 0L;
                    preventRequest = false;
                    break;
                }
                switch (this.recoveryType) {
                    case RANDOM_LINEAR: {
                        preventRequest = this.random.nextDouble() > this.stateMachine.getStateFractionComplete();
                        break block0;
                    }
                    case FULL_CAPACITY: {
                        preventRequest = false;
                        break block0;
                    }
                }
                throw new IllegalStateException("Unhandled recovery type " + (Object)((Object)this.recoveryType));
            }
            case FORCED_NORMAL: {
                if (!this.allOthersBackingOff()) {
                    this.stateMachine.transition(State.RECOVERY, this.stateMachine.getStateDurationNs());
                    preventRequest = false;
                    break;
                }
                preventRequest = true;
                break;
            }
            default: {
                LOG.severe("Unrecognized state: " + (Object)((Object)this.stateMachine.getState()));
                preventRequest = false;
            }
        }
        if (preventRequest) {
            this.backoffs.incrementAndGet();
        }
        return preventRequest;
    }

    private boolean allOthersBackingOff() {
        for (BackoffDecider decider : this.deciderGroup) {
            boolean inBackoffState;
            State deciderState = decider.stateMachine.getState();
            boolean bl = inBackoffState = deciderState == State.BACKOFF || deciderState == State.FORCED_NORMAL;
            if (decider == this || inBackoffState) continue;
            return false;
        }
        return true;
    }

    public void addFailure() {
        this.addResult(false);
    }

    public void addSuccess() {
        this.addResult(true);
    }

    public synchronized void transitionToBackOff(double failRate, boolean force) {
        long prevBackoffMs = (Long)Amount.of((long)this.previousBackoffPeriodNs, (Unit)Time.NANOSECONDS).as((Unit)Time.MILLISECONDS);
        long backoffPeriodNs = (Long)Amount.of((long)this.strategy.calculateBackoffMs(prevBackoffMs), (Unit)Time.MILLISECONDS).as((Unit)Time.NANOSECONDS);
        if (!force) {
            LOG.info(String.format("%s failure rate at %g, backing off for %s ms", this.name, failRate, Amount.of((long)backoffPeriodNs, (Unit)Time.NANOSECONDS).as((Unit)Time.MILLISECONDS)));
        } else {
            LOG.info(String.format("%s forced to back off for %s ms", this.name, Amount.of((long)backoffPeriodNs, (Unit)Time.NANOSECONDS).as((Unit)Time.MILLISECONDS)));
        }
        this.stateMachine.transition(State.BACKOFF, backoffPeriodNs);
        this.previousBackoffPeriodNs = backoffPeriodNs;
    }

    private synchronized void addResult(boolean success) {
        if (this.stateMachine.getState() == State.BACKOFF) {
            return;
        }
        this.requests.addResult(success);
        double failRate = this.requests.getFailureRate();
        boolean highFailRate = this.requests.isSeeded() && failRate > this.toleratedFailureRate;
        switch (this.stateMachine.getState()) {
            case NORMAL: {
                if (!highFailRate) break;
                this.stateMachine.setStateDurationNs(0L);
            }
            case RECOVERY: {
                if (!highFailRate) break;
                this.requests.reset();
                this.transitionToBackOff(failRate, false);
                break;
            }
            case FORCED_NORMAL: {
                if (highFailRate) break;
                this.stateMachine.transition(State.RECOVERY, this.stateMachine.getStateDurationNs());
                break;
            }
            case BACKOFF: {
                throw new IllegalStateException("Backoff state may only be exited by expiration.");
            }
        }
    }

    public static Builder builder(String name) {
        return new Builder(name);
    }

    private class TimedStateMachine {
        final StateMachine<State> stateMachine;
        private long stateEndNs;
        private long stateDurationNs;

        TimedStateMachine(String name) {
            this.stateMachine = StateMachine.builder(name + "_backoff_state_machine").addState(State.NORMAL, State.BACKOFF, State.FORCED_NORMAL).addState(State.BACKOFF, (State[])new State[]{State.RECOVERY, State.FORCED_NORMAL}).addState(State.RECOVERY, (State[])new State[]{State.NORMAL, State.BACKOFF, State.FORCED_NORMAL}).addState(State.FORCED_NORMAL, (State[])new State[]{State.RECOVERY}).initialState(State.NORMAL).build();
        }

        State getState() {
            return this.stateMachine.getState();
        }

        void transitionUnbounded(State state) {
            this.stateMachine.transition(state);
        }

        void transition(State state, long durationNs) {
            this.transitionUnbounded(state);
            this.stateEndNs = BackoffDecider.this.clock.nowNanos() + durationNs;
            this.stateDurationNs = durationNs;
        }

        long getStateDurationNs() {
            return this.stateDurationNs;
        }

        long getStateDurationMs() {
            return (Long)Amount.of((long)this.stateDurationNs, (Unit)Time.NANOSECONDS).as((Unit)Time.MILLISECONDS);
        }

        void setStateDurationNs(long stateDurationNs) {
            this.stateDurationNs = stateDurationNs;
        }

        long getStateRemainingNs() {
            return this.stateEndNs - BackoffDecider.this.clock.nowNanos();
        }

        long getStateRemainingMs() {
            return (Long)Amount.of((long)this.getStateRemainingNs(), (Unit)Time.NANOSECONDS).as((Unit)Time.MILLISECONDS);
        }

        double getStateFractionComplete() {
            return 1.0 - (double)this.getStateRemainingNs() / (double)this.stateDurationNs;
        }

        boolean isStateExpired() {
            return BackoffDecider.this.clock.nowNanos() > this.stateEndNs;
        }
    }

    private static enum State {
        NORMAL,
        BACKOFF,
        RECOVERY,
        FORCED_NORMAL;

    }

    class RequestWindow {
        @VisibleForTesting
        long totalRequests = 0L;
        @VisibleForTesting
        long totalFailures = 0L;
        private final long durationNs;
        private final long bucketLengthNs;
        private final int seedSize;
        private final Deque<TimeSlice> buckets = Lists.newLinkedList();

        RequestWindow(long durationNs, int bucketCount, int seedSize) {
            this.durationNs = durationNs;
            this.bucketLengthNs = durationNs / (long)bucketCount;
            this.buckets.addFirst(new TimeSlice());
            this.seedSize = seedSize;
        }

        void reset() {
            this.totalRequests = 0L;
            this.totalFailures = 0L;
            this.buckets.clear();
            this.buckets.addFirst(new TimeSlice());
        }

        void addResult(boolean success) {
            this.maybeShuffleBuckets();
            ++this.buckets.peekFirst().requestCount;
            ++this.totalRequests;
            if (!success) {
                ++this.buckets.peekFirst().failureCount;
                ++this.totalFailures;
            }
        }

        void maybeShuffleBuckets() {
            if (BackoffDecider.this.clock.nowNanos() - this.buckets.peekFirst().bucketStartNs >= this.bucketLengthNs) {
                while (!this.buckets.isEmpty() && this.buckets.peekLast().bucketStartNs < BackoffDecider.this.clock.nowNanos() - this.durationNs) {
                    TimeSlice removed = this.buckets.removeLast();
                    this.totalRequests -= (long)removed.requestCount;
                    this.totalFailures -= (long)removed.failureCount;
                }
                this.buckets.addFirst(new TimeSlice());
            }
        }

        boolean isSeeded() {
            return this.totalRequests >= (long)this.seedSize;
        }

        double getFailureRate() {
            return this.totalRequests == 0L ? 0.0 : (double)this.totalFailures / (double)this.totalRequests;
        }
    }

    private class TimeSlice {
        int requestCount = 0;
        int failureCount = 0;
        final long bucketStartNs;

        public TimeSlice() {
            this.bucketStartNs = BackoffDecider.this.clock.nowNanos();
        }
    }

    public static class Builder {
        private String name;
        private int seedSize = 100;
        private double toleratedFailureRate = 0.5;
        private Set<BackoffDecider> deciderGroup = null;
        private BackoffStrategy strategy = new TruncatedBinaryBackoff((Amount<Long, Time>)Amount.of((long)100L, (Unit)Time.MILLISECONDS), (Amount<Long, Time>)Amount.of((long)10L, (Unit)Time.SECONDS));
        private Amount<Long, Time> recoveryPeriod = null;
        private long requestWindowNs = (Long)Amount.of((long)10L, (Unit)Time.SECONDS).as((Unit)Time.NANOSECONDS);
        private int numBuckets = 100;
        private RecoveryType recoveryType = RecoveryType.RANDOM_LINEAR;
        private StatsProvider statsProvider = Stats.STATS_PROVIDER;
        private Random random = Random.Util.newDefaultRandom();
        private Clock clock = Clock.SYSTEM_CLOCK;

        Builder(String name) {
            this.name = name;
        }

        public Builder withSeedSize(int seedSize) {
            this.seedSize = seedSize;
            return this;
        }

        public Builder withTolerateFailureRate(double toleratedRate) {
            this.toleratedFailureRate = toleratedRate;
            return this;
        }

        public Builder groupWith(Set<BackoffDecider> deciderGroup) {
            this.deciderGroup = deciderGroup;
            return this;
        }

        public Builder withStrategy(BackoffStrategy strategy) {
            this.strategy = strategy;
            return this;
        }

        public Builder withRecoveryPeriod(@Nullable Amount<Long, Time> recoveryPeriod) {
            this.recoveryPeriod = recoveryPeriod;
            return this;
        }

        public Builder withRequestWindow(Amount<Long, Time> requestWindow) {
            this.requestWindowNs = (Long)requestWindow.as((Unit)Time.NANOSECONDS);
            return this;
        }

        public Builder withBucketCount(int numBuckets) {
            this.numBuckets = numBuckets;
            return this;
        }

        public Builder withRecoveryType(RecoveryType recoveryType) {
            this.recoveryType = recoveryType;
            return this;
        }

        public Builder withStatsProvider(StatsProvider statsProvider) {
            this.statsProvider = statsProvider;
            return this;
        }

        @VisibleForTesting
        public Builder withRandom(Random random) {
            this.random = random;
            return this;
        }

        @VisibleForTesting
        public Builder withClock(Clock clock) {
            this.clock = clock;
            return this;
        }

        public BackoffDecider build() {
            BackoffDecider decider = new BackoffDecider(this.name, this.seedSize, this.toleratedFailureRate, this.deciderGroup, this.strategy, this.recoveryPeriod, this.requestWindowNs, this.numBuckets, this.recoveryType, this.statsProvider, this.random, this.clock);
            if (this.deciderGroup != null) {
                this.deciderGroup.add(decider);
            }
            return decider;
        }
    }

    public static enum RecoveryType {
        RANDOM_LINEAR,
        FULL_CAPACITY;

    }
}

